import os
import logging
import numpy as np
import pandas as pd
import param

from .material_properties import MaterialProperties
from .time_series import TimeSeries
from .file_io import read_bc_file
from .file_io import write_bc_file


log = logging.getLogger('adhmodel.simulation')


class BoundaryConditions(param.Parameterized):
    boundary_strings = param.DataFrame(default=pd.DataFrame({
        "CARD": pd.Series(dtype='str'),
        "ID": pd.Series(dtype='Int64'),
        "ID_0": pd.Series(dtype='Int64'),
        "ID_1": pd.Series(dtype='Int64')}),
        doc='Columns for this DataFrame ["CARD", "ID", "ID_0", "ID_1"]. This holds information for the following: '
            'NDS ID ID_0 - ID: id of this node string, ID_0: node id that is part of this string, '
            'EGS ID ID_0 ID_1 - ID: id of this edge string, ID_0, ID_1: first, second node ids of element edge,'
            'MDS ID ID_0 ID_1 - ID: id of this mid string, ID_0, ID_1: first, second element ids,'
            'MTS ID - ID: id of this material string.', allow_None=True
    )  # allow_None kwarg must be provided in param>=1.10 but must not be provided in param<1.10
    solution_controls = param.DataFrame(
        default=pd.DataFrame(data=[], columns=["CARD", "CARD_2", "STRING_ID", "XY_ID_0", "XY_ID_1", "XY_ID_2"]),
        doc='Columns for this DataFrame ["CARD", "CARD_2", "STRING_ID", "XY_ID_0", "XY_ID_1", "XY_ID_2"]. '
            'This holds information for the following: '
            'NB DIS S_ID XY_ID - Discharge, S_ID: string id, XY_ID: xy series id for discharge, '
            'NB OVL S_ID XY_ID - Flow, S_ID: string id, XY_ID: xy series for flow, flow per unit area for '
                'material strings and flow per unit length for edge strings, '
            'NB OTW S_ID XY_ID - Water surface elevation, S_ID: string id, XY_ID: xy series for elevation, '
            'NB TRN S_ID C_ID XY_ID - Transport, S_ID: string id, C_ID: constituent id, XY_ID: xy series for '
                'constituent concentration (units dependent of the transport type), '
            'NB SPL S_ID XY_ID - Spillway, S_ID: string id, XY_ID: Series for the percent (%) flow out, '
            'NB OUT O_ID I_ID XY_ID - Flow output inside grid, O_ID: outflow edge id, I_ID: inflow edge id, '
                'XY_ID: series outflow id, '
            'NB TID S_ID - Tidal boundary, S_ID: string id, '
            'DB OVL S_ID X_ID Y_ID - Velocity (2D), S_ID: string id, X_ID: series id for x-velocity, Y_ID: series'
                'id for y-velocity, '
            'DB OVH S_ID X_ID Y_ID DEPTH_ID - Velocity and Depth, S_ID: string id, X_ID: series id for x-velocity, '
                'Y_ID: series id for y-velocity, DEPTH_ID: series id for the depth'
            'DB TRN S_ID C_ID XY_ID - Transport Concentration, S_ID: string id, C_ID: constituent id, XY_ID: '
                'xy series for constituent concentration, '
            'DB LDE S_ID XY_ID - Stationary lid elevation, S_ID: string id, XY_ID: xy series for elevation, '
            'DB LDH S_ID XY_ID - Depth of water under stationary lid, S_ID: string id, XY_ID: xy series for depth, '
            'DB LID S_ID XY_ID - Floating stationary object, S_ID: string id, XY_ID: xy series for the draft of lid, '
            'BR USR S_ID XY_ID - User defined breach displacement, S_ID: string id, XY_ID: xy series for elevation, '
            'OB OF S_ID - Natural Outflow, S_ID: string id, '
            'EQ TRN S_ID C_ID XY_ID - Transport, S_ID: string id, C_ID: constituent id, XY_ID: '
                'xy series for constituent concentration, '
            'OFF S_ID - Deactivate string, S_ID: string id, '
            'Not represented here: NB SDR, BR JAI, BR SAS, BR MLM, BR FRO, BR BRC, BR VTG, BR FER, WER, WRS, FLP, '
            'FGT, SLUICE, SLS.',
    )
    stage_discharge_boundary = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'S_ID', 'COEF_A', 'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E']),
        doc='Columns [CARD, S_ID, COEF_A, COEF_B, COEF_C, COEF_D, COEF_E], '
            'NB SDR: Stage discharge boundary. S_ID: String id, COEF_A: coefficient A, COEF_B: coefficient B, '
            'COEF_C: coefficient C, COEF_D: coefficient D, COEF_E: coefficient E.',
    )
    friction_controls = param.DataFrame(
        default=pd.DataFrame(data=[], columns=["CARD", "CARD_2", "STRING_ID", "REAL_01", "REAL_02",
            "REAL_03", "REAL_04", "REAL_05"]),
        doc='Columns for this DataFrame ["CARD", "CARD_2", "STRING_ID", "REAL_01", "REAL_02", '
            '"REAL_03", "REAL_04", "REAL_05"]. '
            'This holds information for the following: '
            "FR MNG S_ID MAN_N - Manning's N, S_ID: string id, MAN_N: manning's n, "
            "FR MNC S_ID MAN_N - Manning's Equation, S_ID: string id, MAN_N: manning's n, "
            "FR ERH S_ID ROUGH_HEIGHT - Equivalent roughness height, S_ID: string id, ROUGH_HEIGHT: roughness height, "
            "FR SAV S_ID ROUGH_HEIGHT STEM_HEIGHT - Submerged aquatic vegetation, S_ID: string id, ROUGH_HEIGHT: "
                "roughness height of the SAV canopy (k), STEM_HEIGHT: undeflected stem height, "
            "FR URV S_ID ROUGH_HEIGHT STEM_DIAMETER STEM_DENS - Un-submerged rigid vegetation, S_ID: string id, "
                "ROUGH_HEIGHT: bed roughness height, STEM_DIAMETER: avg stem diameter, STEM_DENS: avg stem density, "
            "FR EDO S_ID ROUGH_HEIGHT OBSTRUCT_DIAMETER OBSTRUCT_HEIGHT - Equivalent drag obstructions, "
                "S_ID: string id, ROUGH_HEIGHT: bed roughness height, OBSTRUCT_DIAMETER: obstruction diameter, "
                "OBSTRUCT_HEIGHT: obstruction height, "
            "FR ICE S_ID ICE_THICKNESS ICE_DENSITY ICE_MOVEMENT ICE_THICKNESS BED_ROUGHNESS - Ice thickness, "
                "S_ID: string id, ICE_THICKNESS: ice thickness, ICE_DENSITY: ice density, ICE_MOVEMENT: "
                "ice movement, ICE_ROUGHNESS: ice roughness height, BED_ROUGHNESS: bed roughness height, "
            "FR DUN S_ID FACTOR SED_INC D_50 D_90 - Dune Roughness, S_ID: string id, FACTOR: Dune factor, "
                "SED_INC: sedlib inclusion flag, D_50: d_50, D_90: d_90, "
            "FR SDK S_ID HEIGHT - Submerged dike, S_ID: string id, HEIGHT: height of the dike, "
            "FR BRD S_ID ELEV THICKNESS - Bridge deck, S_ID: string id, ELEV: elevation of bridge deck, "
                "thickness of bridge deck",
    )
    breach_controls = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['CARD', 'CARD_1', 'C_00', 'C_01', 'C_02', 'C_03', 'C_04',
                                               'C_05', 'C_06', 'C_07', 'C_08', 'C_09']),
        doc='[CARD][CARD_1][C_00][C_01]...[C_09]'
            'BR JAI S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV FAIL_TIME FURTHEST CLOSEST - Breach johnson and illes, '
                'S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, MIN_ELEV: min breach elev, '
                'CREST_ELEV: Dam/levee crest elevation, FAIL_TIME: breach failure time, FURTHEST: side slope node '
                'furthest from breach, CLOSEST: side slope node closest to breach, '
            'BR SAS S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV FAIL_TIME FURTHEST CLOSEST - Breach singh and snorrason, '
                'S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, MIN_ELEV: min breach elev, '
                'CREST_ELEV: Dam/levee crest elevation, FAIL_TIME: breach failure time, FURTHEST: side slope node '
                'furthest from breach, CLOSEST: side slope node closest to breach, '
            'BR MLM MAX_WATER_DEPTH RESERVOIR_VOL MIN_ELEV CREST_ELEV - Breach macdonald and landgridge-monopolis, '
                'MAX_DEPTH: Max water depth above breach bottom, RESERVOIR_VOL: reservoir volume, MIN_ELEV: minimum '
                'breach elevation, CREST_ELEV: Dam/levee crest elevation, '
            'BR FRO S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV FAIL_TIME EXPO FURTHEST CLOSEST - Breach froelich, '
                'S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, MIN_ELEV: min '
                'breach elev, CREST_ELEV: Dam/levee crest elevation, FAIL_TIME: breach failure time, EXPO: exponent '
                'for main breach, FURTHEST: side slope node furthest from breach, CLOSEST: side slope node closest '
                'to breach, '
            'BR BRC S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV FAIL_TIME FURTHEST CLOSEST - Breach bureau of '
                'reclamation, S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, MIN_ELEV: min '
                'breach elev, CREST_ELEV: Dam/levee crest elevation, FAIL_TIME: breach failure time, FURTHEST: side '
                'slope node furthest from breach, CLOSEST: side slope node closest to breach, '
            'BR VTG S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV EROSION_CHAR FAIL_TIME FURTHEST CLOSEST - Breach '
                'von thun and gillette, S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, '
                'MIN_ELEV: min breach elev, CREST_ELEV: Dam/levee crest elevation, EROSION_CHAR: erosion character, '
                'FAIL_TIME: breach failure time, FURTHEST: side slope node furthest from breach, CLOSEST: side '
                'slope node closest to breach, '
            'BR FER S_ID BREACH_SEC WIDTH MIN_ELEV CREST_ELEV ENG_CHAR FAIL_TIME FURTHEST CLOSEST - Breach '
                'fed energy regulatory disp, S_ID: string id, BREACH_SEC: breach section, WIDTH: width of main breach, '
                'MIN_ELEV: min breach elev, CREST_ELEV: Dam/levee crest elevation, ENG_CHAR: engineering character, '
                'FAIL_TIME: breach failure time, FURTHEST: side slope node furthest from breach, CLOSEST: side '
                'slope node closest to breach, ',
    )
    weirs = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['WRS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'WS_UPSTREAM',
                                               'WS_DOWNSTREAM', 'LENGTH', 'CREST_ELEV', 'HEIGHT']),
        doc='WRS_NUMBER S_UPSTREAM S_DOWNSTREAM WS_UPSTREAM WS_DOWNSTREAM LENGTH CREST_ELEV HEIGHT - Weir Parameters, '
                'NUMBER: Weir number, S_UPSTREAM: string upstream of weir, S_DOWNSTREAM: string downstream of weir, '
                'NUMBER: weir number, WS_UPSTREAM: weir string on upstream, WS_DOWNSTREAM: weir string on downstream, '
                'LENGTH: length of weir, CREST_ELEV: crest elevation, HEIGHT: weir height',
    )
    flap_gates = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['FGT_NUMBER', 'USER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'FS_UPSTREAM',
                                               'FS_DOWNSTREAM', 'COEF_A', 'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E',
                                               'COEF_F', 'LENGTH']),
        doc='FGT_NUMBER USER S_UPSTREAM S_DOWNSTREAM FS_UPSTREAM FS_DOWNSTREAM COEF_A COEF_B COEF_C COEF_D COEF_E '
                'COEF_F LENGTH - Flap gate parameters, NUMBER: flap gate number, USER: user defined parameters, '
                'S_UPSTREAM: string upstream of flap, S_DOWNSTREAM: string downstream of flap, FS_UPSTREAM: flap '
                'string on the upstream, FS_DOWNSTREAM: flap string on the downstream, COEF_A: coeficient A, '
                'COEF_B: coeficient B, COEF_C: coeficient C, COEF_D: coeficient D, COEF_E: coeficient E, '
                'COEF_F: coeficient F, LENGTH: length of flap gate',
    )
    sluice_gates = param.DataFrame(
        default=pd.DataFrame(data=[], columns=['SLS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'SS_UPSTREAM',
                                               'SS_DOWNSTREAM', 'LENGTH', 'TS_OPENING']),
        doc='SLS_NUMBER S_UPSTREAM S_DOWNSTREAM SS_UPSTREAM SS_DOWNSTREAM LENGTH TS_OPENING - Sluice gate parameters, '
                'NUMBER: sluice gate number, S_UPSTREAM: string upstream of sluice gate, S_DOWNSTREAM: string '
                'downstream of sluice gate, SS_UPSTREAM: sluice string on upstream, SS_DOWNSTREAM: sluice string on '
                'downstream, LENGTH: length of sluice gate, TS_OPENING: time series defining the sluice gate opening',
    )

    def __init__(self, **params):
        """
        Initializes and sets default values for attributes in the adhModel object by calling subfunctions for
        initialization based on the version number. Default version is 4.5.


        Initilize an AdH version 5.0 object

        Initializes and sets default values for attributes in the adhModel object, including path controls,
        operating parameters, iteration parameters, mesh refinement parameters, friction parameters,
        solution parameters, time parameters, input and output controls, input and output time parameters,
        mesh configuration, depth data, and linked data inside the adhModel object.

        NOTE: Adh data types that are not yet included:
            WNDLIB: List of wind library parameters
            FLI: Maximum number of linear iterations. Advance after obtaining
            FNI: Number of nonlinear iterations per timestep.  Advance after obtaining.
            RTL: Runge-kutta tolerance for reactive constituents
            SST: Quasi-unsteady tolerance
            BEDLAYERS: Total number of bed layers
            RAD: Dirichlet - short wave radiation and dew point
            SRC: NB SOURCE - rain or evap from an area
            INS: Ice strings
            SXX: Radiation stress tensor, Sxx
            SXY: Radiation stress tensor, Sxy
            SYY: Radiation stress tensor, Syy

        Parameters
        ----------
        self : obj
            The adhModel object for this simulation.

        Returns
        -------
        None
            Sets and modifies boundary condition attributes
        """
        super(BoundaryConditions, self).__init__(**params)
        self.material_properties = {1: MaterialProperties()}
        self.sediment_material_properties = {}
        self.mat_id_to_mat_sed = {}
        self.time_series = {}

    def read(self, file_name, model_control, fmt='bc'):
        """
        Function to read in the boundary condition file into the adhModel object.

        Reads an AdH formatted boundary condition file (*.bc) and stores the parameter data for each variable in the
        appropriate space of the simulation object (self) .

        Parameters
        ----------
        file_name : str
            the full file path for the file that is to read in.
        model_control : ModelControl
            The spatially constant model control options.
        fmt : str
            format of the file. Options: 'bc'

        Returns
        -------
        None
            Saves the results to the appropriate object in adhModel.

        """
        if fmt == 'bc':
            read_bc_file(file_name, self, model_control)
        else:
            raise IOError('Boundary condition file format not recognized. Current options are: "bc"')

    def write(self, file_name, model_control, fmt='bc'):
        """
        Writes the boundary condition file for the o_adhModel object.

        Writes an AdH formatted boundary condition file (*.bc) using data stored in the o_adhModel object file (does
        not include constituents, sediment, winds, or waves).

        Parameters
        ----------
        file_name : str
            the full file path written for the boundary condition data.
        model_control : ModelControl
            The spatially constant model control options.
        fmt : str
            format of the file. Options: 'bc'

        Returns
        -------
        None
            Writes out boundary condition data to specified file and path.

        """
        if fmt == 'bc':
            write_bc_file(file_name, self, model_control)
        else:
            raise IOError('Boundary condition file format not recognized. Current options are: "bc"')

    def add_material(self, material, number=None):
        """
        Set a new material or replace an existing one

        :param material: MaterialProperty()
        :param number: Integer
        :return:

        """
        if number is None:
            number = max(self.material_properties, key=int) + 1

        self.material_properties[number] = material

    def get_strings(self, string_type='EGS'):
        """

        :param string_type: String card. Options are 'EGS', 'NDS', 'MTS', 'MDS'
        :return: dataframe of this string type
        """

        df = self.boundary_strings.loc[self.boundary_strings['CARD'] == string_type]

        return df

    def set_default_sw2(self, mesh=None):
        """
        Method to set default boundary conditions for a d2 AdH simulation. Contains everything required to run a
        simplified simulation. Takes an empty AdH simulation object and fills it with default values.

        Note: only applicable for v5.0

        Parameters
        ----------
        self : AdhModel
            Am empty AdH simulation object.

        Returns
        -------
        None
            Sets operating conditions in the adhModel with default values.


        """
        # For reference:
        # String 1 - material string
        # String 2 - edge string, inflow
        # String 3 - edge string, outflow

        # Solution controls
        solution_dict = {'CARD': ['NB'], 'CARD_2': ['OVL'], 'STRING_ID': [2], 'XY_ID_0': [1], 'XY_ID_1': [np.nan],
                         'XY_ID_2': [np.nan]}
        solution_df = pd.DataFrame.from_dict(solution_dict)
        self.solution_controls = pd.concat([self.solution_controls, solution_df])

        # Natural - tailwater elevation
        solution_controls_dict = {'CARD': ['NB'], 'CARD_2': ['OTW'], 'STRING_ID': [2], 'XY_ID_0': [1],
                                  'XY_ID_1': [np.nan], 'XY_ID_2': [np.nan]}
        solution_controls_df = pd.DataFrame.from_dict(solution_controls_dict)
        self.solution_controls = pd.concat([self.solution_controls, solution_controls_df])

        # Boundary string parameters
        # self.mts.append([1, 1])           # Material strings
        boundary_strings_dict = {'CARD': ['MTS'],
                                 'ID': pd.Series([1], dtype='Int64'),
                                 'ID_0': pd.Series([1], dtype='Int64'),
                                 'ID_1': pd.Series([None], dtype='Int64')}
        boundary_strings_df = pd.DataFrame.from_dict(boundary_strings_dict)
        self.boundary_strings = pd.concat([self.boundary_strings, boundary_strings_df])

        # if a mesh was provided
        if mesh is not None:
            # if the mesh already has boundary nodes determined
            if mesh.boundaryNodes is not None:
                # todo add this back in after integration with genesis mesh
                # grab the first 3 nodes and create a string

                # grab the middle 3 nodes and create a string
                raise RuntimeError('utilizing a mesh for setting defaults is not currently available. ')

            else:
                print(
                    'Mesh does not contain boundary node information, setting arbitrary nodes into boundary condition strings')
                boundary_strings_dict = {'CARD': ['EGS'], 'ID': [2], 'ID_0': [1], 'ID_1': [2]}
                boundary_strings_df = pd.DataFrame.from_dict(boundary_strings_dict)
                self.boundary_strings = pd.concat([self.boundary_strings, boundary_strings_df], ignore_index=True)

                boundary_strings_dict = {'CARD': ['EGS'], 'ID': [3], 'ID_0': [4], 'ID_1': [5]}
                boundary_strings_df = pd.DataFrame.from_dict(boundary_strings_dict)
                self.boundary_strings = pd.concat([self.boundary_strings, boundary_strings_df], ignore_index=True)
        else:
            print('No mesh was provided, setting arbitrary nodes into boundary condition strings')

            boundary_strings_dict = {'CARD': ['EGS'], 'ID': [2], 'ID_0': [1], 'ID_1': [2]}
            boundary_strings_df = pd.DataFrame.from_dict(boundary_strings_dict)
            self.boundary_strings = pd.concat([self.boundary_strings, boundary_strings_df])

            boundary_strings_dict = {'CARD': ['EGS'], 'ID': [3], 'ID_0': [4], 'ID_1': [5]}
            boundary_strings_df = pd.DataFrame.from_dict(boundary_strings_dict)
            self.boundary_strings = pd.concat([self.boundary_strings, boundary_strings_df])

        # Time series object for discharge
        temp_data = pd.DataFrame(data=[[0.0, 100.0], [999999, 100.0]], columns=['X', 'Y'])
        temp_series = TimeSeries(series_id=1, series_type='SERIES BC', time_series=temp_data)
        self.time_series[1] = temp_series

        # Time series object for tailwater
        temp_data = pd.DataFrame(data=[[0.0, 20], [999999, 20]], columns=['X', 'Y'])
        temp_series = TimeSeries(series_id=2, series_type='SERIES BC', time_series=temp_data)
        self.time_series[2] = temp_series

        # Time series object for time step
        temp_data = pd.DataFrame(data=[[0.0, 300], [999999, 300]], columns=['X', 'Y'])
        temp_series = TimeSeries(series_id=3, series_type='SERIES DT', time_series=temp_data)
        self.time_series[3] = temp_series

        # Time series object for output series
        temp_data = pd.DataFrame(data=[[0.0, 999999, 300, 0]],
                                 columns=['START_TIME', 'END_TIME', 'TIME_STEP_SIZE', 'UNITS'])
        temp_series = TimeSeries(series_id=4, series_type='SERIES AWRITE', time_series=temp_data)
        self.time_series[4] = temp_series


    def create_time_series(self, series_type, time_series, units=0, series_id=None, **kwargs):
        """Method to make AdH time series object from an array of time and data.
        Parameters
        ----------
        series_type: str
            Type of time series. Supported types in ADH 5.0 are 'SERIES AWRITE', 'SERIES WRITE', 'SERIES BC',
            'SERIES DT', 'SERIES WIND', 'SERIES WAVE'
        time_series: dataframe
            Time series. Column labels ['X','Y']
        units: int
            time units. Options include:
                0 - seconds
                1 - minutes
                2 - hours
                3 - days
                4 - weeks
        series_id: int
            ID of this xy series
        kwargs
        ------
        output_units: int
            output time units. Options include:
                0 - seconds
                1 - minutes
                2 - hours
                3 - days
                4 - weeks
        x_location: float
            X location for SERIES WIND and SERIES WAVE
        y_location: float
            Y location for SERIES WIND and SERIES WAVE
        Returns
        -------
        None
           Sets self.time_series{key: value}, a time series object inside of adh_model containing all time series information.
        """
        # if no series id was given, append to the end of the existing list of time series
        if series_id is None:
            series_id = len(self.time_series) + 1
        # check to see if series id already exist
        elif series_id in self.time_series.keys():
            log.warning('Overwriting existing time series with same series_id')
        # create new series
        ts = TimeSeries(series_id=series_id, series_type=series_type, time_series=time_series,
                        units=units, **kwargs)
        # set series into model object
        self.time_series[series_id] = ts
