"""Constituent properties."""

# 1. Standard python modules

# 2. Third party modules
import param

# 3. Aquaveo modules

# 4. Local modules
from adhparam.data_frame_utils import create_default_dataframe


# The segment table columns and types
SEGMENTS_COLUMN_TYPES = {
    "SEGMENT_NUMBER": "int64",
    "SEGMENT_TYPE": "int64",
    "END_X_COORDINATE": "float64",
    "END_Y_COORDINATE": "float64",
    "MOTION_VALUE": "float64",  # Either Velocity OR Time of Arrival
    "ARC_CENTER_X": "float64",
    "ARC_CENTER_Y": "float64",
    "TURN_DIRECTION": "int64"
}
_segment_column_docs = (
    """
    SDEF - Sailing Line Segment Definition
    - Vessel number (int) - Must be greater than 0
    - Sailing line segment number (int) - Must be greater than 0
    - Type of Segment (int) - 0 for line, 1 for arc
    - x-coordinate of segment end (real)
    - y-coordinate of segment end (real)
    - Vessel velocity at segment end or time of arrival at segment end (real)
    - If segment is an arc, the x-coordinate of the arc center (real)
    - If segment is an arc, the y-coordinate of the arc center (real)
    - If segment is an arc, the direction of turn (real) - +1.0 for left turns, -1.0 for right turns
    """
)
_default_segments_df = create_default_dataframe(SEGMENTS_COLUMN_TYPES)


class Propeller(param.Parameterized):
    """Class representing propeller properties of a vessel."""

    propeller_type = param.ObjectSelector(
        default="Open wheel",
        objects=["Open wheel", "Kort nozzle"],
        doc="PROP: Propeller type: 1 for open wheel, 2 for Kort nozzle.",
        precedence=1
    )

    propeller_diameter = param.Number(
        default=0.0,
        doc='PROP: Propeller diameter.',
        precedence=2
    )

    propeller_center_distance = param.Number(
        default=0.0,
        doc='PROP: Distance between the centers of the propellers.',
        precedence=3
    )

    tow_boat_length = param.Number(
        default=0.0,
        doc='PROP: Tow boat length.',
        precedence=4
    )

    distance_to_stern = param.Number(
        default=0.0,
        doc='PROP: Distance from the propellers to the stern of the tow boat.',
        precedence=5
    )


class Vessel(param.Parameterized):
    """Class representing a single vessel with its properties and sailing segments."""

    initial_x_coordinate = param.Number(
        default=0.0,
        doc='FDEF: X-coordinate of the vessel’s initial position.',
        precedence=2
    )

    initial_y_coordinate = param.Number(
        default=0.0,
        doc='FDEF: Y-coordinate of the vessel’s initial position.',
        precedence=3
    )

    initial_speed = param.Number(
        default=0.0,
        doc='FDEF: Initial velocity magnitude of the vessel.',
        precedence=4
    )

    vessel_draft = param.Number(
        default=0.0,
        doc='DRFT: Vessel draft.',
        precedence=5
    )

    vessel_length = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='BLEN: Vessel length (real) - Must be greater than 0.',
        precedence=6
    )

    vessel_width = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='BWID: Vessel width (real) - Must be greater than 0.',
        precedence=7
    )

    bow_length_ratio = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='PBOW: Ratio of bow length to the vessel length (real) - Must be greater than 0.',
        precedence=8
    )

    stern_length_ratio = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='PSTR: Ratio of stern length to the vessel length (real) - Must be greater than 0.',
        precedence=9
    )

    bow_draft_ratio = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='CBOW: Bow draft ratio (real) - Must be greater than 0.',
        precedence=10
    )

    stern_draft_ratio = param.Number(
        default=0.0,
        bounds=(0, None),
    doc='CSTR: Stern draft ratio (real) - Must be greater than 0.',
        precedence=11
    )

    propeller = param.ClassSelector(
        class_=Propeller,
        default=None,
        doc='Optional propeller properties for the vessel.',
        precedence=12
    )

    sailing_segments = param.DataFrame(
        default=create_default_dataframe(SEGMENTS_COLUMN_TYPES),
        doc=_segment_column_docs,
        precedence=13
    )


class VesselList(param.Parameterized):
    use_velocity = param.Integer(
        default=0,
        doc=('BOAT\n'
             '- Number of vessels to be modeled (calculated)\n'
             '- Input 0 to define boat by position and velocity, 1 to define boat by position and time of arrival\n'),
        precedence=1,
    )

    vessels = param.List(
        default=[],
        class_=Vessel,  # Ensures only Vessel objects can be added to the list
        doc="List of Vessel objects to be modeled.",
        precedence=2,
    )
