"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.units.angle_or_slope import ConversionAngleOrSlopeCalc
from xms.FhwaVariable.core_data.units.area import ConversionAreaCalc
from xms.FhwaVariable.core_data.units.byte import ConversionByteCalc, normalize_si_unit
from xms.FhwaVariable.core_data.units.concentration import ConversionConcentrationCalc
from xms.FhwaVariable.core_data.units.density import ConversionDensityCalc
from xms.FhwaVariable.core_data.units.energy import ConversionEnergyCalc
from xms.FhwaVariable.core_data.units.flow import ConversionFlowCalc
from xms.FhwaVariable.core_data.units.force import ConversionForceCalc
from xms.FhwaVariable.core_data.units.k import ConversionkCalc
from xms.FhwaVariable.core_data.units.length import ConversionLengthCalc
from xms.FhwaVariable.core_data.units.mass import ConversionMassCalc
from xms.FhwaVariable.core_data.units.pressure import ConversionPressureCalc
from xms.FhwaVariable.core_data.units.specific_weight import ConversionSpecificWeightCalc
from xms.FhwaVariable.core_data.units.temperature import ConversionTemperatureCalc
from xms.FhwaVariable.core_data.units.time import ConversionTimeCalc
from xms.FhwaVariable.core_data.units.velocity import ConversionVelocityCalc
from xms.FhwaVariable.core_data.units.viscosity import ConversionViscosityCalc
from xms.FhwaVariable.core_data.units.volume import ConversionVolumeCalc
from xms.FhwaVariable.core_data.units.weir import ConversionWeirCalc


class ConversionCalc:
    """perform unit conversion computations."""
    coefficient = ['coef', 'coefficient', 'Coefficient', 'COEFFICIENT', 'coefficients', 'Coefficients',
                   'COEFFICIENTS']

    def __init__(self, app_data=None, g=None, use_us_gallon=None, stream_value=None, model_name=None,
                 project_uuid=None):
        """Initialize the ConversionCalc Class.

        Args:
            app_data (AppSettingsData): The settings CalcData
            g (float): The gravity value
            use_us_gallon (bool): Whether to use US gallon
            stream_value (float): The stream value
            model_name (str): The model name
            project_uuid (uuid): The project uuid
        Returns:
            True if converted, False if units not found
        """
        # Note that only SettingsCalc or g, use_us_gallon, and stream_value should be passed in, not both
        gravity = 32.17405
        if app_data is not None:
            _, gravity = app_data.get_setting('Gravity', default=gravity, model_name=model_name,
                                              project_uuid=project_uuid)
        if g is not None:
            gravity = g

        stream = 12.0
        if app_data is not None:
            _, stream = app_data.get_setting('Stream value', default=stream)
        if stream_value is not None:
            stream = stream_value

        us_gallon = True
        if app_data is not None:
            _, gallon = app_data.get_setting('Gallon', default='US gallon')
            us_gallon = gallon == 'US gallon'
        if use_us_gallon is not None:
            us_gallon = use_us_gallon

        self.converters = {}
        self.converters['length'] = ConversionLengthCalc()
        self.converters['angle'] = ConversionAngleOrSlopeCalc()
        self.converters['area'] = ConversionAreaCalc()
        self.converters['volume'] = ConversionVolumeCalc(us_gallon=us_gallon)
        self.converters['flow'] = ConversionFlowCalc(us_gallon=us_gallon, stream_value=stream)
        self.converters['Temperature'] = ConversionTemperatureCalc()
        self.converters['mass'] = ConversionMassCalc()
        self.converters['specific weight'] = ConversionSpecificWeightCalc()
        self.converters['density'] = ConversionDensityCalc()
        self.converters['pressure'] = ConversionPressureCalc()
        self.converters['velocity'] = ConversionVelocityCalc()
        self.converters['k'] = ConversionkCalc()
        self.converters['weir'] = ConversionWeirCalc(g=gravity)
        self.converters['time'] = ConversionTimeCalc()
        self.converters['energy'] = ConversionEnergyCalc()
        self.converters['concentration'] = ConversionConcentrationCalc()
        self.converters['viscosity'] = ConversionViscosityCalc()
        self.converters['force'] = ConversionForceCalc()
        self.converters['byte'] = ConversionByteCalc()

    def convert_units(self, from_unit, to_unit, value):
        """Convert units from one unit to another.

        Args:
            from_unit (string): unit value is currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if isinstance(value, str):
            try:
                value = float(value)
            except ValueError:
                return False, value
        if isinstance(value, int):
            value = float(value)
        if not isinstance(value, float):
            return False, value
        if not isinstance(from_unit, str):
            return False, value
        if not isinstance(to_unit, str):
            return False, value
        from_unit = from_unit.strip()
        to_unit = to_unit.strip()
        if from_unit == '' or to_unit == '' or math.isnan(value):
            return False, value
        if from_unit == to_unit:
            return True, value

        # Some units use whitespace (sq mi) so don't sanitize whitespace

        if from_unit in self.coefficient or to_unit in self.coefficient:
            return True, value

        for converter in self.converters:
            result, converted_value = self.converters[converter].convert_units(from_unit=from_unit, to_unit=to_unit,
                                                                               value=value)
            if result:
                return result, converted_value

        # Not one of our basic types, let's try complex combinations and SI prefixes
        # TODO: Handle more complexity. Currently ignores paranthesis and exponents
        # Ideally, we'd tokenize in pemdas in a logical function/loop so we could
        # handle multiple dividors, exponents, dividors, and multipliers
        # Currently, we assume everything right of a divisor is denominator, which is incorrect
        # Separate numerator and denominator
        from_numerator, from_denominator = self.split_unit_by_divisor(from_unit)
        to_numerator, to_denominator = self.split_unit_by_divisor(to_unit)

        # Don't quit - could still be compound units or SI prefixes
        # if from_numerator == to_numerator and from_denominator == to_denominator:
        #     pass
        #     # return True, value

        # Convert the numerators
        n_value = copy.copy(value)
        if from_numerator != '' or to_numerator != '':
            from_numerator_tokens = from_numerator.split('*')
            to_numerator_tokens = to_numerator.split('*')

            for from_token, to_token in zip(from_numerator_tokens, to_numerator_tokens):
                from_result, from_base_unit, from_ratio = normalize_si_unit(from_token)
                to_result, to_base_unit, to_ratio = normalize_si_unit(to_token)
                if from_result or to_result:
                    n_value *= from_ratio / to_ratio  # Apply the SI value to the conversion

                for converter in self.converters:
                    result, n_value = self.converters[converter].convert_units(
                        from_unit=from_base_unit, to_unit=to_base_unit, value=n_value)
                    if result:
                        break
                if not result:
                    return False, value

        # Convert the denominators
        d_value = 1.0
        if from_denominator != '' or to_denominator != '':
            from_denominator_tokens = from_denominator.split('*')
            to_denominator_tokens = to_denominator.split('*')

            for from_token, to_token in zip(from_denominator_tokens, to_denominator_tokens):
                from_result, from_base_unit, from_si_value = normalize_si_unit(from_token)
                to_result, to_base_unit, to_si_value = normalize_si_unit(to_token)
                if from_result or to_result:
                    d_value *= from_si_value / to_si_value  # Apply the SI value to the conversion

                for converter in self.converters:
                    result, d_value = self.converters[converter].convert_units(
                        from_unit=from_base_unit, to_unit=to_base_unit, value=d_value)
                    if result:
                        break
                if not result:
                    return False, value

        return True, n_value / d_value

    def split_unit_by_divisor(self, unit):
        """Split a unit string at the '/' character.

        - If there is exactly 1 '/', split at that.
        - If there are exactly 3 '/', split at the middle one (the second '/').
        - Otherwise, return the original string and ''.

        Args:
            unit (str): The unit string to split.

        Returns:
            tuple: A tuple containing the numerator and denominator components.
        """
        count = unit.count('/')
        if count == 1:
            return tuple(unit.split('/', 1))
        elif count == 3:
            # Find the positions of all slashes
            first = unit.find('/')
            second = unit.find('/', first + 1)
            # third = unit.find('/', second + 1)
            # Split at the second slash (the middle one)
            return unit[:second], unit[second + 1:]
        else:
            return unit, ''

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit is None:
            return False, from_unit
        if not isinstance(from_unit, str):
            return False, from_unit
        if from_unit == '':
            return False, from_unit

        for converter in self.converters:
            result, complementary_unit = self.converters[converter].get_si_complementary_unit(from_unit=from_unit)
            if result:
                return result, complementary_unit

        return False, from_unit
