"""Script used to create an HyReport.exe executable."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import re
import subprocess

# 2. Third party modules
import PyInstaller.__main__

# 3. Aquaveo modules

# 4. Local modules


class CreateExe:
    """Class to handle creating the executable for Hydraulic Toolbox."""

    def __init__(self, app_name: str, app_icon: str, main_file: str, resource_paths: list,
                 frozen_resource_paths: list, app_folder: str = None, frozen_app_path: str = None):
        """Initialize the CreateExe class."""
        app_name_safe = re.sub(r'[^a-zA-Z0-9]', '_', app_name)  # Sanitize app name for file paths
        if app_folder is None:
            self.exe_version_file_path = f"../FhwaVariable/xms/FhwaVariable/core_data/{app_name_safe}_exe_version.txt"
            self.version_file_path = f"../FhwaVariable/xms/FhwaVariable/core_data/{app_name_safe}_version.txt"
            self.build_date_file = f"../FhwaVariable/xms/FhwaVariable/core_data/{app_name_safe}_build_date.txt"
            self.frozen_app_path = "xms/FhwaVariable/core_data/"
        else:
            self.exe_version_file_path = f"{app_folder}/{app_name_safe}_exe_version.txt"
            self.version_file_path = f"{app_folder}/{app_name_safe}_version.txt"
            self.build_date_file = f"{app_folder}/{app_name_safe}_build_date.txt"
            self.frozen_app_path = frozen_app_path
        self.app_name = app_name
        self.app_name_safe = app_name_safe
        self.app_icon = app_icon
        self.main_file = main_file
        self.resource_paths = ['../FhwaVariable/xms/FhwaVariable/core_data/resources']
        self.resource_paths.extend(resource_paths)
        self.frozen_resource_paths = ['xms/FhwaVariable/core_data/resources']
        self.frozen_resource_paths.extend(frozen_resource_paths)
        self.app_folder = app_folder

    def get_git_version(self):
        """
        Retrieve the latest Git version tag.

        Returns:
            str: The Git version tag, or '0.0.0' if no tag is found.
        """
        try:
            version = subprocess.check_output(
                ["git", "describe", "--tags", "--always"],
                stderr=subprocess.STDOUT,
                text=True
            ).strip()
            return version
        except subprocess.CalledProcessError:
            return "0.0.0"

    def get_clean_version(self):
        """
        Retrieve and clean the git version tag.

        Returns:
            str: The cleaned version number (e.g., '1.0.0').
        """
        version = self.get_git_version()
        # Use a regex to extract the version number (e.g., 'v1.0.0' -> '1.0.0')
        match = re.match(r"v?(\d+\.\d+\.\d+)", version)
        if match:
            return match.group(1)
        return "0.0.0"  # Default version if no match is found

    def get_csv_version(self, version):
        """
        Retrieve the latest CSV version tag.

        Returns:
            str: The CSV version tag, or '0.0.0' if no tag is found.
        """
        version_parts = version.split(".")
        while len(version_parts) < 4:
            version_parts.append("0")  # Ensure there are four parts
        csv_version = ", ".join(version_parts)
        return csv_version

    def get_build_date(self):
        """
        Get the current date as the build date.

        Returns:
            str: The current date in YYYY-MM-DD format.
        """
        return datetime.datetime.now().strftime("%Y-%m-%d")

    def update_version_file(self, version, build_date):
        """
        Update the version.txt file with the given version and build date.

        Args:
            version (str): The version to set in the version.txt file.
            build_date (str): The build date to include in the version.txt file.
        """
        csv_version = self.get_csv_version(version)
        with open(self.exe_version_file_path, "r") as file:
            lines = file.readlines()

        # Update the FileVersion, ProductVersion, and add BuildDate
        updated_lines = []
        for line in lines:
            if "StringStruct(u'FileVersion'" in line:
                updated_lines.append(f"    StringStruct(u'FileVersion', u'{version}'),\n")
            elif "StringStruct(u'ProductVersion'" in line:
                updated_lines.append(f"    StringStruct(u'ProductVersion', u'{version}')])\n")
            elif "filevers=(" in line:
                updated_lines.append(f"filevers=({csv_version}),\n")
            elif "prodvers=(" in line:
                updated_lines.append(f"prodvers=({csv_version}),\n")
            elif "StringStruct(u'BuildDate'" in line:
                updated_lines.append(f"    StringStruct(u'BuildDate', u'{build_date}'),\n")
            else:
                updated_lines.append(line)

        with open(self.exe_version_file_path, "w") as file:
            file.writelines(updated_lines)
        # # Write the version to a separate file
        with open(self.version_file_path, "w") as f:
            f.write(version)

        # # Write the build date to a separate file
        with open(self.build_date_file, "w") as f:
            f.write(build_date)

    def generate_resource_data_string(self):
        """Generate the --add-data strings for PyInstaller."""
        add_data_strings = [
            f"--add-data={resource};{frozen_resource}"
            for resource, frozen_resource in zip(self.resource_paths, self.frozen_resource_paths)
        ]
        return add_data_strings

    def create_exe(self, include_major_version: bool = True, include_minor_version: bool = True,
                   include_patch_version: bool = True):
        """
        Create the executable using PyInstaller.
        """
        # Get the version and build date
        version = self.get_clean_version()
        build_date = self.get_build_date()

        # Update the version.txt file
        self.update_version_file(version, build_date)

        resource_strings = self.generate_resource_data_string()

        app_name = self.app_name_safe
        if include_major_version:
            app_name += f"_v{version.split('.')[0]}"
            if include_minor_version:
                app_name += f"_{version.split('.')[1]}"
                if include_patch_version:
                    app_name += f"_{version.split('.')[2]}"

        # Run PyInstaller
        PyInstaller.__main__.run([
            f'{self.main_file}',
            # Executable options
            '--onefile',
            f'--icon={self.app_icon}',
            '--noconsole',  # Add this line to prevent the console window from appearing
            # Versioning and build date
            f'--version-file={self.exe_version_file_path}',
            f'--add-data={self.version_file_path};{self.frozen_app_path}',
            f'--add-data={self.build_date_file};{self.frozen_app_path}',
            # Resources
            *resource_strings,
            # Name of executable
            f'--name={app_name}',
        ])


# Note the following is provided as an example: main.py doesn't exist and these resource paths are already added.
# Include app_folder & frozen_app_path if you want to place these files in your app code
if __name__ == "__main__":
    # Create an instance of CreateExe and run the process
    icon = '../FhwaVariable/xms/FhwaVariable/core_data/resources/icons/root.ico'
    creator = CreateExe(app_name='FhwaVariable',
                        app_icon=icon,
                        main_file='main.py',
                        resource_path=['../FhwaVariable/xms/FhwaVariable/core_data/resources'],
                        frozen_resource_path=['xms/FhwaVariable/core_data/resources'])
    creator.create_exe()
