"""Functions to assist with setting up dialogs."""

__copyright__ = "(C) Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import os
from pathlib import Path
import subprocess

# 2. Third party modules
import cv2

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.model.commands.command import CommandManager
# from xms.FhwaVariable.interface_adapters.controller.dialog.dialog_controller import DialogController
from xms.FhwaVariable.interface_adapters.presenter.dialog.dialog_presenter import DialogPresenter
from xms.FhwaVariable.interface_adapters.view_model.main.message_box import MessageBox


class DialogInteractor():
    """A class to define the model-view of a dialog."""
    def __init__(self, view, dlg_uuid, edit_calculator_functor):
        """Initialize the dialog class.

        Args:
            view: The view to use
            dlg_uuid (str): the UUID of the dialog
            edit_calculator_functor: The functor to edit the CalcData
        """
        self.app_view = view
        self.dlg_view = None
        self.dlg_uuid = dlg_uuid
        self.dlg_calculator = None
        # self.controller = DialogController()
        self.dialog_presenter = None

        self.edit_calculator_functor = edit_calculator_functor

        self.plot_windows = 0

    def launch_dialog_with_calculator(self, calcdata, dlg_uuid, current_model, modal=None, icon=None,
                                      parent_uuid=None):
        """Dialog that will launch a calcdata dialog.

        Args:
            calcdata (CalcData): the CalcData to be launched
            dlg_uuid (str): the UUID of the dialog
            current_model (str): the current model
            modal (bool): whether the dialog is modal (True) or modeless (False)
            icon (str): the icon to be used for the dialog
            parent_uuid (str): the UUID of the parent dialog

        Return:
            True if the user clicks 'OK'
            calcdata (CalcData): the modified CalcData
        """
        app_data = calcdata.app_data
        self.dialog_presenter = DialogPresenter(app_data, dlg_uuid)
        self.dlg_calculator = copy.deepcopy(calcdata)

        _, max_history = app_data.get_setting('Undo list length', 100)
        _, selected_unit_system = app_data.get_setting('Selected unit system', 'U.S. Customary Units')
        self.dlg_calculator.command_manager = CommandManager(max_history)
        self.dlg_calculator.selected_unit_system = selected_unit_system

        self.dlg_view = self.app_view.create_dialog_view(dlg_uuid)

        result, self.dlg_calculator = self.dialog_presenter.run_dialog(self.dlg_calculator, self.app_view,
                                                                       self.dlg_view, current_model,
                                                                       modal=modal, icon=icon,
                                                                       parent_uuid=parent_uuid)

        if result is True:
            calcdata = copy.deepcopy(self.dlg_calculator)
            return True, calcdata
        else:
            # If we are working with the settings calcdata, the changes are applied immmediately and we need to
            # reset the settingscalculator to the original values
            if self.dlg_calculator.type == 'AppSettingsData':
                AppData.app_settings = copy.deepcopy(calcdata)

            # dialog.show()
            # dialog.size_dlg()

            # result = dialog.exec()
            # if result == QDialog.Accepted:
            #     dlg_calculator.command_manager = None
            #     calcdata = copy.deepcopy(dlg_calculator)
            #     return True, calcdata
            # elif result == QDialog.Rejected:
            #     if calcdata.type == 'SettingsCalc':
            #         AppData.app_data = copy.deepcopy(calcdata)
            #     return False, calcdata
        return False, None

    def update_dialog(self):
        """Update the dialog."""
        self.dialog_presenter.update_dialog(self.dlg_calculator, self.dlg_view, self.plot_windows)

    def set_val_by_uuid(self, row_uuid, new_value, prev_value=None, index=None, is_unit_change=False,
                        add_undo=True, uuid_index=None, update_dialog=True):
        """Set the value of a variable by its UUID.

        Args:
            calcdata: The CalcData to use.
            row_uuid (str): the UUID of the variable
            new_value (str): the new value
            index (int): index of the variable
            is_unit_change (bool): whether the change is a unit change
            prev_value (str): the previous value
            add_undo (bool): whether to add to the undo stack
            uuid_index (int): the index of the UUID in the list
            update_dialog (bool): whether to update the dialog
        """
        result = self.dlg_calculator.set_val_by_uuid(row_uuid, new_value, index, is_unit_change, prev_value,
                                                     add_undo, uuid_index)
        if result:
            if self.dlg_calculator.type == 'AppSettingsData':
                AppData.app_settings = copy.deepcopy(self.dlg_calculator)
            if update_dialog:
                self.update_dialog()

    def btn_clicked(self, row_uuid, uuid_index, is_unit_col, btn_txt, add_undo, icon):
        """Button clicked event.

        Args:
            row_uuid (str): the UUID of the row
            uuid_index (int): the index of the UUID
            is_unit_col (bool): whether the column is a unit column
            btn_txt (str): the text of the button
            add_undo (bool): whether to add to the undo stack
            icon (str): the icon of the button
        """
        result, item = self.dlg_calculator.find_item_by_uuid(row_uuid, uuid_index)
        if not result:
            return False
        if uuid_index is not None:
            # This is a calc/var list item
            # Determine if the user clicked duplicate, delete, or define
            button_txt = btn_txt[0:5]
            if button_txt == 'Dupli':
                result, calc_list_item = self.dlg_calculator.find_item_by_uuid(row_uuid)
                if result:
                    calc_list_item.duplicate_item(uuid_index)
            elif button_txt == 'Delet':
                result, calc_list_item = self.dlg_calculator.find_item_by_uuid(row_uuid)
                if result:
                    calc_list_item.delete_item(uuid_index)
            elif button_txt == 'Defin' or button_txt == 'View.' or button_txt == 'Edit ':
                result, calc = self.edit_calculator_functor(item, item.model_name, icon=icon,
                                                            parent_uuid=self.dlg_uuid)
                if result:
                    # result, calc_list_item = self.dlg_calculator.find_item_by_uuid(row_uuid)
                    # calc_list_item.set_item_by_index(calc, uuid_index)
                    self.dlg_calculator.set_val_by_uuid(row_uuid, calc, prev_value=item, add_undo=add_undo,
                                                        uuid_index=uuid_index)
                    result = True

        elif not is_unit_col:
            if item.type in ['calc', 'table',]:
                result, calc = self.edit_calculator_functor(item.value, item.value.model_name, icon=icon,
                                                            parent_uuid=self.dlg_uuid)
                if result:
                    self.dlg_calculator.set_val_by_uuid(row_uuid, calc, prev_value=item, add_undo=add_undo)
                    result = True
            elif item.type in ['image']:
                self.dialog_presenter.display_image(item.get_val(), self.dlg_view)
            elif item.type in ['color']:
                old_color = item.get_val()
                # if not isinstance(old_color, QColor):
                #     old_color = QColor(self.rgb_to_hex(old_color))
                color = self.select_color(old_color)
                if color:
                    rgb_color = (color.red(), color.green(), color.blue())
                    self.dlg_calculator.set_val_by_uuid(row_uuid, rgb_color, prev_value=item.get_val(),
                                                        add_undo=add_undo)
                    result = True
            else:
                raise ValueError(f'Button clicked on an unsupported type: {item.type}')
        elif is_unit_col:  # Buttons in the unit column
            if item.type in ['file', 'image']:
                new_val = None
                old_file = item.get_val()
                result, file = self.dialog_presenter.select_file_from_variable(self.dlg_view, item)
                # file = self.select_file_or_folder(item.get_val(), item.file_mode, item.value_options)
                if file and len(file) > 0 and item.type in ['image']:
                    new_val = cv2.imread(str(file[0]))
                elif file and len(file) > 0 and item.type in ['file']:
                    new_val = file[0]
                if new_val is not None:
                    self.dlg_calculator.set_val_by_uuid(row_uuid, new_val, prev_value=old_file,
                                                        add_undo=add_undo)
                result = True
            else:
                raise ValueError(f'Button clicked on an unsupported type: {item.type}')

        if result:
            self.update_dialog()

        if result and self.dlg_calculator.type == 'SettingsCalc':
            AppData.settings_calc = self.dlg_calculator
        return result

    def show_panel(self, panel_name: str, state: int):
        """Show a panel.

        Args:
            dlg_uuid (UUID): The dialog UUID
            panel_name (str): The name of the panel
            state (int): Whether the panel is checked
        """
        if state == 2:
            checked = True
        else:
            checked = False
        if panel_name == 'Wiki':
            self.dlg_calculator.show_wiki = checked
        elif panel_name == 'Plot':
            self.dlg_calculator.show_plot = checked

        self.update_dialog()
        return True

    def plot_btn(self, plot_interactor):
        """Handle the plot button click.

        Args:
            plot_interactor: The dialog interactor for the plot
        """
        self.plot_windows += 1
        self.update_dialog()

    def store_size(self, app_settings, screen_setup_str, name, model, x, y, width, height, maximized):
        """Store the dialog size.

        Args:
            app_settings: The app settings
            screen_setup_str (str): The screen setup string
            name (str): The name of the dialog
            model (str): The model name
            x (int): The x position
            y (int): The y position
            width (int): The width
            height (int): The height
            maximized (bool): Whether the dialog is maximized
        """
        if maximized:
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'maximized', True)
        else:
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'maximized', False)
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'x', x)
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'y', y)
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'width', width)
            app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'height', height)

    def close_plot(self, x, y, width, height, maximized, index, change_count=True):
        """Close the plot window, but store the dialog size.

        Args:
            x (int): The x position
            y (int): The y position
            width (int): The width
            height (int): The height
            maximized (bool): Whether the dialog is maximized
            index (int): The index of the plot
            change_count (bool): Whether to change the plot count
        """
        app_settings = self.dlg_calculator.app_data.app_settings
        model = self.dlg_calculator.model_name

        name = self.dlg_calculator.name
        if hasattr(self.dlg_calculator, 'class_name') and self.dlg_calculator.class_name != '':
            name = self.dlg_calculator.class_name

        if self.app_view is not None:
            screens_info = self.app_view.get_screens_info()
        screen_setup_str = screens_info['screen_setup_str']

        if name == 'Settings calc' and model == '' or model is None:
            model = 'AppData'

        plot_name = f'{name} plot {index}'

        self.store_size(app_settings, screen_setup_str, plot_name, model, x, y, width, height, maximized)

        if change_count:
            self.plot_windows -= 1
            if self.plot_windows < 0:
                self.plot_windows = 0
            app_settings.save_persistant_dialog_data(screen_setup_str, plot_name, model, 'plot_windows',
                                                     self.plot_windows)

    def store_dlg_size(self, x, y, width, height, maximized, show_wiki, show_plot, splitter_h, splitter_v):
        """Store the dialog size."""
        app_settings = self.dlg_calculator.app_data.app_settings
        model = self.dlg_calculator.model_name

        name = self.dlg_calculator.name
        if hasattr(self.dlg_calculator, 'class_name') and self.dlg_calculator.class_name != '':
            name = self.dlg_calculator.class_name

        if self.app_view is not None:
            screens_info = self.app_view.get_screens_info()
        screen_setup_str = screens_info['screen_setup_str']

        if name == 'Settings calc' and model == '' or model is None:
            model = 'AppData'

        self.store_size(app_settings, screen_setup_str, name, model, x, y, width, height, maximized)

        app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'show_wiki', show_wiki)
        app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'show_plot', show_plot)

        app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_h_size', len(splitter_h))
        for index, size in enumerate(splitter_h):
            if len(size) == 0:
                pass
            if len(size) >= 1:
                app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_h_x', size[0], index)
            if len(size) >= 2:
                app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_h_y', size[1], index)

        app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_v_size', len(splitter_v))
        for index, size in enumerate(splitter_v):
            if len(size) == 0:
                pass
            if len(size) >= 1:
                app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_v_x', size[0], index)
            if len(size) >= 2:
                app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'splitter_v_y', size[1], index)

        app_settings.save_persistant_dialog_data(screen_setup_str, name, model, 'plot_windows', self.plot_windows)

        # TODO: Save splitter locations
        # state_h = []
        # state_v = []
        # index = 0
        # for splitter in self.splitters_h:
        #     state_h.append(splitter.saveState())
        #     app_settings.save_persistant_dialog_data(dialog.window_title, current_model, f'HSplitter{index}',
        #                                              state_h[-1])
        #     index += 1
        #     # sizes = splitter.sizes()
        #     # self.settings_calc.save_persistant_dialog_data(f'HSplitter{index}-num', len(sizes))
        #     # inner_index = 0
        #     # for size in sizes:
        #     #     self.settings_calc.save_persistant_dialog_data(f'HSplitter{index}-{inner_index}', size)
        #     #     inner_index += 1
        # app_settings.save_persistant_dialog_data(dialog.window_title, current_model, 'num_h_splitters', len(state_h))

        # for splitter in self.splitters_v:
        #     state_v.append(splitter.saveState())
        #     app_settings.save_persistant_dialog_data(dialog.window_title, current_model, f'VSplitter{index}',
        #                                               state_v[-1])
        #     index += 1
        #     # sizes = splitter.sizes()
        #     # self.settings_calc.save_persistant_dialog_data(f'HSplitter{index}-num', len(sizes))
        #     # inner_index = 0
        #     # for size in sizes:
        #     #     self.settings_calc.save_persistant_dialog_data(f'HSplitter{index}-{inner_index}', size)
        #     #     inner_index += 1
        # app_settings.save_persistant_dialog_data(dialog.window_title, current_model, 'num_v_splitters', len(state_v))
        # # TODO: Make a button in settings to clear the registry of dialog settings
        # # TODO: Make a button to clear all registry settings ?

    def launch_pdf(self, path_to_acrobat, path_to_pdf, page_number):
        """Start the process to open a PDF file.

        Args:
            path_to_acrobat (Path): the path to the Adobe Acrobat Reader executable
            path_to_pdf (Path): the path to the PDF file
            page_number (int): the page number to open
        """
        if page_number:
            _ = subprocess.Popen(
                [path_to_acrobat, '/A', f'page={page_number}', path_to_pdf],
                shell=False,
                stdout=subprocess.PIPE
            )
        else:
            _ = subprocess.Popen([path_to_acrobat, path_to_pdf], shell=False, stdout=subprocess.PIPE)
        # process.wait()

    def open_pdf(self, reference_name):
        """Open a PDF file.

        Args:
            reference_name (str): the name of the reference
        """
        if reference_name in self.dlg_calculator.reference_pdfs:
            reference = self.dlg_calculator.reference_pdfs[reference_name]
            if isinstance(reference, tuple):
                filename = reference[0]
                page_number = reference[1]
            else:
                filename = reference
                page_number = 0

            _, document_folder = self.dlg_calculator.get_setting('Document folder')
            _, path_to_acrobat = self.dlg_calculator.get_setting('Adobe Reader')

            if isinstance(document_folder, str):
                document_folder = Path(document_folder)
            if isinstance(path_to_acrobat, str):
                path_to_acrobat = Path(path_to_acrobat)

            path_to_pdf = document_folder / filename

            if os.path.exists(path_to_pdf):
                if os.path.exists(path_to_acrobat):
                    self.launch_pdf(path_to_acrobat, path_to_pdf, page_number)
                else:
                    msg_box = MessageBox()
                    msg_box.window_title = "Error"
                    msg_box.message_type = "warning"
                    msg_box.message_text = \
                        'Adobe Acrobat was not found. Please specify the filename and path in the Settings, on the ' \
                        'Preferences tab.'
                    msg_box.buttons = ['ok']
                    self.dlg_view.message_box(msg_box)

                    # TODO: Add a way to specify the path to the Adobe Acrobat Reader
                    # msg_box.message_text = \
                    #     'Adobe Acrobat was not found. You specify the filename and path in the Settings, on the ' \
                    #     'Preferences tab or click OK to specify the path now.'
                    # msg_box.buttons = ['ok', 'cancel']
                    # response = self.dlg_view.message_box(msg_box)

                    # if response == 'ok':
                    #     select_file = SelectFile()
                    #     select_file.parent = None
                    #     if hasattr(self.dlg_view, 'dlg'):
                    #         select_file.parent = self.dlg_view.dlg
                    #     select_file.window_title = "Select Adobe Acrobat Reader"
                    #     select_file.starting_path = Path.home()
                    #     select_file.file_mode = True
                    #     select_file.file_types = ['Executable (*.exe)']
                    #     select_file.allow_multiple_files = False
                    #     success, selected_file = self.dlg_view.select_file(select_file)
                    #     if success:
                    #         path_to_acrobat = selected_file[0]
                    #         self.dlg_calculator.set_setting('Adobe Reader', path_to_acrobat)
                    #         self.launch_pdf(path_to_acrobat, path_to_pdf, page_number)

            else:
                msg_box = MessageBox()
                msg_box.window_title = "Error"
                msg_box.message_type = "warning"
                msg_box.message_text = \
                    'PDF file not found. Check the document folder in the Settings, on the Preferences tab.'
                msg_box.buttons = ['ok']
                self.dlg_view.message_box(msg_box)
