"""Abstract Class that defines the base functionality for a Graphics View plug-in."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
import math
import sys
from typing import Tuple
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid as CoGrid, read_grid_from_string
# from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.window import GraphicsImage, Orientation
from xms.FhwaVariable.interface_structures.graphics_view import GraphicsViewBase


@dataclass
class _GraphicsView():
    """Provides a class that will take orientation, settings, and datahandler and returna rendered image."""
    image_count: int = 0
    image_width: int = 800
    image_height: int = 600
    image_bytes: bytes = bytes()
    # image_bytes: bytes = bytes(neutral_blue * (image_width * image_height))

    grids: list[CoGrid] = field(default_factory=list)

    _orientation: Orientation = field(default_factory=Orientation)

    def render(self, view_uuid: uuid.UUID, orientation: Orientation, width: int, height: int) -> Tuple[bytes, int, int]:
        """Renders the image using the given orientation, settings, and data handler.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the orientation of the view
            width (px): the width of the view
            height (px): the height of the view

        Returns:
            image (GraphicsImage): the rendered image
        """
        self.image_width = width
        self.image_height = height

        remainder = self.image_width % 4
        output_width = self.image_width if remainder == 0 else self.image_width + 4 - remainder

        # Get a bg_color based on the rotation of the orientation
        # This is a placeholder for the actual color calculation logic
        bg_color = self._get_color(orientation)

        # Simulate rendering process
        # self.image_bytes = bytes(bg_color * (output_width * height))
        # Simulate rendering process
        self.image_bytes = bytearray(bg_color * (output_width * height))

        self.draw_plus(output_width, height, orientation, bg_color)

        return bytes(self.image_bytes), output_width, height

    def draw_plus(self, width, height, orientation, bg_color) -> None:
        """Draw a plus sign that adjusts based on zoom and pan."""
        # Calculate the size of the "+" symbol in world coordinates
        plus_size_world = min(orientation.width, orientation.height) * 0.05  # 5% of the smaller dimension

        # Ensure orientation.width and orientation.height are valid
        if orientation.width <= 0 or orientation.height <= 0:
            orientation.width = 100  # Default width
            orientation.height = 100

        # Convert world coordinates to image coordinates
        pixels_per_unit_x = width / orientation.width
        pixels_per_unit_y = height / orientation.height

        # Calculate the center of the "+" symbol in image coordinates
        center_x_image = int((orientation.at[0]) * pixels_per_unit_x + width // 2)
        center_y_image = int((orientation.at[1]) * pixels_per_unit_y + height // 2)
        plus_size_pixels = int(plus_size_world * pixels_per_unit_x)

        # + color
        pen_color = (255, 255, 255)  # White color
        bg_color_added = bg_color[0] + bg_color[1] + bg_color[2]
        if bg_color_added > 300:
            pen_color = (0, 0, 0)

        # Draw the "+" symbol
        for i in range(-plus_size_pixels, plus_size_pixels + 1):
            # Horizontal line of the "+"
            if 0 <= center_x_image + i < width:
                self._set_pixel(center_x_image + i, center_y_image, pen_color)

            # Vertical line of the "+"
            if 0 <= center_y_image + i < height:
                self._set_pixel(center_x_image, center_y_image + i, pen_color)

    def _set_pixel(self, x: int, y: int, color: Tuple[int, int, int]) -> None:
        """Sets a pixel in the image at the given coordinates.

        Args:
            x (int): The x-coordinate of the pixel.
            y (int): The y-coordinate of the pixel.
            color (Tuple[int, int, int]): The RGB color of the pixel.
        """
        if 0 <= x < self.image_width and 0 <= y < self.image_height:
            index = (y * self.image_width + x) * 3  # Each pixel has 3 bytes (RGB)
            self.image_bytes[index:index + 3] = bytes(color)

    def _apply_transformations(self, orientation: Orientation, width: int, height: int) -> None:
        """Applies transformations to distort the "+" symbol based on the orientation.

        Args:
            orientation (Orientation): The orientation of the view.
            width (int): The width of the image.
            height (int): The height of the image.
        """
        # Example: Apply a simple distortion based on the bearing and dip
        distortion_factor_x = 1.0 + (orientation.bearing / 360.0)
        distortion_factor_y = 1.0 + (orientation.dip / 180.0)

        center_x = width // 2
        center_y = height // 2

        for y in range(height):
            for x in range(width):
                # Calculate the distorted coordinates
                distorted_x = int((x - center_x) * distortion_factor_x + center_x)
                distorted_y = int((y - center_y) * distortion_factor_y + center_y)

                # Copy the pixel to the distorted location
                if 0 <= distorted_x < width and 0 <= distorted_y < height:
                    original_index = (y * width + x) * 3
                    distorted_index = (distorted_y * width + distorted_x) * 3
                    self.image_bytes[distorted_index:distorted_index + 3] = \
                        self.image_bytes[original_index:original_index + 3]

    def _eye_vector(self, orientation) -> tuple[float, float, float]:
        """
        Get the direction the eye is looking.

        Returns:
            A unit vector in the direction the eye is looking.
        """
        pi_over_180 = math.pi / 180.0
        bearing = orientation.bearing * pi_over_180
        dip = orientation.dip * pi_over_180
        cosine_dip = math.cos(dip)

        eye_x = cosine_dip * math.sin(bearing)
        eye_y = -cosine_dip * math.cos(bearing)
        eye_z = math.sin(dip)
        return eye_x, eye_y, eye_z

    def _get_color(self, orientation) -> tuple[int, int, int]:
        """
        Get color to display for a given orientation.

        Returns:
            Tuple with red, green, and blue colors.
        """
        eye_vector = self._eye_vector(orientation)
        red = int((-eye_vector[0] + 1.0) * 255.0 / 2.0)
        green = int((-eye_vector[1] + 1.0) * 255.0 / 2.0)
        blue = int((-eye_vector[2] + 1.0) * 255.0 / 2.0)
        return red, green, blue


class GraphicsView(GraphicsViewBase):
    """Provides a class that will take orientation, settings, and datahandler and returna rendered image."""
    def __init__(self):
        """Initialize the GraphicsView class."""
        self._graphics_view = _GraphicsView()

        two_cell_grid_string = (
            "ASCII RectilinearGrid2d Version 1\n"
            "DIMENSIONS 2\n"
            "ORIGIN 0.0 0.0 0.0\n"
            "ANGLE 0.0\n"
            "NUMBERING KIJ\n"
            "ORIENTATION X_INCREASE Y_INCREASE\n"
            "LOCATIONS_X 2\n"
            "  0.0\n"
            "  1.0\n"
            "LOCATIONS_Y 3\n"
            "  0.0\n"
            "  1.0\n"
            "  2.0\n")
        grid = read_grid_from_string(two_cell_grid_string)

        self.add_grid(grid, '{}')  # Add a default grid to the list

    def render(self, view_uuid: uuid.UUID, orientation: Orientation, width: int, height: int) -> GraphicsImage:
        """Renders the image using the given orientation, settings, and data handler.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the orientation of the view
            width (px): the width of the view
            height (px): the height of the view
        Returns:
            image (GraphicsImage): the rendered image
        """
        graphic = GraphicsImage()
        graphic.image_bytes, graphic.image_width, graphic.image_height = self._graphics_view.render(
            view_uuid, orientation, width, height)
        if self._graphics_view.image_count == sys.maxsize - 1:
            self._graphics_view.image_count = 0
        self._graphics_view.image_count = self._graphics_view.image_count + 1
        graphic.image_count = self._graphics_view.image_count

        return graphic

    def add_grid(self, grid: CoGrid, display_settings: str) -> None:
        """Adds a grid to the list of grids.

        Args:
            grid (CoGrid): the grid to be added
            display_settings (str): the display_settings of the grid
        """
        self._graphics_view.grids.append(grid)

    def set_vtk_scalars(self, view_uuid: uuid.UUID, grid_uuid: uuid.UUID, dataset, timestep):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            grid_uuid (uuid.UUID): the uuid of the grid
            dataset: the dataset to be set for the view
            timestep: the timestep to be set for the view
        """
        pass

    def set_vtk_data_cache(self, view_uuid: uuid.UUID, view_data):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            view_data: the data to be set for the view
        """
        pass

    def set_vtk_display_settings(self, view_uuid: uuid.UUID, grid_uuid: uuid.UUID, settings):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            grid_uuid (uuid.UUID): the uuid of the grid
            settings: the display settings for the view
        """
        pass

# What we discussed
    def gui_tool_change_old(self, view_uuid: uuid.UUID, orientation: Orientation, dh: int, df: int, tool='pan'
                            ) -> Orientation:
        """Handles the tool change event.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the current orientation of the view
            dh (DataHandler): the data handler for the view
            df (DataFrame): the data frame for the view
            tool (str): the tool to be used

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    # Proposed
    def gui_tool_change(self, view_uuid: uuid.UUID, orientation: Orientation, point_1: Tuple[int, int],
                        point_2: Tuple[int, int], tool='pan') -> Orientation:
        """Handles the tool change event.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the current orientation of the view
            dh (DataHandler): the data handler for the view
            df (DataFrame): the data frame for the view
            tool (str): the tool to be used

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        if tool in ['zoom_wheel_in', 'zoom_wheel_out']:
            # Zoom wheel modifies the width and height based on the tool type
            scale_factor = 0.9 if tool == 'zoom_wheel_in' else 1.1
            orientation.width *= scale_factor
            orientation.height *= scale_factor

            return orientation

        dx = point_2[0] - point_1[0]
        dy = point_2[1] - point_1[1]

        if tool == 'pan':
            # Pan modifies the "at" position by shifting it based on dx and dy
            orientation.at = (
                orientation.at[0] + dx * 0.01,  # Scale factor for panning
                orientation.at[1] + dy * 0.01,
                orientation.at[2]
            )

        elif tool == 'rotate':
            # Rotate modifies the bearing and dip based on dx and dy
            orientation.bearing += dx * 0.1  # Scale factor for rotation
            orientation.dip += dy * 0.1
            orientation.bearing %= 360  # Keep bearing within 0-360 degrees
            orientation.dip = max(0, min(180, orientation.dip))  # Clamp dip between 0 and 180 degrees

        elif tool == 'zoom_in':
            # Zoom in decreases the width and height
            # orientation.width *= 0.9  # Scale factor for zooming
            # orientation.height *= 0.9
            orientation.at[2] * 0.9

        elif tool == 'zoom_out':
            # Zoom out increases the width and height
            # orientation.width *= 1.1  # Scale factor for zooming
            # orientation.height *= 1.1
            orientation.at[2] * 1.1

        elif tool == 'zoom_window':
            # Zoom window adjusts the width and height based on the distance between point_1 and point_2
            # window_width = abs(point_2[0] - point_1[0])
            window_height = abs(point_2[1] - point_1[1])
            # orientation.width = max(1.0, window_width * 0.01)  # Scale factor for zoom window
            # orientation.height = max(1.0, window_height * 0.01)
            orientation.at[2] = max(1.0, window_height * 0.01)

        return orientation

    def frame_view(self, view_uuid: uuid.UUID) -> Orientation:
        """Frames the view using the given orientation and uuid list.

        Args:
            view_uuid (uuid.UUID): the uuid of the view

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    def set_orientation(self, view_uuid: uuid.UUID, orientation: Orientation, use_height=False) -> Orientation:
        """Sets the orientation of the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the orientation to be set for the view
            use_height (bool): whether to use height or not

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass
