"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_calc import ScourBaseCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_hole_data import ScourHole


class ScourBaseData(CalcData):
    """A class that defines a pier scour at a bridge contraction."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Scour Base Calc"
        self.type = "ScourBaseCalc"

        self.calc_support_dual_input = False

        self.calculator = ScourBaseCalc()

        self.gradations = None

        self.approach_gradation_dict = {}
        self.contracted_gradation_dict = {}

        self.d50 = 0.0
        self.d50_ft = 0.0
        self.upstream_d50 = 0.0
        self.upstream_d50_ft = 0.0
        self.critical_shear_stress = 0.0

        # Input

        # Approach Gradation
        # self.input['Approach gradation'] = Variable(
        #     'Approach gradation (D50)',
        #     'uuid_dict',
        #     None, {},
        #     note='')
        self.input['Approach D50'] = Variable(
            'Approach D50 (D50)', 'float', 0.0, [], precision=6, unit_type=['length'], native_unit='ft',
            us_units=self.us_short_length, si_units=self.si_short_length, note='')
        self.input['Approach D50'].selected_us_unit = 'mm'
        self.input['Approach D50'].selected_si_unit = 'mm'
        self.input['Selected approach location'] = Variable(
            'Selected gradation location along approach cross-section',
            'list', -1, ['left overbank', 'main channel', 'right overbank'], note='')
        # Contracted Gradation
        # self.input['Contracted gradation'] = Variable(
        #     'Contracted gradation (D50)',
        #     "uuid_dict",
        #     None, {},
        #     note='')
        self.input['Contracted D50'] = Variable(
            'Contracted D50 (D50)', 'float', 0.0, [], precision=6, unit_type=['length'], native_unit='ft',
            us_units=self.us_short_length, si_units=self.si_short_length,
            note='Used for clear water scour depth calculations, 0.2 mm is the lower limit for noncohesive material')
        self.input['Contracted D50'].selected_us_unit = 'mm'
        self.input['Contracted D50'].selected_si_unit = 'mm'
        self.input['Selected borehole'] = Variable(
            'Selected borehole along contracted cross-section', 'uuid_dict', -1, {}, note='')
        self.input['Critical shear stress (τc)'] = Variable(
            "Critical shear stress (τc)", "float", 0.0, [], precision=4, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']], note='The level of shear required to erode the soil layer')

        self.input['Scour hole geometry'] = Variable(
            'Scour hole geometry', "class",
            ScourHole(app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Scour hole geometry'].is_pier_scour_hole = True

        self.theme = self.get_theme()
        # _, null_data = self.get_setting('Null data')
        # _, complexity_var = self.get_setting_var('Complexity')
        # self.sys_complexity = complexity_var.get_index(null_data)
        # if self.sys_complexity == null_data:
        #     self.sys_complexity = 0
        self.sys_complexity = self.get_system_complexity_index()

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend([
            'unknown', 'theme', 'gradations', 'approach_gradation_dict', 'contracted_gradation_dict', 'd50', 'd50_ft',
            'upstream_d50', 'upstream_d50_ft', 'critical_shear_stress', 'scour_depth', 'centerline',
            'centerline_streambed', 'channel_location', 'scour_reference_point', 'bh_dict', 'bh_uuid', 'bh',
            'bh_layers', 'layer_index', 'surface_d50', 'bcs_data', 'cross_section_x', 'cross_section_y',
            'is_computing_shear_decay', 'clear_my_own_results'])
        self.scour_depth = 0.0

        self.centerline = 0.0
        self.centerline_streambed = 0.0
        self.channel_location = 'main'
        self.scour_reference_point = None
        self.bh_dict = {}
        self.bh_uuid = None
        self.bh = None
        self.bh_layers = None
        self.layer_index = None

        self.surface_d50 = 0.0

        self.bcs_data = None

        self.cross_section_x = []
        self.cross_section_y = []

        self.is_computing_shear_decay = False

        # Results
        self.clear_my_own_results = clear_my_own_results

        # Approach Gradation
        self.results['Results'] = {}
        self.results['Results']['Approach D50'] = Variable(
            'Approach D50 (D50)', 'float', 0.0, [], precision=6, unit_type=['length'], native_unit='ft',
            us_units=self.us_short_length, si_units=self.si_short_length)
        self.results['Results']['Approach D50'].selected_si_unit = 'mm'
        self.results['Results']['Approach D50'].selected_us_unit = 'mm'

        # Contracted Gradation
        self.results['Results']['Contracted D50'] = Variable(
            'Contracted D50 (D50)', 'float', 0.0, [], precision=6, unit_type=['length'], native_unit='ft',
            us_units=self.us_short_length, si_units=self.si_short_length)
        self.results['Results']['Contracted D50'].selected_si_unit = 'mm'
        self.results['Results']['Contracted D50'].selected_us_unit = 'mm'

        self.results['Results']['Critical shear stress (τc)'] = Variable(
            "Critical shear stress (τc)", "float", 0.0, [], precision=4, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']], note='The level of shear required to erode the soil layer')

        self.results['Results']['Scour depth (ys)'] = Variable(
            'Scour depth (ys)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay results'] = {}

        self.results['Shear decay results']['Scour depth without shear decay (ys)'] = Variable(
            'Scour depth without shear decay (ys)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Shear decay results']['Scour depth with shear decay (ys)'] = Variable(
            'Scour depth with shear decay (ys)', 'float', 0.0, [], precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Shear decay results']['Unit discharge'] = Variable(
            'Unit discharge', 'float', 0.0, [], precision=2, unit_type=['area'], native_unit='ft',
            us_units=[['cfs/ft']], si_units=[['cms/m']])
        self.results['Shear decay results']['Shear decay curve shear'] = Variable(
            'Shear decay curve shear', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=4, unit_type=['stress'], native_unit='psf', us_units=[['psf']], si_units=[['pa']])
        self.results['Shear decay results']['Shear decay curve elevation'] = Variable(
            'Shear decay curve elevation', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Shear decay results']['Shear decay curve shear markers'] = Variable(
            'Shear decay curve shear markers', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=4, unit_type=['stress'], native_unit='psf', us_units=[['psf']], si_units=[['pa']])
        self.results['Shear decay results']['Shear decay curve elevation markers'] = Variable(
            'Shear decay curve elevation markers', 'float_list', 0.0, [],
            limits=(-sys.float_info.max, sys.float_info.max), precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay results']['Computed Hager number'] = Variable(
            'Computed Hager number', 'float', 0.0, [], precision=4, unit_type=['coefficient'])
        self.results['Shear decay results']['Max computed scour depth'] = Variable(
            'Max computed scour depth', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Shear decay results']['Max depth computed from Hager number'] = Variable(
            'Max depth computed from Hager number', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Shear decay results']['Max depth computed from CSU equation'] = Variable(
            'Max depth computed from CSU equation', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay results']['Unadjusted computed local shear'] = Variable(
            'Unadjusted computed local shear', 'float', 0.0, [], precision=4, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])

        self.warnings = []

        # Reference Manuals
        self.reference_pdfs['HEC-18'] = (
            'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 167)

        # plot
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Profile"
        # self.plots['profile']['Legend'] = 'best'
        # self.plots['profile']['index'] = 1

    def set_intermediate_d50(self):
        """Set the intermediate D50 values."""
        self.upstream_d50 = self.input['Approach D50'].get_val()
        if 'Results' in self.results:
            if 'Approach D50' in self.results['Results']:
                self.results['Results']['Approach D50'].set_val(self.upstream_d50)

        self.d50 = self.input['Contracted D50'].get_val()
        if 'Results' in self.results:
            if 'Contracted D50' in self.results['Results']:
                self.results['Results']['Contracted D50'].set_val(self.d50)

        self.critical_shear_stress = self.input['Critical shear stress (τc)'].get_val()
        if 'Results' in self.results:
            if 'Critical shear stress (τc)' in self.results['Results']:
                self.results['Results']['Critical shear stress (τc)'].set_val(self.critical_shear_stress)

    def update_gradation_lists(self):
        """Updates the gradation lists and dicts."""
        if self.gradations is not None:
            self.calculator.update_gradation_lists()
            self.d50 = self.calculator.d50
            return

    #     # Contracted gradation
    #     self.contracted_bhs = None
    #     self.contracted_bh = None
    #     if self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
    #         self.contracted_bh = self.gradations.input['Single borehole'].get_val()
    #     elif self.gradations.input['Contracted gradation input options'].get_val() == 'Multiple boreholes':
    #         self.contracted_bhs = self.gradations.input['Boreholes'].get_val()
    #         if self.input['Selected borehole'].get_val() == 'None' or self.input['Selected borehole'].get_val() == -1:
    #             self.contracted_bh = None
    #     else:  # if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50':
    #         self.input['Contracted D50'] = self.gradations.input['Contracted D50']

    #     # Approach gradation
    #     self.approach_gradation = None
    #     if self.gradations.input['Approach gradation input options'].get_val() == 'Single D50':
    #         self.input['Approach D50'] = self.gradations.input['Approach D50']
    #     elif self.gradations.input['Approach gradation input options'].get_val() == 'Gradations':
    #         if self.gradations.input['Approach gradations'].get_val().input['Approach locations'].get_val() == \
    #                 'Single gradation for entire cross-section':
    #             self.approach_gradation = self.gradations.input['Approach gradations'].get_val().input[
    #                 'Approach layer'].get_val()
    #         else:
    #             if self.input['Selected approach location'].get_val() == 'left overbank':
    #                 self.approach_gradation = self.gradations.input['Approach gradations'].get_val().input[
    #                     'Left overbank layer'].get_val()
    #             elif self.input['Selected approach location'].get_val() == 'right overbank':
    #                 self.approach_gradation = self.gradations.input['Approach gradations'].get_val().input[
    #                     'Right overbank layer'].get_val()
    #             else:
    #                 self.approach_gradation = self.gradations.input['Approach gradations'].get_val().input[
    #                     'Approach layer'].get_val()
    #     else:  # self.gradations.input['Approach gradation input options'].get_val() == 'None':
    #         self.input['Approach D50'].set_val(0.0)
    #         self.approach_gradation = None

    #     self.set_closest_borehole()
    #     self.set_layers()
    #     self.set_d50_for_approach_gradation()
    #     self.set_intermediate_d50()

    # def set_closest_borehole(self):
    #     """Get the closest contracted D50."""
    #     # Update the selectable boreholes
    #     if self.contracted_bhs is None:
    #         self.input['Selected borehole'].set_val(None)
    #         return
    #     self.bh_dict = {}
    #     bh_dict_names = {}
    #     bh_dict_names['None'] = 'None'
    #     for borehole in self.contracted_bhs.input['Boreholes'].get_val().item_list:
    #         self.bh_dict[borehole.uuid] = borehole
    #         bh_dict_names[borehole.uuid] = borehole.name
    #     self.input['Selected borehole'].value_options = bh_dict_names
    #     if not self.input['Selected borehole'].get_val() in self.bh_dict:
    #         self.input['Selected borehole'].set_val('None')
    #     if self.input['Selected borehole'].get_val() != 'None':
    #         sel_bh = self.input['Selected borehole'].get_val()
    #         self.contracted_bh = self.bh_dict[sel_bh]
    #         # Get the D50 at the surface (to set as the approach D50 for contraction scour)
    #         # Get the D50 at the eroded surface for starting D50 on Contracted section
    #         layer_index, _, _, _ = self.get_layer_index_from_elevation(self.centerline_streambed)
    #         self.contracted_bh.current_layer = layer_index
    #         self.surface_d50 = self.contracted_bh.get_d50(0)
    #         self.d50 = self.contracted_bh.get_d50(0)
    #         d50 = self.contracted_bh.get_d50()
    #         self.input['Contracted D50'].set_val(d50)
    #         self.critical_shear_stress = self.contracted_bh.get_critical_shear_stress()
    #         self.input['Critical shear stress (τc)'].set_val(self.critical_shear_stress)
    #         return

    #     closest_borehole = None
    #     closest_d50_distance = sys.float_info.max
    #     for borehole in self.contracted_bhs.input['Boreholes'].get_val().item_list:
    #         distance = abs(self.centerline - borehole.input['Centerline'].get_val())
    #         if distance < closest_d50_distance:
    #             closest_d50_distance = distance
    #             closest_borehole = borehole
    #     if closest_borehole is None:
    #         return
    #     self.input['Selected borehole'].set_val(closest_borehole.uuid)
    #     self.contracted_bh = self.bh_dict[self.input['Selected borehole'].get_val()]

    # def set_layers(self):
    #     """Set the layers for the selected borehole."""
    #     if self.contracted_bh == 'None' or self.contracted_bh is None:
    #         return
    #     self.bh_layers = self.contracted_bh.input['Layers'].value.item_list
    #     self.contracted_bh.compute_data()

    #     for i in range(len(self.bh_layers)):
    #         self.bh_layers[i].input['Plot options']['Gradation'].value.input['Data series'].value.item_list[0].input[
    #             'Plot series'].value.item_list[0] = self.contracted_bh.input['Plot options'][
    #                 'Soil layers'].value.input['Data series'].value.item_list[0].input[
    #                       'Plot series'].value.item_list[i]

    #     d50 = self.contracted_bh.get_d50()
    #     self.input['Contracted D50'].set_val(d50)
    #     self.critical_shear_stress = self.contracted_bh.get_critical_shear_stress()
    #     self.input['Critical shear stress (τc)'].set_val(self.critical_shear_stress)

    # def set_d50_for_approach_gradation(self):
    #     """Get the D50 for the approach gradation."""
    #     if self.approach_gradation is None:
    #         return
    #     if self.gradations.input['Approach gradation input options'].get_val() != 'Gradations':
    #         return
    #     self.upstream_d50 = self.approach_gradation.get_d50()
    #     self.input['Approach D50'].set_val(self.upstream_d50)

    # def get_scour_depth(self):
    #     """Returns the scour depth."""
    #     if self.get_can_compute():
    #         self.compute_data()
    #     return self.scour_depth

    def pop_property_vars(self, input_vars):
        """Drops the property variables."""
        if 'Unit weight of water (γw)' in input_vars:
            input_vars.pop('Unit weight of water (γw)')
        if 'Water density' in input_vars:
            input_vars.pop('Water density')
        if 'Unit weight of soil (γs)' in input_vars:
            input_vars.pop('Unit weight of soil (γs)')
        if 'Angle of repose (Θ)' in input_vars:
            input_vars.pop('Angle of repose (Θ)')

        return input_vars

    # def set_angle_of_repose(self):
    #     """Returns the angle of repose."""
    #     if self.gradations:
    #         if self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
    #             return
    #     else:
    #         return

    #     if self.bh_layers is None:
    #         return

    #     if self.layer_index is None:
    #         self.layer_index = 0

    #     if self.layer_index >= len(self.bh_layers):
    #         self.layer_index = len(self.bh_layers) - 1

    #     if 'Scour hole geometry' in self.input:
    #         self.input['Scour hole geometry'].get_val().input['Angle of repose'].set_val(self.bh_layers[
    #             self.layer_index].input['Angle of repose'].get_val())

    # def compute_scour_hole_geometry(self, pier_width=0.0):
    #     """Computes the scour hole geometry based on the scour depth.

    #     Returns:
    #         None
    #     """
    #     self.set_angle_of_repose()

    #     scour_hole = self.input['Scour hole geometry'].get_val()
    #     scour_hole.input['Scour depth'].set_val(self.scour_depth)
    #     if pier_width > 0.0:
    #         scour_hole.pier_width = pier_width
    #         scour_hole.is_pier_scour_hole = True
    #     if self.input['Scour hole geometry'].get_val().get_can_compute():
    #         self.input['Scour hole geometry'].get_val().compute_data()
    #         self.results['Bottom width'] = self.input['Scour hole geometry'].get_val().results['Bottom width']
    #         self.results['Top width'] = self.input['Scour hole geometry'].get_val().results['Top width']

    # def compute_shear(self, wse, elevation, mannings_n, unit_q):
    #     """Compute the contraction curve point for a given depth.

    #     Args:
    #         wse_elevation (float): The elevation of the water surface.
    #         elevation (float): The elevation at which to compute the contraction curve.
    #         mannings_n (float): The Manning's n value for the soil layer.
    #         unit_q (float): The unit discharge.

    #     Returns:
    #         float: The shear stress at the given elevation
    #     """
    #     _, gamma_w = self.get_setting('Unit weight of water (γw)', 62.4)
    #     _, k = self.get_setting('Manning constant')
    #     depth = wse - elevation
    #     if depth < 0.0:
    #         return 0.0
    #     shear = gamma_w * ((depth) ** (-7.0 / 3.0)) * ((mannings_n * unit_q / k) ** 2.0)
    #     return shear

    # def compute_alpha_beta_for_pier_shape(self):
    #     """Compute the alpha and beta values for the pier shape.

    #     Returns:
    #         tuple of float: The alpha and beta values
    #     """
    #     # shear_alpha=self.shear_alpha # Local shear stress amplification factor (4.37)
    #     # shear_beta=self.shear_beta   # Local shear decay factor (1.33)
    #     # shape = self.input['Pier stem parameters'].get_val().input['Pier shape (K1)'].get_val()
    #     shear_alpha = 4.37
    #     shear_beta = 1.33

    #     if 'Alpha value for local shear decay' in self.input:
    #         shear_alpha = self.input['Alpha value for local shear decay'].get_val()
    #     if 'Beta value for local shear decay' in self.input:
    #         shear_beta = self.input['Beta value for local shear decay'].get_val()

    #     return shear_alpha, shear_beta

    # def get_layer_index_from_elevation(self, elevation):
    #     """Get the index of the layer that contains the elevation.

    #     Args:
    #         elevation (float): The elevation to search for.

    #     Returns:
    #         tuple of int: The index of the layer, the top elevation of the layer, the bottom elevation of the layer,
    #         and the critical shear stress of the layer.
    #     """
    #     layer_index = -1
    #     cur_layer_top = -sys.float_info.max
    #     cur_layer_bottom = sys.float_info.max

    #     if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50' or \
    #             self.contracted_bh is None or self.bh_layers is None:
    #         return None, None, None, None

    #     while not cur_layer_top >= elevation > cur_layer_bottom and layer_index + 1 < len(self.bh_layers):
    #         layer_index += 1
    #         cur_layer = self.bh_layers[layer_index]
    #         cur_layer_top = cur_layer.input['Top elevation'].get_val()
    #         cur_layer_bottom = cur_layer.input['Bottom elevation'].get_val()
    #         cur_critical_shear = cur_layer.results['Shear decay parameters']['Critical shear stress (τc)'].get_val()

    #     return layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear

    # def get_next_layer_details(self, layer_index, wse, unit_q):
    #     """Get the next layer in the borehole.

    #     Args:
    #         layer_index (int): The index of the current layer.
    #         wse (float): The water surface elevation.
    #         unit_q (float): The unit discharge.

    #     Returns:
    #         tuple of float: The Manning's n, the shear stress, and the critical shear stress of the next layer.
    #     """
    #     if layer_index + 1 < len(self.bh_layers):
    #         lower_layer_n = self.bh_layers[layer_index + 1].results['Shear decay parameters']['Manning n'].get_val()
    #         lower_layer_top = self.bh_layers[layer_index + 1].input['Top elevation'].get_val()
    #         lower_layer_shear = self.compute_shear(wse, lower_layer_top, lower_layer_n, unit_q)
    #         lower_layer_critical = self.bh_layers[layer_index + 1].results['Shear decay parameters'][
    #             'Critical shear stress (τc)'].get_val()
    #         return lower_layer_n, lower_layer_shear, lower_layer_critical
    #     return None, None, None

    # def set_layer_plot_data(self, decay_x, decay_y, marker_x, marker_y, y_max=None, use_csu=None):
    #     """Set the data for the layer plot.

    #     Args:
    #         decay_x (list): The x values for the decay curve.
    #         decay_y (list): The y values for the decay curve.
    #         marker_x (list): The x values for the markers.
    #         marker_y (list): The y values for the markers.
    #         y_max (float): The maximum depth.
    #         use_csu (bool): Whether to use the CSU equation or the Hager equation for ymax.
    #     """
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.input['Number of items'].set_val(2)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.check_list_length()

    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #         'Plot series'].value.input['Number of items'].set_val(len(self.bh_layers))
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #         'Plot series'].value.check_list_length()

    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot series'].value.input['Number of items'].set_val(1)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot series'].value.check_list_length()

    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot points'].value.input['Number of items'].set_val(1)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot points'].value.check_list_length()

    #     if y_max is not None:
    #         num_lines = self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.input[
    #             'Number of items'].get_val()
    #         if num_lines == 0:
    #             grey_rgb = (127, 127, 127)
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.input[
    #                 'Number of items'].set_val(1)
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.check_list_length()

    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #                 'Label'].set_val('Y max')
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #                 'Line alignment'].set_val('horizontal')
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #                 'Line color'].set_val(grey_rgb)
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #                 'Line type'].set_val('dotted')
    #             # self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #             #      'Line width'].set_val('Y max')
    #             self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #                 'Text color'].set_val(grey_rgb)
    #         name = 'Y max'
    #         if use_csu:
    #             name = 'Y max (CSU)'
    #         elif use_csu is False:
    #             name = 'Y max (Hager)'
    #         self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #             'Label'].set_val(name)
    #         self.input['Plot options']['Shear decay'].value.input['Plot lines'].value.item_list[0].input[
    #             'Line intercepts'].set_val(y_max)

    #     shear_var = Variable('Shear stress', 'float_list', 0, [0.0], precision=2,
    #                          limits=(0.0, sys.float_info.max), unit_type=['stress'],
    #                          native_unit='psf', us_units=[['psf']], si_units=[['pa']],)

    #     elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
    #                              limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
    #                              native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

    #     index = 0
    #     for layer in self.bh_layers:
    #         layer_top = layer.input['Top elevation'].get_val()
    #         layer_bottom = layer.input['Bottom elevation'].get_val()
    #         critical_shear = layer.results['Shear decay parameters']['Critical shear stress (τc)'].get_val()
    #         x = [0.0, critical_shear, critical_shear, 0.0, 0.0]
    #         y = [layer_bottom, layer_bottom, layer_top, layer_top, layer_bottom]

    #         shear_var.set_val(x)
    #         elevation_var.set_val(y)

    #         self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #             'Plot series'].value.item_list[index].name = layer.input['Name'].get_val()

    #         self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #             'Plot series'].value.item_list[index].x_var = copy.deepcopy(shear_var)
    #         self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #             'Plot series'].value.item_list[index].y_var = copy.deepcopy(elevation_var)

    #         self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[0].input[
    #             'Plot series'].value.item_list[index].input = layer.input['Plot options']['Gradation'].value.input[
    #                 'Data series'].value.item_list[0].input['Plot series'].value.item_list[0].input

    #         index += 1

    #     shear_var.set_val(decay_x)
    #     elevation_var.set_val(decay_y)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot series'].value.item_list[0].input['Name'].set_val('Shear decay curve')
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot series'].value.item_list[0].x_var = copy.deepcopy(shear_var)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot series'].value.item_list[0].y_var = copy.deepcopy(elevation_var)

    #     shear_var.set_val(marker_x)
    #     elevation_var.set_val(marker_y)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot points'].value.item_list[0].input['Name'].set_val('')
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot points'].value.item_list[0].x_var = copy.deepcopy(shear_var)
    #     self.input['Plot options']['Shear decay'].value.input['Data series'].value.item_list[1].input[
    #         'Plot points'].value.item_list[0].y_var = copy.deepcopy(elevation_var)
