"""Provides a class that will Handle user-selected for boundary conditions."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import all_shapes, \
    closed_shapes, open_shapes
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control_calc import InletControlCalc


class InletControlData(VariableGroup):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Inlet Control Class.

        Args:
            app_data (AppData): The application data
            model_name (str): The name of the model
            project_uuid (str): The project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Inlet Control CalcData'
        self.type = 'InletControlCalc'

        self.calculator = InletControlCalc()

        self.update_inpute_after_compute = True

        # Input
        self.inlet_type_list = [
            'best available',
            'interpolation: circular or elliptical', 'interpolation: arch or embedded',
            'manual polynomial',
        ]
        self.input['inlet type'] = Variable(
            'inlet type', "list", 0, self.inlet_type_list,
            note='"best available" uses polynomial if available, otherwise interpolation, if shape is user-defined, '
            'arch/embedded interpolation is used')

        self.inlet_configuration_list = [
            'beveled edge', 'mitered to conform to slope', 'square edge with headwall', 'thin edge projecting'
        ]
        self.input['Inlet configuration'] = Variable('Inlet configuration', "list", 0, self.inlet_configuration_list)

        # Computation variables
        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow,
                                              select_name='specify flow(s) as:', name_append='flow'))

        allowed_shapes = 'closed'
        self.open_shapes = open_shapes
        self.closed_shapes = closed_shapes
        self.all_shapes = all_shapes
        self.shapes = self.all_shapes

        # if allowed_shapes == 'open':
        #     self.shapes = self.open_shapes
        if allowed_shapes == 'closed':
            self.shapes = self.closed_shapes
        self.input['Shape'] = Variable('Shape', "list", 0, self.shapes)

        self.input['Rise'] = Variable(
            'Rise', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Span'] = Variable(
            'Span', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.input['Full flow area'] = Variable(
            'Full flow area', 'float', 0.0, [], precision=2, unit_type=['area'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.input['Slope'] = Variable(
            'Slope', 'float', 0.0, [], limits=[-10.0, 10.0], precision=6, unit_type=['slope'], native_unit='ft/ft',
            us_units=[['ft/ft']], si_units=[['m/m']])

        # self.ih_hw_list = []
        # self.flow_list = []
        self.flow_unk_sorted = None
        self.flow_interp = None

        self.flow_at_half_rise = 0.0
        self.flow_at_triple_rise = 0.0

        self.mannings_n_calc = None

        max_value = sys.float_info.max

        # Polynomial
        self.input['A'] = Variable('A', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['BS'] = Variable('BS', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['C'] = Variable('C', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['DIP'] = Variable('DIP', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['EE'] = Variable('EE', 'float', 0.0, precision=14, limits=(-max_value, max_value))
        self.input['F'] = Variable('F', 'float', 0.0, precision=14, limits=(-max_value, max_value))

        self.use_qad_poly = False

        # Interpolation
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'inlet_type_list', 'inlet_configuration_list', 'open_shapes', 'closed_shapes', 'all_shapes',
            'shapes', 'flow_at_half_rise', 'flow_at_triple_rise', 'user_x',
            'user_cir_beveled_ke', 'user_cir_beveled_hwd', 'user_cir_mitered_ke', 'user_cir_mitered_hwd',
            'user_cir_square_ke', 'user_cir_square_hwd', 'user_cir_projecting_ke', 'user_cir_projecting_hwd',
            'user_arch_beveled_ke', 'user_arch_beveled_hwd', 'user_arch_mitered_ke', 'user_arch_mitered_hwd',
            'user_arch_square_ke', 'user_arch_square_hwd', 'user_arch_projecting_ke', 'user_arch_projecting_hwd',
            'use_qad_poly', 'flow_unk_sorted', 'flow_interp'
        ])

        # Both
        self.input['KE'] = Variable('KE', 'float', 0.0, limits=(-max_value, max_value))
        self.input['SR'] = Variable('SR', 'float', 0.0, limits=(-max_value, max_value))

        # Interpolation
        # X for interpolation: Q/AD^05; for all interpolation curves
        self.user_x = [0, 0.25, 0.5, 1.0, 2, 3, 3.5, 4, 5, 6, 7, 8, 9]

        # Chart 51-B Circular and Elliptical Structural Plate Corrugated Metal Conduits
        # Beveled Edge
        # KE = .2
        self.user_cir_beveled_ke = 0.2
        # SR = .5
        # HW/D
        self.user_cir_beveled_hwd = [
            0.031, 0.225, 0.33, 0.49, 0.74, 0.96, 1.0825, 1.205, 1.48, 1.81, 2.2, 2.66, 3.16
        ]
        # Mitered
        # KE = .7
        self.user_cir_mitered_ke = 0.7
        # SR = -.7
        self.user_cir_mitered_hwd = [
            0.06, 0.285, 0.39, 0.56, 0.84, 1.07, 1.23, 1.42, 1.89, 2.43, 3.05, 3.67, 4.29
        ]
        # Square Edge
        # KE = 0.5
        self.user_cir_square_ke = 0.5
        # SR = 0.5
        self.user_cir_square_hwd = [
            0.03, 0.23, 0.33, 0.51, 0.775, 1.025, 1.14, 1.27, 1.59, 1.98, 2.5, 3.07, 3.64
        ]
        # Projecting
        # KE = 0.9
        self.user_cir_projecting_ke = 0.9
        # SR = 0.5
        self.user_cir_projecting_hwd = [
            0.04, 0.235, 0.345, 0.52, 0.84, 1.125, 1.2625, 1.4, 1.79, 2.34, 2.99, 3.64, 4.29
        ]

        # Chart 52-B High and low profile Structural Plate Corrugated Metal Arch
        # Beveled Edge
        # KE = .2
        self.user_arch_beveled_ke = 0.2
        # SR = .5
        # HW/D
        self.user_arch_beveled_hwd = [
            0.031, 0.1655, 0.3, 0.44, 0.69, 0.88, 1.015, 1.15, 1.47, 1.8, 2.22, 2.66, 3.16
        ]
        # Mitered
        # KE = .7
        self.user_arch_mitered_ke = 0.7
        # SR = -.7
        self.user_arch_mitered_hwd = [
            0.04, 0.22, 0.35, 0.5, 0.77, 1.06, 1.255, 1.45, 1.91, 2.46, 3.07, 3.68, 4.29
        ]
        # Square Edge
        # KE = 0.5
        self.user_arch_square_ke = 0.5
        # SR = 0.5
        self.user_arch_square_hwd = [
            0.03, 0.155, 0.28, 0.46, 0.73, 0.96, 1.11, 1.26, 1.59, 2.01, 2.51, 3.08, 3.64
        ]
        # Projecting
        # KE = 0.9
        self.user_arch_projecting_ke = 0.9
        # SR = 0.5
        self.user_arch_projecting_hwd = [
            0.03, 0.19, 0.3, 0.49, 0.81, 1.11, 1.26, 1.41, 1.82, 2.38, 3.02, 3.66, 4.3
        ]

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

        self.hw = []

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        input_vars['inlet type'] = copy.deepcopy(self.input['inlet type'])
        sys_complexity = self.get_setting_var('Complexity')[1].value
        if sys_complexity < 2:
            if self.input['inlet type'].get_val() == 'manual polynomial':
                self.input['inlet type'].set_val(0)
                input_vars['inlet type'].set_val(0)
            input_vars['inlet type'].value_options.remove('manual polynomial')

        if self.input['inlet type'].get_val() == 'manual polynomial':
            input_vars['KE'] = self.input['KE']
            input_vars['SR'] = self.input['SR']
            input_vars['A'] = self.input['A']
            input_vars['BS'] = self.input['BS']
            input_vars['C'] = self.input['C']
            input_vars['DIP'] = self.input['DIP']
            input_vars['EE'] = self.input['EE']
            input_vars['F'] = self.input['F']
        else:
            input_vars['Inlet configuration'] = self.input['Inlet configuration']

        return input_vars

    def set_ke(self):
        """Sets the KE value, if not specified by the user."""
        self.prepare_for_compute()
        self.calculator.input_dict, self.calculator.plot_dict = self.prepare_input_dict()

        return self.calculator.set_ke()

    def initialize_inlet_data_for_culvert(self, shape, span, rise, fullflow_area, slope, mannings_n_data):
        """Compute and set coefficient and data for the culvert to perform inlet computations.

        Args:
            rise (float): the rise of the culvert
            critical_depth_with_flow_at_half_rise (float): critical depth for the flow at half-rise
            critical_flow_area_with_flow_at_half_rise (float): flow area for the flow at half-rise

        Returns:
            bool: True if successful

        """
        self.input['Shape'].set_val(shape)
        self.input['Rise'].set_val(rise)
        self.input['Span'].set_val(span)
        self.input['Full flow area'].set_val(fullflow_area)
        self.input['Slope'].set_val(slope)

        mannings_n_data.prepare_for_compute()
        mannings_n_calc = mannings_n_data.calculator
        mannings_n_calc.input_dict, mannings_n_calc.plot_dict = mannings_n_data.prepare_input_dict()

        self.get_can_compute()
        result = False
        if self.can_compute:
            result = self.calculator.initialize_inlet_data_for_culvert(shape, span, rise, fullflow_area, slope,
                                                                       mannings_n_calc)
        return result
