"""Bridge Deck Table Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class RatingCurveTable(TableData):
    """Override the table class, so we can draw plots based on widths."""
    def __init__(self, theme, name, plot_names, name_of_items='points', stand_alone_calc=True, input=None, min_items=1,
                 max_items=sys.maxsize, show_increment=False, app_data=None, model_name=None, project_uuid=None,
                 define_cross_section=True):
        """Initializes the calculator.

        Args:
            theme (dict): the theme
            name (str): the name of the calculator
            plot_names (list of str): list of plot names
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the calculator is a stand-alone calculator
            input (dict): dictionary of input variables
            min_items (int): minimum number of items
            max_items (int): maximum number of items
            show_increment (bool): whether to show the increment
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
            define_cross_section (bool): True to define the cross-section by station and elevation; False to define
                the Manning's n values by station
        """
        super().__init__(theme=theme, name=name, plot_names=plot_names, name_of_items=name_of_items,
                         stand_alone_calc=stand_alone_calc, input=input, min_items=min_items, max_items=max_items,
                         show_increment=show_increment, app_data=app_data, model_name=model_name,
                         project_uuid=project_uuid)

        # Set to False, if this is to display the cross-section only, but define the Manning's n values by station
        self.calculator = TableCalc()

        # self.head = 'Elevation'
        self.channel_invert_elevation = 0.0

        self.flow_var = copy.copy(input['Flow'])
        self.elevation_var = copy.copy(input['Elevation'])
        self.depths_var = copy.copy(input['Depth'])
        self.velocity_var = copy.copy(input['Velocity'])

        self.input['Table options'].value.input['Head'] = Variable('Head', "list", 0, ['Depth', 'Elevation'],
                                                                   complexity=1)

        self.input['Data input'].value.input['Flow'] = copy.copy(self.flow_var)
        self.input['Data input'].value.input['Elevation'] = copy.copy(self.elevation_var)
        self.input['Data input'].value.input['Depth'] = copy.copy(self.depths_var)
        self.input['Data input'].value.input['Velocity'] = copy.copy(self.velocity_var)

        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['X axis'].set_val('Tailwater Flows')
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Y axis'].set_val('Tailwater depths')

        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Plot yy'].set_val(True)

        # self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
        #     'Plot series'].value.item_list[0].input['XX axis'].set_val("Manning's n stations display")
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['YY axis'].set_val('Tailwater velocities')

        # Cross-section data
        wsp_color = (70, 118, 202)
        wsp_fill_color = (0, 0, 255)
        velocity_color = (255, 140, 0)
        if self.theme is not None:
            wsp_color = self.theme['Plot WSP color']
            wsp_fill_color = self.theme['Plot WSP fill color']
            velocity_color = self.theme['Plot velocity color']

        # Set cross-section plot options
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line color'].set_val(wsp_color)
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill color'].set_val(wsp_fill_color)
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill pattern'].set_val('solid')
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line type'].set_val('solid')
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line width'].set_val(2.0)

        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line color yy'].set_val(velocity_color)
        # self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
        #     'Plot series'].value.item_list[0].input['Fill color yy'].set_val(soil_fill_color)
        # self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
        #     'Plot series'].value.item_list[0].input['Fill pattern yy'].set_val('sand')
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line type yy'].set_val('dash-dot')
        self.input['Plot options'][name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line width yy'].set_val(1.5)

    def get_input_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = super().get_input_tab_group(unknown=unknown)
        input_vars['Table options']['Head'] = self.input['Table options'].value.input['Head']

        return input_vars

    def get_input_dual_tab_group(self, unknown=None):
        """Get the input dual tab group.

        Args:
            unknown (str): the unknown variable

        Returns:
            dict: the input tab group
        """
        input_dual = super().get_input_dual_tab_group(unknown=unknown)

        if self.input['Table options'].value.input['Head'].get_val() == 'Elevation':
            elevations = self.input['Data input'].value.input['Elevation'].get_val()
            depths = [x + self.channel_invert_elevation for x in elevations]
            self.input['Data input'].value.input['Depth'].set_val(depths)
            if 'Depth' in input_dual['Data table']:
                input_dual['Data table'].pop('Depth')
            self.input['Plot options']['Rating curve'].value.input['Data series'].value.item_list[0].input[
                'Plot series'].value.item_list[0].input['Y axis'].set_val('Tailwater elevation')
        else:
            depths = self.input['Data input'].value.input['Depth'].get_val()
            elevations = [x + self.channel_invert_elevation for x in depths]
            self.input['Data input'].value.input['Elevation'].set_val(elevations)
            if 'Elevation' in input_dual['Data table']:
                input_dual['Data table'].pop('Elevation')
            self.input['Plot options']['Rating curve'].value.input['Data series'].value.item_list[0].input[
                'Plot series'].value.item_list[0].input['Y axis'].set_val('Tailwater depths')

        return input_dual
