"""Provides a class that will handle the weir-flow computations across a roadway."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.culvert.roadway_calc import RoadwayCalc
from xms.HydraulicToolboxCalc.hydraulics.weir.weir_data import WeirData


class RoadwayData(VariableGroup):
    """Provides a class that will handle the weir-flow computations across a roadway."""

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Roadway Class.

        Args:
            app_data (AppData): application data (settings, etc.).
            model_name (str): the name of the model.
            project_uuid (uuid): the project uuid.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Roadway CalcData'
        self.type = 'RoadwayCalc'

        self.calculator = RoadwayCalc()

        # Input
        max_value = sys.float_info.max

        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(
                2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow, select_name='specify flow(s) as:',
                name_append='flow'))

        roadway_shape_list = ['constant roadway elevation', 'irregular roadway']
        self.input['Roadway profile shape'] = Variable('Roadway profile shape', "list", 0, roadway_shape_list)

        # self.input['Stations'] = Variable(
        #     'Stations', 'float_list', 0, [0.0, 0.0, 0.0], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        # self.input['Elevations'] = Variable(
        #     'Elevations', 'float_list', 0, [0.0, 0.0, 0.0], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        embankment_color = (10, 10, 10)
        embankment_fill_color = (10, 10, 10)
        self.theme = app_data.get_theme()
        if self.theme is not None:
            embankment_color = self.theme['Plot embankment color']
            embankment_fill_color = self.theme['Plot embankment fill color']

        crest_geom = {
            'Roadway stations': Variable(
                'Roadway stations', 'float_list', 0, [0.0, 0.0, 0.0], limits=(-max_value, max_value), precision=2,
                unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Roadway elevations': Variable(
                'Roadway elevations', 'float_list', 0, [0.0, 0.0, 0.0], precision=2, limits=(-max_value, max_value),
                unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        }

        name = 'Irregular roadway geometry'
        self.input[name] = Variable(
            name, 'table', TableData(self.theme, name=name, plot_names=['Bridge cross-section'],
                                     input=copy.deepcopy(crest_geom), min_items=2,
                                     app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Elevation',
            line_color=embankment_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=embankment_fill_color, pattern='sand')

        self.input['Roadway surface'] = Variable(
            'Roadway surface', "list", 0, ['paved', 'gravel', 'user-defined'])
        self.input['Weir coefficient'] = Variable(
            'Weir coefficient', 'float', 0.0, [], limits=[0.0, 10.0], precision=4, unit_type=['coefficient'],
            native_unit='coefficient', us_units=[['coefficient']], si_units=[['coefficient']])

        self.input['Roadway top width'] = Variable(
            'roadway top width (across roadway)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['First roadway station'] = Variable(
            'First roadway station', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.input['Crest length'] = Variable(
            'crest length (along roadway)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Crest elevation'] = Variable(
            'Crest elevation', 'float', 0.0, [], limits=(-max_value, max_value), precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Tailwater'] = Variable(
            'Tailwater', 'UserArray', UserArray(
                2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                select_name='specify tailwater depth(s) as:', name_append='Tailwater'))
        self.input['Tailwater elevations'] = Variable(
            'Tailwater elevations', 'UserArray', UserArray(
                2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                select_name='specify tailwater elevation(s) as:', name_append='Tailwater'))

        # Intermediate
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'weir'
        ])

        weir_data = WeirData(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        weir_data.prepare_for_compute()
        self.weir = weir_data.calculator
        self.weir.input_dict, self.weir.plot_dict = weir_data.prepare_input_dict()
        # self.weir = WeirCalc(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.warnings = []

        # Results
        self.results = {}

        self.results['Flows'] = Variable(
            'Flow(s)', 'float_list', 0.0, [], unit_type=['flow'], native_unit='cfs', us_units=self.us_flow,
            si_units=self.si_flow)

        self.results['Tailwater elevations'] = Variable(
            'Tailwater elevations', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Tailwater depths'] = Variable(
            'Tailwater depths', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Tailwater velocities'] = Variable(
            'Tailwater velocities', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['mph', 'yd/s', 'ft/s', 'in/s']], si_units=[['km/hr', 'm/s', 'mm/s']])

        self.hw = []

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        input_vars['Roadway profile shape'] = self.input[
            'Roadway profile shape']

        roadway_type = self.input['Roadway profile shape'].get_val()
        if roadway_type == 'constant roadway elevation':
            input_vars['First roadway station'] = self.input[
                'First roadway station']
            input_vars['Crest length'] = self.input['Crest length']
            input_vars['Crest elevation'] = self.input['Crest elevation']
        elif roadway_type == 'irregular roadway':
            input_vars['Irregular roadway geometry'] = self.input['Irregular roadway geometry']

        input_vars['Roadway surface'] = self.input['Roadway surface']
        if input_vars['Roadway surface'].get_val() == 'user-defined':
            input_vars['Weir coefficient'] = self.input['Weir coefficient']

        input_vars['Roadway top width'] = self.input['Roadway top width']

        return input_vars

    def compute_data_for_wse(self, wse, tailwater=0.0):
        """Computes the data possible; stores results in self.

        Args:
            wse (float): the water surface elevation upstream
            tailwater (float): the tailwater surface elevation downstream
        Returns:
            tuple(bool, flow): (True if successful, resulting flow for given wse
        """
        self.get_can_compute()
        return self.calculator.compute_data_for_wse(wse, tailwater)
