"""CalcData for performing Manning's N operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math

# 2. Third party modules
from sortedcontainers import SortedDict

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class ManningNCalc(Calculator):
    """A class that defines a channel and performs Manning's n computations."""
    def __init__(self):
        """Initializes the Manning's n calculator."""
        super().__init__()
        self.unk_flow_sorted = None

        self.original_results = [
            'Flows', 'WSE', 'Depths',
            'Width of spread', 'Gutter depression', 'Ratio of gutter flow to total flow', 'Grate flow area',
            'Ratio of grate flow to total flow', 'Gutter flow area',
            'Sequent depths', 'WSE stations', 'Critical depth', 'Critical elevation',
            'Flow area', 'Wetted perimeter', 'Hydraulic radius', 'Hydraulic depth', 'Average velocity', 'Top width',
            'Froude number', 'Critical velocity', 'Critical slope', 'Critical top width', 'Critical flow area',
            'Max shear stress', 'Average shear stress', 'Composite n computation method', 'Composite n value',
            'Reynolds number', 'Flow stability'
        ]

    # def assign_results(self, unk_list):
    #     """Assigns the results from a computational run to the appropriate locations.

    #     Args:
    #         unk_list (list of floats): list of the unknowns that were solved by Manning n calc
    #     """
    #     # Assign the results
    #     if self.unknown == 'Head':
    #         # WSE
    #         self.normal_wses = unk_list
    #         # WSE stations (so the normal depth can be displayed)
    #         self.results['WSE stations'].append(self.normal_wses_stations)
    #         # depths
    #         depth_list = []
    #         for wse in unk_list:
    #             depth_list.append(wse - self.low_elev)
    #             self.results['Depths'].append(depth_list[-1])
    #             self.results['WSE'].append(wse)
    #             self.normal_depths = depth_list[-1]
    #             self.compute_hydraulic_parameters()
    #         self.normal_depths = depth_list
    #     elif self.unknown == 'Flows':
    #         for flow in unk_list:
    #             self.results[self.unknown].append(flow)
    #             self.compute_hydraulic_parameters()
    #     else:
    #         if self.unknown not in self.results:
    #             self.results[self.unknown] = []
    #         self.results[self.unknown].append(unk_list[0])

    #         self.compute_hydraulic_parameters()

    def assign_result(self, unk: float, cur_wse: float):
        """Assigns the results from a computational run to the appropriate locations.

        Args:
            unk (float): the unknown that was solved by Manning n calc
            cur_wse (float): the current water surface elevation
        """
        # Assign the results
        if self.unknown == 'Head':
            # WSE
            self.normal_wse = unk
            self.normal_wses.append(unk)
            # WSE stations (so the normal depth can be displayed)
            self.results['WSE stations'].append(self.normal_wses_stations)
            # depths
            wse = unk
            #
            if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter':
                wse = self.wse
                self.normal_depth = self.curb_depth
            else:
                self.normal_depth = wse - self.low_elev

            self.compute_hydraulic_parameters(flow=self.flow, depth=self.normal_depth, wse=wse)
            self.normal_depths.append(self.normal_depth)
        elif self.unknown == 'Flows':
            flow = unk
            self.results[self.unknown].append(flow)
            self.normal_wse = cur_wse
            self.normal_wses.append(cur_wse)
            self.compute_hydraulic_parameters()
        else:
            self.normal_wse = cur_wse
            self.normal_wses.append(cur_wse)
            if self.unknown not in self.results:
                self.results[self.unknown] = []
            self.results[self.unknown].append(unk)

            self.compute_hydraulic_parameters()

    def check_for_vertical_wall_warning(self):
        """Checks whether there was vertical walls added to the channel.

        Returns:
            (bool): True if there was vertical walls added, False if not
        """
        return self.input_dict['calc_data']['Geometry']['calculator'].check_for_vertical_wall_warning(
            self.normal_wses, self.results['Critical elevation'])

    def get_can_compute_critical(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): the variable that is unknown and is being calculated

        Returns:
            bool: True if can compute
        """
        self.warnings = {}
        result = True
        int_result = True  # intermediate results

        # Check for and gather warnings from the geometry
        geom_calc = self.input_dict['calc_data']['Geometry']['calculator']
        geom_calc.unknown = unknown
        int_result = geom_calc._get_can_compute()
        if not int_result:
            result = False
            self.warnings.update(geom_calc.warnings)

        n_calc = self.input_dict['calc_data']['Composite n']['calculator']
        n_calc.unknown = unknown
        int_result, warnings = n_calc.get_can_compute()
        if not int_result:
            result = False
            self.warnings.update(warnings)

        found = False
        if unknown != 'Flows':
            flows = self.input_dict['calc_data']['Flows']
            if hasattr(flows, '__len__') and len(flows) < 1:
                self.warnings['Flows'] = "Please enter a flow"
                result = False
            else:
                for flow in flows:
                    if flow > 0.0:
                        found = True
                        break
                if not found:
                    self.warnings['Flows'] = "Please enter a positive, non-zero flow"
                    result = False
        if unknown != 'Head':
            depths = self.input_dict['calc_data']['Depths']
            elevations = self.input_dict['calc_data']['WSE']
            if self.input_dict['calc_data']['Head'] == 'Depth':
                if hasattr(depths, '__len__') and len(depths) < 1:
                    self.warnings['Depths'] = "Please enter a depth"
                    result = False
                else:
                    for depth in depths:
                        if depth > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['Depths'] = "Please enter a positive, non-zero depth"
                        result = False
            elif self.input_dict['calc_data']['Head'] == 'Elevation':
                if hasattr(elevations, '__len__') and len(elevations) < 1:
                    self.warnings['Elevations'] = "Please enter a water surface elevation"
                    result = False
                else:
                    for wse in elevations:
                        if wse > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['Elevations'] = "Please enter a positive, non-zero water surface elevation"
                        result = False
            elif self.input_dict['calc_data']['Head'] == 'Width of spread':
                width_of_spread = self.input_dict['calc_data']['Width of spread']
                if hasattr(width_of_spread, '__len__') and len(width_of_spread) < 1:
                    self.warnings['Width of spread'] = "Please enter a width of spread"
                    result = False
                else:
                    for width in width_of_spread:
                        if width > 0.0:
                            found = True
                            break
                    if not found:
                        self.warnings['Width of spread'] = "Please enter a positive, non-zero width of spread"
                        result = False

        return result

    def _get_can_compute(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): the variable being computed

        Returns:
            bool: True if can compute
        """
        _, self.zero_tol = self.get_data('Zero tolerance', 1e-6)

        if unknown is None:
            self.unknown = self.input_dict['calc_data']['Calculate']
            unknown = self.unknown

        self.update_mannings_n_data()

        self.can_compute_critical = self.get_can_compute_critical(unknown)
        self.can_compute_normal = copy.copy(self.can_compute_critical)
        _, zero_slope = self.get_data('Horizontal tolerance', 0.000001)
        if unknown != 'Slope' and self.input_dict['calc_data']['Slope'] < zero_slope:
            if unknown in ['Flows', 'Head']:
                self.warnings['Slope'] = "Please enter the slope for normal depth computations"
            else:
                self.warnings['Slope'] = "Please enter the slope"
            self.can_compute_normal = False
        return self.can_compute_critical

    def update_mannings_n_data(self):
        """Update the Manning's n data for the composite n calculations."""
        # Update variables needed for the Manning's n computations
        geom_var = self.input_dict['calc_data']['Geometry']['calculator']
        geom_var.input_dict = copy.copy(self.input_dict)
        geom_var.input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Geometry'])
        geom_var.update_determine_embedment_depth()
        embed_var = self.input_dict['calc_data']['Geometry']['Embedment']['calculator']
        embed_var.input_dict = copy.copy(self.input_dict)
        embed_var.input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Geometry']['Embedment'])
        shape = self.input_dict['calc_data']['Geometry']['Shape']
        closed_shape = geom_var.determine_if_shape_is_closed()
        embedment_depth = embed_var.get_embedment_depth()
        embedment_present = embedment_depth > self.zero_tol
        gradation_present = embedment_present and embed_var.input_dict['calc_data']['Specify gradation']

        # Gradations calculations data
        gradation_layer = geom_var.input_dict['calc_data']['Embedment']['Gradation layer']['calculator']
        gradation_layer.input_dict = copy.copy(self.input_dict)
        gradation_layer.input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Geometry']['Embedment'][
            'Gradation layer'])
        d50 = gradation_layer.get_d50()
        d84 = gradation_layer.get_d84()
        self.input_dict['calc_data']['Composite n']['calculator'].d50 = d50
        self.input_dict['calc_data']['Composite n']['calculator'].d84 = d84
        self.input_dict['calc_data']['Composite n']['calculator'].hyd_radius = self.hyd_radius
        self.input_dict['calc_data']['Composite n']['calculator'].average_flow_depth = self.average_flow_depth
        self.input_dict['calc_data']['Composite n']['calculator'].friction_slope = self.input_dict['calc_data']['Slope']

        self.input_dict['calc_data']['Composite n']['calculator'].shape = shape
        self.input_dict['calc_data']['Composite n']['calculator'].open_shape = not closed_shape
        self.input_dict['calc_data']['Composite n']['calculator'].embedment_present = embedment_present
        self.input_dict['calc_data']['Composite n']['calculator'].total_embedment = embedment_depth
        self.input_dict['calc_data']['Composite n']['calculator'].gradation_present = gradation_present
        self.input_dict['calc_data']['Composite n']['calculator'].stations = geom_var.stations
        self.input_dict['calc_data']['Composite n']['calculator'].elevations = geom_var.elevations
        self.input_dict['calc_data']['Composite n']['calculator'].input_dict = copy.copy(self.input_dict)
        self.input_dict['calc_data']['Composite n']['calculator'].input_dict['calc_data'] = copy.copy(
            self.input_dict['calc_data']['Composite n'])
        no_station_overlap = self.determine_station_overlap(geom_var)
        self.input_dict['calc_data']['Composite n']['calculator'].update_lists(
            no_station_overlap=no_station_overlap)

        # Data exchange between Geometry and Manning's n
        self.input_dict['calc_data']['Geometry']['Composite n'] = self.input_dict['calc_data']['Composite n']

        geom_var.input_dict = copy.copy(self.input_dict)
        geom_var.input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Geometry'])

    def determine_station_overlap(self, geom_var):
        """Determine the n parameters for the composite n calculations."""
        direction = None
        no_station_overlap = True

        if geom_var.shape_is_closed:
            # If the shape is closed, we don't need to check for station overlap
            no_station_overlap = False
        else:
            for index in range(len(geom_var.stations) - 1):
                distance = geom_var.stations[index + 1] - geom_var.stations[index]
                # No longer handle 'negative' direction, because the stations are already processed to be increasing
                if distance < 0.0:
                    no_station_overlap = False
                    break
                elif distance > 0.0:
                    if direction is None:
                        direction = 'positive'
                    elif direction == 'positive':
                        continue
                else:
                    no_station_overlap = False
                    break

        return no_station_overlap

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation.
        """
        # self.input_dict['calc_data']['Geometry']['calculator'].clear_results()

        # Remove any results created from calculating an unknown other than depth or flow:
        self.results = {k: v for k, v in self.results.items() if k in self.original_results}

        # Intermediate Results
        self.high_elev = 0.0
        self.low_elev = 0.0
        self.normal_wses_stations = []
        self.flows = 0.0
        self.normal_wse = 0.0
        self.normal_wses = []
        self.normal_depth = 0.0
        self.normal_depths = []
        self.cur_wse = 0.0

        self.results['Flows'] = []
        self.results['WSE'] = []
        self.results['Depths'] = []
        self.results['WSE stations'] = []
        self.results['Critical depth'] = []
        self.results['Critical elevation'] = []

        self.results['Sequent depths'] = []

        self.results['Flow area'] = []
        self.results['Wetted perimeter'] = []
        self.results['Hydraulic radius'] = []
        self.results['Hydraulic depth'] = []
        self.results['Average velocity'] = []
        self.results['Top width'] = []
        self.results['Froude number'] = []
        self.results['Critical velocity'] = []
        self.results['Critical slope'] = []
        self.results['Critical top width'] = []
        self.results['Critical flow area'] = []
        self.results['Max shear stress'] = []
        self.results['Average shear stress'] = []

        self.results['Composite n computation method'] = []
        self.results['Composite n value'] = []

        self.results['Reynolds number'] = []
        self.results['Flow stability'] = []

    def initialize_geometry(self):
        """Initialize the geometry of the cross-section to prepare for calculations.
        """
        self.input_dict['calc_data']['Geometry']['calculator'].update_cross_section = True
        if self.input_dict['calc_data']['Composite n']['n entry'] == 'specify n by station':
            self.input_dict['calc_data']['Geometry']['calculator'].update_cross_section = False
            self.input_dict['calc_data']['Geometry']['calculator'].stations = copy.copy(
                self.input_dict['calc_data']['Composite n']["Manning's n by station"]['calculator'].stations)
            self.input_dict['calc_data']['Geometry']['calculator'].elevations = copy.copy(
                self.input_dict['calc_data']['Composite n']["Manning's n by station"]['calculator'].elevations)
            self.input_dict['calc_data']['Geometry']['calculator'].channel_mannings_n = copy.copy(
                self.input_dict['calc_data']['Composite n']["Manning's n by station"]['calculator'].mannings_n_long)
        self.input_dict['calc_data']['Geometry']['calculator'].initialize_geometry()

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Set this variable to None, so we rebuild it:
        self.unk_flow_sorted = None

        self.initialize_geometry()  # Initialize geometry before getting thalweg and modifying depths/wse

        thalweg_elev = self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
            'Thalweg invert elevation']
        if self.input_dict['calc_data']['Geometry']['Shape'] == 'cross-section' and len(self.input_dict['calc_data'][
                'Geometry']['calculator'].channel_elevations) > 0:
            thalweg_elev = min(self.input_dict['calc_data']['Geometry']['calculator'].channel_elevations)

        # Convert depths to wse
        if self.input_dict['calc_data']['Head'] == 'Depth':
            # self.input['Elevations'].set_val([])
            elevations = []
            for depth in self.input_dict['calc_data']['Depths']:
                wse = depth + thalweg_elev + self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth
                elevations.append(wse)
                self.normal_depth = depth
            self.input_dict['calc_data']['WSE'] = elevations
        # Convert wse to depths
        elif self.input_dict['calc_data']['Head'] == 'Elevation':
            # self.input['Depths'].set_val([])
            depths = []
            for wse in self.input_dict['calc_data']['WSE']:
                depth = wse - thalweg_elev - self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth
                depths.append(depth)
                self.normal_depth = depth
            self.input_dict['calc_data']['Depths'] = depths

        # Update the Composite n calculator input_dict

        # Data exchange between Geometry and Manning's n
        self.input_dict['calc_data']['Geometry']['Head'] = self.input_dict['calc_data']['Head']
        self.input_dict['calc_data']['Geometry']['WSE'] = self.input_dict['calc_data']['WSE']
        if 'Width of spread' in self.input_dict['calc_data']:
            self.input_dict['calc_data']['Geometry']['Width of spread'] = self.input_dict['calc_data'][
                'Width of spread']
        self.input_dict['calc_data']['Geometry']['Depths'] = self.input_dict['calc_data']['Depths']

        self.initialize_geometry()  # Initialize geometry before getting thalweg and modifying depths/wse

        # If we have a shape that grows, determine the max needed wse
        self.determine_max_wse_for_growing_shapes()

        if 'WSE stations' in self.input_dict['calc_data']['Geometry']['calculator'].results:
            # Clear the results
            self.input_dict['calc_data']['Geometry']['calculator'].results['WSE stations'] = []
            self.input_dict['calc_data']['Geometry']['calculator'].results['Fill stations'] = []
            self.input_dict['calc_data']['Geometry']['calculator'].results['Fill elevations'] = []

        if self.can_compute_critical:
            # self.input_dict['calc_data']['Geometry']['calculator'].initialize_geometry()
            if self.standalone_calc:
                self.clear_results()

            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Compute cross-section'] = \
                False

            # Update the geometry's composite n calc
            n_input_dict = copy.copy(self.input_dict)
            n_input_dict['calc_data'] = copy.copy(self.input_dict['calc_data']['Composite n'])
            self.input_dict['calc_data']['Composite n']['calculator'].input_dict = n_input_dict
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n'] = \
                self.input_dict['calc_data']['Composite n']

            embed_depth = self.input_dict['calc_data']['Geometry']['Embedment']['calculator'].get_embedment_depth()
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n'][
                'calculator'].embedment_present = embed_depth > self.zero_tol

            if self.unknown == "Flows":
                self._compute_flow()
                # return True
            else:
                self._compute_unknown_from_flow()
                # return True

        return True

    def set_plotting_data(self, index):
        """Sets the plotting data for the cross-section.

        Args:
            index (int): The index of the cross-section to plot.
        """
        if 'Cross-section' not in self.plot_dict:
            return
        critical_elevations = self.results['Critical elevation']
        if index <= len(critical_elevations) and len(critical_elevations) > 0 or index == -1 and len(
                critical_elevations) > 0:
            self.plot_dict['Cross-section']['lines'][4]['Line intercepts'] = [critical_elevations[index]]

        # Flow polygon
        if 'Flow stations' in self.results and 'Flow elevations' in self.results:
            flow_stations = self.results['Flow stations']
            flow_elevations = self.results['Flow elevations']
            if index <= len(flow_stations) and len(flow_stations) > 0 or index == -1 and len(flow_stations) > 0:
                self.plot_dict['Cross-section']['series'][0]['x var'].set_val(flow_stations[index])
                self.plot_dict['Cross-section']['series'][0]['y var'].set_val(flow_elevations[index])

        # Embedment polygon
        self.plot_dict['Cross-section']['series'][1]['x var'].set_val(self.input_dict['calc_data']['Geometry'][
            'calculator'].embedment_stations)
        self.plot_dict['Cross-section']['series'][1]['y var'].set_val(self.input_dict['calc_data']['Geometry'][
            'calculator'].embedment_elevations)

        # Channel Geometry
        if 'Channel stations' in self.results and 'Channel elevations' in self.results:
            self.plot_dict['Cross-section']['series'][2]['x var'].set_val(self.results['Channel stations'])
            self.plot_dict['Cross-section']['series'][2]['y var'].set_val(self.results['Channel elevations'])
        else:
            self.plot_dict['Cross-section']['series'][2]['x var'].set_val(self.input_dict['calc_data']['Geometry'][
                'calculator'].channel_stations)
            self.plot_dict['Cross-section']['series'][2]['y var'].set_val(self.input_dict['calc_data']['Geometry'][
                'calculator'].channel_elevations)
        self.plot_dict['Cross-section']['series'][2]['Fill color'] = None

    def determine_max_wse_for_growing_shapes(self):
        """Determines the maximum water surface elevation (WSE) for growing shapes."""
        _, zero_slope = self.get_data('Horizontal tolerance', 1e-6)
        slope = self.input_dict['calc_data']['Slope']

        if self.input_dict['calc_data']['Calculate'] == 'Head' and slope > zero_slope:
            flows = self.input_dict['calc_data']['Flows']
            max_flow = max(flows) if flows else 0.0
            if self.input_dict['calc_data']['Geometry']['calculator'].low_elev is None or \
                self.input_dict['calc_data']['Geometry']['calculator'].high_elev is None or \
                self.input_dict['calc_data']['Geometry']['calculator'].low_elev == \
                    self.input_dict['calc_data']['Geometry']['calculator'].high_elev:
                computed_flow = 0.0
                cur_wse = 0.0
                unk_flow_sorted = SortedDict({cur_wse: 0.0})
                self.input_dict['calc_data']['Geometry']['calculator'].low_elev = cur_wse
                low_elev = cur_wse
                self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                    'Compute cross-section'] = False
                while computed_flow < max_flow:
                    cur_wse = 1.5 * cur_wse + 1.0  # Update our guess
                    computed_flow = self._compute_flow_from_elevation(cur_wse)
                    unk_flow_sorted[cur_wse] = computed_flow

                _, num_iterations = self.get_data('Max number of iterations', 500)
                _, null_data = self.get_data('Null data', -999.99)
                flow_interp = Interpolation([], [], null_data=null_data, zero_tol=self.zero_tol)

                # Not user-generated results; very basic within 10% or half a cfs
                curr_flow_err = max(max_flow * 0.1, 0.5)
                count = 0
                cur_flow = 0.0
                while abs(cur_flow - max_flow) > curr_flow_err and count < num_iterations:
                    flow_list = list(unk_flow_sorted.values())
                    unk_list = list(unk_flow_sorted.keys())
                    flow_interp.x = flow_list
                    flow_interp.y = unk_list
                    flow_interp.use_second_interpolation = True
                    cur_wse, _ = flow_interp.interpolate_y(max_flow)
                    cur_flow = self._compute_flow_from_elevation(cur_wse)
                    unk_flow_sorted[cur_wse] = cur_flow
                    count += 1

                # One more time to set the cross-section
                self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                    'Compute cross-section'] = True
                flow_list = list(unk_flow_sorted.values())
                unk_list = list(unk_flow_sorted.keys())
                flow_interp.x = flow_list
                flow_interp.y = unk_list
                # flow_interp.use_second_interpolation = True
                cur_wse, _ = flow_interp.interpolate_y(max_flow)
                cur_flow = self._compute_flow_from_elevation(cur_wse)

                # if the user provided us a depth of channel, use that; otherwise 125 percent of the depth for
                # visualizing
                channel_depth = cur_wse - low_elev
                self.input_dict['calc_data']['Geometry']['calculator'].high_elev = channel_depth * 1.25 + low_elev

    def _compute_unknown_from_flow(self):
        """Computes Water Surface normal and critical Elevation from the specified flows."""
        if self.can_compute_normal:
            self._compute_normal_depth_from_unknown()
            self.set_last_flow_plot_data()
        if self.input_dict['calc_data']['Geometry']['Shape'] != 'curb and gutter' or self.input_dict[
                'calc_data']['Geometry']["Use Manning's equation for flow computations"]:
            self._compute_critical_depth_from_flow()

    def _compute_flow(self):
        """Computes Water Surface normal and critical Elevation from the specified flows."""
        if self.can_compute_normal:
            self._compute_normal_flow_from_elevation_or_depth()
        # Use the same function as wse_from_flow, because we want the critical depth of the flow just determined
        if self.input_dict['calc_data']['Geometry']['Shape'] != 'curb and gutter' or self.input_dict[
                'calc_data']['Geometry']["Use Manning's equation for flow computations"]:
            self._compute_critical_depth_from_flow()

    def _compute_normal_depth_from_unknown(self):
        """Computes Water Surface Elevations (wse-s) and depths from specified flows using normal depth function."""
        self.normal_wses_stations = [[]]
        self.flows = self.input_dict['calc_data']['Flows']
        _, self.normal_wses_stations = self._compute_wse_from_flow(
            self._compute_flow_from_elevation, self.normal_wses_stations, self.unknown)
        # Assign the results  -- Assign per flow - it is already too late
        # self.assign_results(unk_list)
        # self.input_dict['calc_data']['Composite n']['calculator'].finalize_last_result()

    def _compute_normal_flow_from_elevation_or_depth(self):
        """Computes normal flow from elevation or depth."""
        # Compute the normal flow at that depth
        self.flows = []
        wse_list = self.input_dict['calc_data']['WSE']
        if self.input_dict['calc_data']['Head'] == 'Width of spread':
            wse_list = self.input_dict['calc_data']['Width of spread']
        for wse in wse_list:
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                'Compute cross-section'] = True
            flow = self._compute_flow_from_elevation(wse)
            self.flows.append(flow)
            self.results['Flows'].append(flow)
            self.results['WSE stations'].append(self.normal_wses_stations)
            self.normal_depth = wse - self.low_elev
            if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter':
                self.normal_depth = self.curb_depth
                self.compute_hydraulic_parameters(depth=self.normal_depth, wse=self.wse)
            else:
                self.normal_depth = wse - self.low_elev
                self.compute_hydraulic_parameters(depth=self.normal_depth, wse=wse)
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                'Compute cross-section'] = False

            self.set_last_flow_plot_data()

    def set_last_flow_plot_data(self):
        """Sets the plotting data for the last flow."""
        self.input_dict['calc_data']['Composite n']['calculator'].finalize_last_result()

        if 'Flow stations' not in self.results:
            self.results['Flow stations'] = []
        self.results['Flow stations'].append(self.input_dict['calc_data']['Geometry'][
            'calculator'].flow_stations[-1])
        if 'Flow elevations' not in self.results:
            self.results['Flow elevations'] = []
        self.results['Flow elevations'].append(self.input_dict['calc_data']['Geometry'][
            'calculator'].flow_elevations[-1])
        if 'Channel stations' not in self.results:
            self.results['Channel stations'] = []
            self.results['Channel elevations'] = []
        channel_stations = self.input_dict['calc_data']['Geometry']['calculator'].channel_stations
        channel_elevations = self.input_dict['calc_data']['Geometry']['calculator'].channel_elevations
        if len(self.results['Channel elevations']) == 0 or max(self.results['Channel elevations']) < max(
                channel_elevations):
            self.results['Channel stations'] = channel_stations
            self.results['Channel elevations'] = channel_elevations

    def _compute_critical_depth_from_flow(self):
        """Computes Water Surface Elevations (wse-s) and depths from specified flows using critical depth function."""
        station_list_to_drop = [[]]
        input_elevations_backup = copy.copy(self.input_dict['calc_data']['WSE'])
        # This variable is needed when horiz/adverse
        self.flows = self.input_dict['calc_data']['Flows']
        if len(self.flows) <= 0:
            self.flows = self.results['Flows']
        wse_list, station_list_to_drop = self._compute_wse_from_flow(
            self._compute_critical_flow_for_elevation, station_list_to_drop, 'Head', computing_normal=False)
        self.low_elev = self.input_dict['calc_data']['Geometry']['calculator'].low_elev
        for wse, velocity, slope, top_width, flow_area in zip(
                wse_list, self.critical_velocities, self.critical_slopes,
                self.critical_top_widths, self.critical_flow_areas):
            # Don't use thalweg - it doesn't account for embedment
            self.results['Critical depth'].append(wse - self.low_elev)
            self.results['Critical elevation'].append(wse)
            self.results['Critical velocity'].append(velocity)
            self.results['Critical slope'].append(slope)
            self.results['Critical top width'].append(top_width)
            self.results['Critical flow area'].append(flow_area)

        self.input_dict['calc_data']['WSE'] = input_elevations_backup

        return wse_list

    def get_low_and_high_value(self, unknown=None):
        """Returns the low and high value for a given unknown.

        Args:
            unknown (string): unknown variable that we are calculating

        Returns:
            low_val (float): lowest value for the unknown variable
            high_val (float): highest value for the unknown variable
        """
        low_val = 0.0
        high_val = 0.0
        if not unknown:
            unknown = self.input_dict['calc_data']['Calculate']

        result, num_divisions = self.get_data('Number of divisions for interpolation curve')
        if not result or num_divisions < 2:
            num_divisions = 10
        unreasonably_large_number = 1e6

        if unknown in ['Depth', 'Head',]:
            low_val = self.input_dict['calc_data']['Geometry']['calculator'].low_elev
            high_val = self.input_dict['calc_data']['Geometry']['calculator'].high_elev
            if self.input_dict['calc_data']['Head'] == 'Width of spread':
                low_val = self.zero_tol
                high_val = self.input_dict['calc_data']['Geometry']['Road width'] * 4.0
                if high_val < self.zero_tol:
                    high_val = 50.0
            if low_val is None or high_val is None or low_val == high_val:
                low_val, high_val = self.input_dict['calc_data']['Geometry']['calculator'].\
                    determine_low_and_high_of_elevations()
            max_depth = (high_val - low_val)
            if max_depth <= 0.0:
                thalweg = self.input_dict['calc_data']['Geometry']['Thalweg invert elevation']
                if thalweg is not None and thalweg > low_val:
                    low_val = thalweg
                high_val = thalweg + 20.0  # Just give it a value if one was not found
            list_range = [low_val + (high_val - low_val) * (i / num_divisions) for i in range(num_divisions + 1)]
            return low_val, high_val, list_range
        elif unknown in ['Flows']:
            num_dense = num_divisions - 2
            dense_start = 1.0
            dense_end = 5000.0
            power = 4.0
            high_val = unreasonably_large_number
            dense_range = [
                dense_start + (dense_end - dense_start) * ((i / (num_dense - 1)) ** power)
                for i in range(num_dense)
            ]
            list_range = [low_val] + dense_range + [high_val]
            return low_val, high_val, list_range
        else:
            if unknown in ['Slope', 'Cross-slope pavement', 'Cross-slope gutter']:
                _, low_val = self.get_data('Horizontal tolerance', None)
                if low_val is None:
                    low_val = self.zero_tol
                high_val = 10.0
                # List comprehension that applies a bias in our divisions to give us more values at the smaller end
                # (Where I expect more channels to exist and where the calculations are more sensitive)
                list_range = [low_val * (high_val / low_val) ** (i / (num_divisions - 1)) for i in range(num_divisions)]

            elif unknown in ['Composite n']:
                low_val = self.zero_tol * 1.5  # Needs to be above zero_tol
                high_val = 1.0
                # Make an an intelligent list that will cover the range (low to high), but put most values in
                # reasonable n ranges.  With 10 divisions the following values are generated:
                # [1e-05, 0.01, 0.018000000000000002, 0.026000000000000002, 0.034, 0.042, 0.05, 0.1, 0.2, 1.0]
                fixed_vals = [0.1, 0.2, high_val]
                remaining = num_divisions - len(fixed_vals) - 1
                list_range = [low_val]
                list_range.extend([0.01 + (0.05 - 0.01) * (i / (remaining - 1)) for i in range(remaining)
                                   ] if remaining > 0 else [])
                list_range.extend(fixed_vals)
            elif unknown in ['Road width',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] + [5 + (200 - 5) * (i / (num_divisions - 1)) for i in range(num_divisions)] + [
                    high_val]
            elif unknown in ['Side slope 1', 'Side slope 2', 'Side slope',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [10 ** (math.log10(0.01) + (math.log10(100) - math.log10(0.01))
                                     * (i / (num_divisions - 1))) for i in range(num_divisions)]
            elif unknown in ['Curb height', 'Gutter width', 'b',]:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] + [0.01 + (10 - 0.01) * (i / (num_divisions - 1)) for i in range(num_divisions)
                                          ] + [high_val]
            elif unknown in ['Diameter', 'Bottom width', 'Span', 'Rise', 'Top width', 'Top radius', 'Bottom radius',
                             'Width of spread']:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] + [0.5 + (100 - 0.5) * (i / (num_divisions - 1)) for i in range(num_divisions)
                                          ] + [high_val]
            else:
                low_val = self.zero_tol
                high_val = unreasonably_large_number
                list_range = [low_val] + [0.5 + (100 - 0.5) * (i / (num_divisions - 1)) for i in range(num_divisions)
                                          ] + [high_val]

        return low_val, high_val, list_range

    def _set_current_value(self, value: any, unknown: str = None):
        """Returns the low and high value for a given unknown.

        Args:
            value (float): sets the current value of the unknown variable
            unknown (string): unknown variable that we are calculating

        Returns:
            True, if successful; otherwise, False
        """
        found = False
        if not unknown:
            unknown = self.input_dict['calc_data']['Calculate']

        if unknown == 'Head':
            self.cur_wse = value
            return True
        else:
            if unknown in self.input_dict['calc_data']:
                self.input_dict['calc_data'][unknown] = value
                found = True
            if unknown in self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'] and \
                    isinstance(self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][unknown],
                               (int, float)):
                self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][unknown] = value
                found = True
            if unknown in self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n']:
                self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n'][
                    unknown] = value
                found = True
            if unknown in self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n'][
                    'calculator'].input_dict['calc_data']:
                self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Composite n'][
                    'calculator'].input_dict['calc_data'][unknown] = value
                found = True
        return found

    def _compute_wse_from_flow(self, compute_function, wse_stations, unknown, computing_normal=True):
        """Computes the Water Surface Elevation (wse) for the specified flows given a computing functor.

        Args:
            compute_function: functor that computes a wse for a given flow (normal or critical; perhaps shape specific)
            wse_stations: The stations of where the WSE meets the channels with 'nan's between segments (used for
             plotting)
            unknown (string): unknown variable that we are calculating

        Returns:
            tuple (list, list): The WSE at each flow, the stations of where the WSE meets the channels with 'nan's
                between segments (used for plotting)
        """
        # initialize the cur_wse
        self.cur_wse = 0.0

        low_val, high_val, list_range = self.get_low_and_high_value(unknown)
        if unknown != 'Head':
            if self.input_dict['calc_data']['Head'] == 'Width of spread':
                self.cur_wse = self.input_dict['calc_data']['Width of spread'][0]
            else:
                self.cur_wse = self.input_dict['calc_data']['WSE'][0]
        else:
            self.cur_wse = low_val
            self.low_elev = low_val

        diff_val = high_val - low_val

        # Special case for circular pipes
        max_flow_in_pipe_ratio = 0.9382
        if unknown == 'Head' and self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                'Shape'] == 'circle':
            # Find the depth of max flow
            max_val = diff_val * max_flow_in_pipe_ratio + low_val
            # Include max flow depth in our list and nothing above, except full depth
            filtered_list = [x for x in list_range if x <= max_val] + [max_val, high_val]
            list_range = filtered_list

        # Add Zero (floor of computation)
        self.unk_flow_sorted = SortedDict({low_val: 0.0})

        for cur_val in list_range:
            self._set_current_value(cur_val, unknown)
            cur_flow = compute_function(self.cur_wse)
            self.unk_flow_sorted[cur_val] = cur_flow

        max_flow = max(self.unk_flow_sorted.values())

        # Clear critical data
        self.critical_velocities = []
        self.critical_slopes = []
        self.critical_top_widths = []
        self.critical_flow_areas = []

        # Backup original results during intermediate calculations
        original_channel_results = self.input_dict['calc_data']['Geometry']['calculator'].results
        original_results = self.results

        # tol = 0.001
        wse_stations_initialized = False
        _, num_iterations = self.get_data('Max number of iterations', 500)
        _, flow_err = self.get_data('Flow error', 0.001)
        _, flow_err_p = self.get_data('Flow % error', 0.005)
        _, null_data = self.get_data('Null data', -999.99)

        flow_interp = Interpolation([], [], null_data=null_data, zero_tol=self.zero_tol)
        if unknown in ['Flows', 'Depth', 'Head']:
            flow_interp.use_second_interpolation = True
        val_list = []
        wse_stations = []
        flows = copy.copy(self.flows)
        for flow in flows:
            curr_flow_err = min(flow * flow_err_p, flow_err)
            cur_val = -999.0
            cur_flow = 0.0
            count = 0
            if flow <= 0.0:
                cur_val = low_val
            elif flow > max_flow:
                # determine a good depth guess
                if max_flow > 0.0:
                    depth_guess = (flow / max_flow) * diff_val + low_val
                else:
                    # Give us somewhere to go...
                    max_flow = flow
                    depth_guess = 1.5 * self.high_elev
                self._set_current_value(depth_guess, unknown)
                flow_guess = compute_function(self.cur_wse)
                self.unk_flow_sorted[depth_guess] = flow_guess
            if flow > 0.0:
                while abs(cur_flow - flow) > curr_flow_err and count < num_iterations:
                    flow_list = list(self.unk_flow_sorted.values())
                    unk_list = list(self.unk_flow_sorted.keys())
                    # if unknown == 'Composite n':
                    #     unk_list.reverse()
                    # for unk_val in unk_list:
                    #     flow_list.append(unk_flow_list[unk_val])
                    flow_interp.x = flow_list
                    flow_interp.y = unk_list
                    cur_val, _ = flow_interp.interpolate_y(flow)
                    self._set_current_value(cur_val, unknown)
                    cur_flow = compute_function(self.cur_wse)
                    self.unk_flow_sorted[cur_val] = cur_flow
                    count += 1
                    if count == 5:
                        flow_interp.use_second_interpolation = False
                if count >= num_iterations:
                    warning = "Calculations were unable to converge on a solution when trying to find a flow of "
                    warning += str(flow) + " cfs. "
                    warning += str(count) + " attempts were made."
                    self.warnings['Convergence'] = warning

            # One last computation to set the cross-section data
            # Restore original results (without intermediate)
            self.input_dict['calc_data']['Geometry']['calculator'].results = original_channel_results
            self.results = original_results
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Compute cross-section'] = \
                True
            self._set_current_value(cur_val, unknown)
            self.normal_wses_stations = []
            _ = compute_function(self.cur_wse, True)
            val_list.append(cur_val)
            cur_wse_stations = self.normal_wses_stations

            if computing_normal:
                self.assign_result(unk=cur_val, cur_wse=self.cur_wse)
                self.input_dict['calc_data']['Composite n']['calculator'].finalize_last_result()
            else:
                self.critical_velocities.append(self.critical_velocity)
                self.critical_slopes.append(self.critical_slope)
                self.critical_top_widths.append(self.critical_top_width)
                self.critical_flow_areas.append(self.critical_flow_area)

            if not wse_stations_initialized:
                wse_stations = cur_wse_stations
                wse_stations_initialized = True
            else:
                if len(cur_wse_stations):
                    wse_stations.append(cur_wse_stations[-1])

            if 'Channel stations' not in self.results:
                self.results['Channel stations'] = []
                self.results['Channel elevations'] = []
            channel_stations = self.input_dict['calc_data']['Geometry']['calculator'].channel_stations
            channel_elevations = self.input_dict['calc_data']['Geometry']['calculator'].channel_elevations
            if len(self.results['Channel elevations']) == 0 or max(self.results['Channel elevations']) < max(
                    channel_elevations):
                self.results['Channel stations'] = channel_stations
                self.results['Channel elevations'] = channel_elevations

        return val_list, wse_stations

    def _compute_flow_from_elevation(self, head: float, final_calc: bool = False):
        """Compute the normal flow for one specified elevation.

        Args:
            head (float): given head of the water (wse or width of spread)

        Returns:
            float: See description
        """
        wse = head
        if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter':
            # Determine Depth / Width of Spread - Do this whether HEC-22 or Manning's n
            if self.input_dict['calc_data']['Head'] == 'Width of spread':
                # WSE is a proxy for width of spread
                self.width_of_spread = head
                wse, self.curb_depth = self.calculate_wse_from_width_of_spread(head)
                self.wse = wse
            else:
                self.width_of_spread, self.curb_depth = self.calculate_width_of_spread_from_wse(head)
                self.wse = wse
        if final_calc:
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['WSE'] = [wse]
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Head'] = 'Elevation'
            self.input_dict['calc_data']['Geometry']['calculator']._compute_data_after_initializations()
        else:
            self.input_dict['calc_data']['Geometry']['calculator'].compute_channel_geometry(wse)

        base_elevation = self.input_dict['calc_data']['Geometry']['calculator'].base_elevation
        min_embedment_depth = self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth

        # Gather Manning's equation data
        a = self.input_dict['calc_data']['Geometry']['calculator'].results['Flow area'][-1]
        w = self.input_dict['calc_data']['Geometry']['calculator'].results['Wetted perimeter'][-1]
        r = a / w if w > 0 else 0
        _, k = self.get_data('Manning constant')
        s = self.input_dict['calc_data']['Slope']
        t = self.input_dict['calc_data']['Geometry']['calculator'].results['Top width'][-1]
        ya = a / t if t > 0 else 0

        # Update Composite n
        self.input_dict['calc_data']['Composite n'] = self.input_dict['calc_data']['Geometry'][
            'calculator'].input_dict['calc_data']['Composite n']
        n_var = self.input_dict['calc_data']['Composite n']['calculator']
        n, _ = self.update_n_data_and_compute_n(base_elevation, wse, min_embedment_depth, n_var, r, ya)
        if n_var.gradation_calculation:
            n_value_has_not_converged = True
            n_prev = n
            count = 0
            _, num_iter = self.get_data('Max number of iterations')
            _, n_error_tol = self.get_data('n error')
            while n_value_has_not_converged:
                self.update_mannings_n_data()
                self.input_dict['calc_data']['Geometry']['calculator'].compute_channel_geometry(wse)
                n, _ = self.update_n_data_and_compute_n(base_elevation, wse, min_embedment_depth, n_var, r, ya)
                n_diff = abs(n_prev - n)
                n_prev = n
                count += 1
                if n_diff < n_error_tol or count > num_iter:
                    n_value_has_not_converged = False

        if n is None or n <= self.zero_tol:
            return 0.0

        if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter' and not self.input_dict[
                'calc_data']['Geometry']["Use Manning's equation for flow computations"]:
            # Use HEC-22 equations
            return self.compute_curb_and_gutter_with_hec22(n)
        # Manning's equation: Q = (k/n) A R^(2/3) S^(1/2)
        self.flow = (k / n) * a * (r**(2.0 / 3.0)) * s**0.5

        self.normal_wses_stations = self.input_dict['calc_data']['Geometry']['calculator'].wse_stations

        return self.flow

    def compute_curb_and_gutter_with_hec22(self, n):
        """Compute the flow for curb and gutter using HEC-22 equations.

        Args:
            n (float): The Manning's n value
        """
        _, ku = self.get_data('Unit conversion constant (curbs)')
        curb_height = self.input_dict['calc_data']['Geometry']['Curb height']
        long_slope = self.input_dict['calc_data']['Slope']
        pavement_slope = self.input_dict['calc_data']['Geometry']['Cross-slope pavement']
        road_width = self.input_dict['calc_data']['Geometry']['Road width']
        gutter_slope = self.input_dict['calc_data']['Geometry']['Cross-slope gutter']
        gutter_width = self.input_dict['calc_data']['Geometry']['Gutter width']
        if not self.input_dict['calc_data']['Geometry']['Define gutter cross-slope']:
            gutter_slope = pavement_slope
        grate_width = self.input_dict['calc_data']['Geometry']['Grate width']
        grate_type = self.input_dict['calc_data']['Geometry']['Grate type']

        # Determine flow
        five_3 = 5.0 / 3.0
        eight_3 = 8.0 / 3.0
        if gutter_width <= self.zero_tol:
            self.flow = ku / n * (pavement_slope ** five_3) * long_slope ** 0.5 * self.width_of_spread ** eight_3
        elif self.width_of_spread <= gutter_width:
            self.flow = ku / n * (gutter_slope ** five_3) * long_slope ** 0.5 * self.width_of_spread ** eight_3
        else:
            denom = 1 + gutter_slope / pavement_slope / (
                (1 + gutter_slope / pavement_slope / (self.width_of_spread / gutter_width - 1)) ** eight_3 - 1)
            self.flow = (ku / n) * pavement_slope ** five_3 * long_slope ** 0.5 \
                * (self.width_of_spread - gutter_width) ** eight_3 / (1 - 1 / denom)

        # Determine flow area
        self.determine_flow_area(pavement_slope, self.width_of_spread, self.gutter_depression, gutter_width)
        self.determine_flow_polygon(self.curb_depth, self.width_of_spread, self.gutter_depression, gutter_width,
                                    pavement_slope)

        # Add the triangle, then the rectangle on top of the triangle.
        rect_height = (pavement_slope * self.width_of_spread) - (pavement_slope * gutter_width)
        if rect_height < 0.0:
            rect_height = 0.0

        self.gutter_flow_area = (0.5 * pavement_slope * (gutter_width ** 2.0)) + (
            0.5 * (self.gutter_depression) * gutter_width) + (rect_height * gutter_width)

        if gutter_width > self.zero_tol and grate_width > gutter_width:
            self.warnings['Grate width'] = "The grate width is larger than the gutter width."

        if grate_type in ['curb opening', 'slotted drain', ]:
            grate_width = gutter_width

        grate_change = grate_width * gutter_slope
        average_depth = (self.curb_depth + (self.curb_depth - grate_change)) / 2.0
        self.grate_flow_area = average_depth * grate_width
        # If the grate has more elevation change than the curb depth, we have 100% of the flow!
        if grate_change > self.curb_depth:
            self.grate_flow_area = self.gutter_flow_area

        self.gutter_flow_ratio = self.compute_gutter_flow_ratio(width_of_spread=self.width_of_spread,
                                                                gutter_slope=gutter_slope)

        self.grate_flow_ratio = self.compute_gutter_flow_ratio(
            width_of_spread=self.width_of_spread, gutter_slope=gutter_slope, grate_width=grate_width,
            inlet_type=grate_type)

        if curb_height > self.zero_tol and self.curb_depth > curb_height:
            self.warnings['Curb height'] = "The depth of flow exceeds the curb height."
        if road_width > self.zero_tol and self.width_of_spread > road_width:
            self.warnings['Road width'] = "The width of spread exceeds the roadway width."

        return self.flow

    def calculate_wse_from_width_of_spread(self, width_of_spread):
        """Calculate the curb depth from the width of spread.

        Args:
            width_of_spread (float): The width of spread

        Returns:
            float: The water surface elevation
        """
        pavement_slope = self.input_dict['calc_data']['Geometry']['Cross-slope pavement']
        gutter_slope = self.input_dict['calc_data']['Geometry']['Cross-slope gutter']
        gutter_width = self.input_dict['calc_data']['Geometry']['Gutter width']
        thalweg = self.input_dict['calc_data']['Geometry']['Thalweg invert elevation']
        if not self.input_dict['calc_data']['Geometry']['Define gutter cross-slope']:
            gutter_slope = pavement_slope

        self.gutter_depression = (gutter_slope - pavement_slope) * gutter_width

        curb_depth = (pavement_slope * width_of_spread) + self.gutter_depression
        if width_of_spread < gutter_width:
            curb_depth = gutter_slope * width_of_spread
        wse = curb_depth + thalweg
        return wse, curb_depth

    def calculate_width_of_spread_from_wse(self, wse):
        """Calculate the width of spread from the water surface elevation (WSE).

        Args:
            wse (float): The water surface elevation

        Returns:
            float: The width of spread
        """
        pavement_slope = self.input_dict['calc_data']['Geometry']['Cross-slope pavement']
        gutter_slope = self.input_dict['calc_data']['Geometry']['Cross-slope gutter']
        gutter_width = self.input_dict['calc_data']['Geometry']['Gutter width']
        thalweg = self.input_dict['calc_data']['Geometry']['Thalweg invert elevation']
        if not self.input_dict['calc_data']['Geometry']['Define gutter cross-slope']:
            gutter_slope = pavement_slope

        self.gutter_depression = (gutter_slope - pavement_slope) * gutter_width

        curb_depth = wse - thalweg
        width_of_spread = (curb_depth - self.gutter_depression) / pavement_slope
        if curb_depth < self.gutter_depression:
            width_of_spread = gutter_slope * curb_depth
        return width_of_spread, curb_depth

    def determine_flow_area(self, pavement_slope, width_of_spread, gutter_depression, gutter_width):
        """Determine the flow area.

        Args:
            pavement_slope (float): The pavement slope
            width_of_spread (float): The width of spread
            gutter_depression (float): The gutter depression
            gutter_width (float): The gutter width
        """
        flow_area = 0.5 * pavement_slope * width_of_spread ** 2.0 + 0.5 * gutter_depression * gutter_width
        self.input_dict['calc_data']['Geometry']['calculator'].results['Flow area'][-1] = flow_area

    def determine_flow_polygon(self, wse, width_of_spread, gutter_depression, gutter_width, pavement_slope):
        """Determine the flow polygon.

        Args:
            wse (float): The water surface elevation
            width_of_spread (float): The width of spread
            gutter_depression (float): The gutter depression
            gutter_width (float): The gutter width
            pavement_slope (float): The pavement slope
        """
        # Flow polygon
        flow_x = [0.0, width_of_spread]
        flow_y = [wse, wse]
        if width_of_spread > gutter_width:
            flow_x.append(gutter_width)
            flow_y.append(gutter_depression + pavement_slope * gutter_width)
        flow_x.append(0.0)
        flow_y.append(0.0)
        flow_x.append(0.0)
        flow_y.append(wse)
        self.input_dict['calc_data']['Geometry']['calculator'].flow_stations[-1] = flow_x
        self.input_dict['calc_data']['Geometry']['calculator'].flow_elevations[-1] = flow_y

        self.normal_wses_stations = [0.0, width_of_spread]
        self.normal_wses_elevations = [wse, wse]
        self.input_dict['calc_data']['Geometry']['calculator'].wse_stations = self.normal_wses_stations

    def compute_gutter_flow_ratio(self, width_of_spread: float, gutter_slope: float, grate_width: float = None,
                                  inlet_type: str = None) -> float:
        """Compute the gutter flow ratio.

        Args:
            width_of_spread (float): The width of spread
            gutter_slope (float): The gutter slope
            grate_width (float, optional): The grate width
            inlet_type (str, optional): The inlet type

        Returns:
            float: The gutter flow ratio
        """
        flow_ratio = 0.0

        pavement_slope = self.input_dict['calc_data']['Geometry']['Cross-slope pavement']
        gutter_width = self.input_dict['calc_data']['Geometry']['Gutter width']

        width = gutter_width

        # This function can be used to determine the flow ratio over a grate as well.
        # Just pass the grate width and inlet type, and we'll compute the grate flow ratio
        if grate_width is not None and grate_width > self.zero_tol and inlet_type is not None and inlet_type != '':
            if inlet_type not in ['curb opening', 'slotted drain', ]:
                width = grate_width

        if width >= width_of_spread:
            flow_ratio = 1.0
        elif pavement_slope == gutter_slope:
            flow_ratio = 0.0
            if width_of_spread > self.zero_tol:
                flow_ratio = 1 - (1 - (width / width_of_spread)) ** 2.67
        else:
            # Intermediate gutter flow ratio Calculatio
            sw_over_sx = gutter_slope / pavement_slope
            # Eo = 1/{1+(Sw/Sx)/([1+(Sw/Sx)/((T/W)-1)]^2.67-1)}
            flow_ratio = 0.0
            flow_ratio = 1.0 / (
                1.0 + (
                    sw_over_sx / (
                        (1.0 + (sw_over_sx / ((width_of_spread / width) - 1.0))) ** 2.67 - 1.0
                    )
                )
            )

        return flow_ratio

    def update_n_data_and_compute_n(self, base_elevation, wse,
                                    min_embedment_depth, n_var, r, ya):
        """Update the Manning's n data for the composite n calculations.

        Args:
            base_elevation (float): The Datum elevation of the channel
            wse (float): The specified water surface elevation
            min_embedment_depth (float): The minimum embedment depth
            n_var (ManningsN): The Manning's n variable
            r (float): The hydraulic radius
            ya (float): average flow depth
        """
        if n_var.input_dict['calc_data']['n entry'] == 'compute n values for gradation size' or \
                n_var.input_dict['calc_data']['Embedment n entry'] == 'specify gradation equation':
            self.hyd_radius = r
            n_var.hyd_radius = r
            self.average_flow_depth = ya
            n_var.average_flow_depth = self.average_flow_depth

        n_var.d50 = self.input_dict['calc_data']['Geometry']['Embedment']['Gradation layer']['calculator'].get_d50()
        n_var.d84 = self.input_dict['calc_data']['Geometry']['Embedment']['Gradation layer']['calculator'].get_d84()
        n_var.zero_tol = self.zero_tol
        n, result = n_var.get_intermediate_n_and_applicable()
        return n, result

    def _compute_critical_flow_for_elevation(self, head: float, final_calc: bool = False):
        """Compute the critical flow for one specified elevation.

        Args:
            head (float): given head of the water (wse or width of spread)

        Returns:
            float: See description
        """
        wse = head
        if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter':
            # Determine Depth / Width of Spread - Do this whether HEC-22 or Manning's n
            if self.input_dict['calc_data']['Head'] == 'Width of spread':
                # WSE is a proxy for width of spread
                self.width_of_spread = head
                wse, self.curb_depth = self.calculate_wse_from_width_of_spread(head)
                self.wse = wse
            else:
                self.width_of_spread, self.curb_depth = self.calculate_width_of_spread_from_wse(head)
                self.wse = wse

        if final_calc:
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['WSE'] = [wse]
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Head'] = 'Elevation'
            result = self.input_dict['calc_data']['Geometry']['calculator']._compute_data_after_initializations()
            if result is False:
                return -1
        else:
            self.input_dict['calc_data']['Geometry']['calculator'].compute_channel_geometry(wse)

        # Critical Depth (Q) = sqrt(A**3*g/(TopWidth))
        _, g = self.get_data('Gravity')
        a = self.input_dict['calc_data']['Geometry']['calculator'].results['Flow area'][-1]
        t = self.input_dict['calc_data']['Geometry']['calculator'].results['Top width'][-1]
        ya = a / t if t > 0 else 0
        # Consider protecting the top width from going to zero as can happen
        if t <= 0.0:
            self.critical_flow = 0.0
        else:
            self.critical_flow = (a**3.0 * g / t)**0.5

        # Perform Additional computations for velocity, slope, and top width
        r = self.input_dict['calc_data']['Geometry']['calculator'].results['Hydraulic radius'][-1]
        critical_velocity = 0.0
        if a > 0.0:
            critical_velocity = self.critical_flow / a

        # Data exchange between Geometry and Manning's n
        base_elevation = self.input_dict['calc_data']['Geometry']['calculator'].base_elevation
        min_embedment_depth = self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth
        self.input_dict['calc_data']['Composite n'] = self.input_dict['calc_data']['Geometry'][
            'calculator'].input_dict['calc_data']['Composite n']
        n_var = self.input_dict['calc_data']['Composite n']['calculator']
        n, result = self.update_n_data_and_compute_n(base_elevation, wse,
                                                     min_embedment_depth,
                                                     n_var, r, ya)

        if n_var.gradation_calculation:
            n_value_has_not_converged = True
            n_prev = n
            count = 0
            _, num_iter = self.get_data('Max number of iterations')
            _, n_error_tol = self.get_data('n error')
            while n_value_has_not_converged:
                self.update_mannings_n_data()
                self.input_dict['calc_data']['Geometry']['calculator'].compute_channel_geometry(wse)
                n, _ = self.update_n_data_and_compute_n(
                    base_elevation, wse, min_embedment_depth, n_var, r, ya)
                n_diff = abs(n_prev - n)
                n_prev = n
                count += 1
                if n_diff < n_error_tol or count > num_iter:
                    n_value_has_not_converged = False

        _, k = self.get_data('Manning constant')
        critical_slope = 0.0
        if r > 0.0 and n is not None:
            critical_slope = ((critical_velocity * n) / (k * r**(2.0 / 3.0)))**2.0
        self.critical_slope = critical_slope
        self.critical_velocity = critical_velocity
        self.critical_top_width = t
        self.critical_flow_area = a

        return self.critical_flow

    def compute_sequent_depth_and_froude(self, depth, flow):
        """Computes the sequent depth of a geometry given a depth AND flow.

        Args:
             depth (float): specified depth
             flow (float): specified flow
        """
        return self.input_dict['calc_data']['Geometry']['calculator'].compute_sequent_depth_and_froude(depth, flow)

    def compute_hydraulic_parameters(self, flow: float = None, depth: float = None, wse: float = None):
        """Computes the hydraulic parameters for a specific flow in self.flow."""
        if self.can_compute:
            if flow is not None:
                if 'Flows' not in self.results:
                    self.results['Flows'] = []
                self.results['Flows'].append(flow)
            if depth is not None:
                if 'Depths' not in self.results:
                    self.results['Depths'] = []
                self.results['Depths'].append(depth)
            if wse is not None:
                if 'WSE' not in self.results:
                    self.results['WSE'] = []
                self.results['WSE'].append(wse)

            depth = self.normal_depth
            flow = self.flow

            flow_area = self.input_dict['calc_data']['Geometry']['calculator'].results['Flow area'][-1]
            self.results['Flow area'].append(flow_area)

            if self.input_dict['calc_data']['Geometry']['Shape'] == 'curb and gutter' and not self.input_dict[
                    'calc_data']['Geometry']["Use Manning's equation for flow computations"]:
                if 'Width of spread' not in self.results:
                    self.results['Width of spread'] = []
                self.results['Width of spread'].append(self.width_of_spread)
                if 'Gutter flow area' not in self.results:
                    self.results['Gutter flow area'] = []
                self.results['Gutter flow area'].append(self.gutter_flow_area)
                if 'Grate flow area' not in self.results:
                    self.results['Grate flow area'] = []
                self.results['Grate flow area'].append(self.grate_flow_area)
                if 'Ratio of gutter flow to total flow' not in self.results:
                    self.results['Ratio of gutter flow to total flow'] = []
                self.results['Ratio of gutter flow to total flow'].append(self.gutter_flow_ratio)
                if 'Ratio of grate flow to total flow' not in self.results:
                    self.results['Ratio of grate flow to total flow'] = []
                self.results['Ratio of grate flow to total flow'].append(self.grate_flow_ratio)
                if 'Gutter depression' not in self.results:
                    self.results['Gutter depression'] = []
                self.results['Gutter depression'].append(self.gutter_depression)
                return
            top_width = self.input_dict['calc_data']['Geometry']['calculator'].results['Top width'][-1]
            wetted_perim = self.input_dict['calc_data']['Geometry']['calculator'].results['Wetted perimeter'][-1]
            froude = 0.0
            velocity = 0.0
            _, g = self.get_data('Gravity')
            _, null_data = self.get_data('Null data')
            if flow_area > 0.0:
                velocity = flow / flow_area
                if top_width > 0.0:
                    froude = velocity / pow((g * flow_area) / top_width, 0.5)
            long_slope = self.input_dict['calc_data']['Slope']
            hydraulic_radius = self.input_dict['calc_data']['Geometry']['calculator'].results['Hydraulic radius'][-1]
            if top_width > 0.0:
                hyd_depth = flow_area / top_width
            else:
                hyd_depth = 0.0
            _, gamma = self.get_data('Unit weight of water')

            self.results['Wetted perimeter'].append(wetted_perim)
            self.results['Hydraulic radius'].append(hydraulic_radius)
            self.results['Hydraulic depth'].append(hyd_depth)
            self.results['Average velocity'].append(velocity)
            self.results['Top width'].append(top_width)
            self.results['Froude number'].append(froude)
            self.results['Max shear stress'].append(gamma * depth * long_slope)
            self.results['Average shear stress'].append(gamma * hydraulic_radius * long_slope)

            manning_n = self.input_dict['calc_data']['Composite n']['calculator']
            manning_n.finalize_last_result()
            self.warnings.update(manning_n.warnings)
            self.results['Composite n computation method'].append(
                manning_n.results['Composite n computation method'][-1])
            n_values = manning_n.results['Composite n value']
            if len(n_values):
                self.results['Composite n value'].append(n_values[-1])

            if hasattr(self, 'keep_all_mannings_n_calcs'):
                self.results['final_horton_manning_value'] = manning_n.final_horton_manning_value
                self.results['final_lotter_manning_value'] = manning_n.final_lotter_manning_value
                self.results['final_pavlovskii_manning_value'] = manning_n.final_pavlovskii_manning_value

            _, kinematic_viscosity = self.get_data('Kinematic viscosity of water')
            reynolds_number = 0.0
            if kinematic_viscosity > 0.0:
                reynolds_number = velocity * hydraulic_radius / kinematic_viscosity
            self.results['Reynolds number'].append(reynolds_number)
            if reynolds_number < 2000.0:
                stability_str = 'laminar flow'
            elif reynolds_number < 4000.0:
                stability_str = 'unstable flow'
            else:  # reynolds_number > 4000.0:
                stability_str = 'turbulent flow'
            self.results['Flow stability'].append(stability_str)

            # Sequent needs to be at the end, so we don't mix up previously computed data
            if froude > 1.0:
                sequent_depth = self.compute_sequent_depth_and_froude(depth, flow)
            else:
                sequent_depth = (null_data, froude)
            self.results['Sequent depths'].append(sequent_depth[0])
