"""CalcData for performing Manning's N operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys
from uuid import uuid4

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import ChannelGeometryData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_data import CompositeManningNData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_calc import ManningNCalc


class ManningNData(CalcData):
    """A class that defines a channel and performs Manning's n computations."""

    def __init__(self, standalone_calc: bool = True, allowed_shapes: str = 'all', default_shape: str = None,
                 hide_depth: bool = False,
                 app_data: AppData = None, model_name: str = None, project_uuid: uuid4 = None):
        """Initializes the Manning's n calculator.

        Args:
            standalone_calc (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            allowed_shapes (string): sets which channel shapes the user can select: 'open', 'closed', 'both'
            default_shape (string): Which shape should be the default shape
            hide_depth (bool): Is this class being used for tailwater calculations?
            app_data (AppData): The application data object.
            model_name (string): The name of the model.
            project_uuid (string): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Open-Channel Flow Calculator"
        self.type = "ManningNCalc"
        self.class_name = 'Open-Channel Flow Calculator'

        self.calculator = ManningNCalc()

        self.calc_support_dual_input = False

        self.max_value = sys.float_info.max

        self.hide_depth = hide_depth

        # Variable to track whether we should set the head to Width of spread for 'curb and gutter'
        self.prev_curb_head_set = True

        # Input
        self.input['Calculate'] = Variable('Calculate', "list", 1, ['Head', 'Flows'])

        self.input['Head'] = Variable('specify head as a:', "list", 0, ['Depth', 'Elevation'], complexity=1)

        self.input['Geometry'] = Variable('Geometry', 'class', ChannelGeometryData(
            stand_alone_calc=False, allowed_shapes=allowed_shapes, default_shape=default_shape,
            hide_depth=hide_depth, app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['Composite n'] = Variable("Manning's n", 'class', CompositeManningNData(
            shape='circle', open_shape=True, embedment_present=False, gradation_present=False, total_embedment=0.0,
            stations=[], elevations=[], app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['Slope'] = Variable('Slope', 'float', 0.0, [], limits=[0.0, 10.0], precision=6,
                                       unit_type=['slope'], native_unit='ft/ft', us_units=self.us_slope,
                                       si_units=self.si_slope)
        # TODO make a discharge class, and have an option to specify flow as a project that then feeds all
        # calculators with that flow
        self.input['Flows'] = Variable('Flow(s)', 'UserArray', UserArray(
            precision=2, unit_type=['flow'], native_unit='cfs', us_units=self.us_flow, si_units=self.si_flow,
            select_name='specify flow(s) as:', name_append='flow'))

        self.input['Width of spread'] = Variable(
            'Width(s) of spread', 'UserArray', UserArray(
                precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
                si_units=self.si_mid_length, select_name='specify width of spread as:', name_append='width of spread'),
            note='Lateral distance water covers from the gutter onto the pavement')

        self.input['WSE'] = Variable(
            'Water surface elevation(s)', 'UserArray', UserArray(
                2, ['length'], 'ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
                select_name='specify elevation(s) as:', name_append='Elevation'), limits=(-self.max_value,
                                                                                          self.max_value))

        self.input['Depths'] = Variable('Depth(s)', 'UserArray', UserArray(
            2, ['length'], 'ft', us_units=self.us_mid_length, si_units=self.si_mid_length,
            select_name='specify depth(s) as:', name_append='Depth'))

        self.input['Selected profile'] = Variable(
            'Selected profile', 'int', 0, [],
            note='This selects which flow profile to plot and show in the specified results.')
        self.plot_index_set = False

        # Plot options
        self.input['Plot options'] = {}
        self.add_plot_options()

        # Intermediate
        self.compute_prep_functions.extend([self.prepare_cross_section])
        self.compute_finalize_functions.extend([self.assign_unknown_results, self.set_plotting_data])
        self.intermediate_to_copy.extend([
            'unknown', 'high_elev', 'low_elev', 'normal_wses_stations', 'flows', 'flow',
            'cur_wse', 'normal_wses', 'normal_depths', 'critical_slope', 'critical_slopes', 'critical_velocity',
            'critical_velocities', 'critical_top_width', 'critical_top_widths', 'critical_flow_area',
            'critical_flow_areas', 'hyd_radius', 'average_flow_depth', 'standalone_calc', 'can_compute_normal',
            'can_compute_critical', 'width_of_spread', 'curb_depth', 'wse', 'gutter_depression', 'gutter_flow_area',
            'gutter_flow_ratio', 'grate_flow_area', 'grate_flow_ratio',
        ])

        self.unknown = 'Flows'
        self.high_elev = 0.0
        self.low_elev = 0.0
        self.normal_wses_stations = [[]]
        self.flows = 0.0
        self.flow = 0.0
        self.cur_wse = 0.0
        self.normal_wse = 0.0
        self.normal_wses = []
        self.normal_depths = 0.0
        self.critical_slope = 0.0
        self.critical_slopes = []
        self.critical_velocity = 0.0
        self.critical_velocities = []
        self.critical_top_width = 0.0
        self.critical_top_widths = []
        self.critical_flow_area = 0.0
        self.critical_flow_areas = []
        # For gradation n calculations
        self.hyd_radius = 0.0
        self.average_flow_depth = 0.0

        self.standalone_calc = standalone_calc

        self.can_compute_normal = False
        self.can_compute_critical = False

        # For curb and gutter
        self.width_of_spread = 0.0
        self.curb_depth = 0.0
        self.wse = 0.0
        self.gutter_depression = 0.0
        self.gutter_flow_area = 0.0
        self.gutter_flow_ratio = 0.0
        # for grate computations
        self.grate_flow_area = 0.0
        self.grate_flow_ratio = 0.0

        # Results
        self.results = {}

        self.results['Flows'] = Variable(
            'Flow(s)', 'float_list', 0.0, [], unit_type=['flow'], native_unit='cfs', us_units=self.us_flow,
            si_units=self.si_flow)
        self.results['WSE'] = Variable(
            'Water surface elevation(s)', 'float_list', 0.0, [], precision=2,
            limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Depths'] = Variable(
            'Depth(s)', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        # Curb and gutter
        self.results['Width of spread'] = Variable(
            'Width(s) of spread', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length,
            note='Lateral distance water covers from the gutter onto the pavement')

        self.results['Gutter depression'] = Variable(
            'Gutter depression', 'float_list', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Gutter depression'].selected_us_unit = 'in'

        self.results['Ratio of gutter flow to total flow'] = Variable(
            'Ratio of gutter flow to total flow (E₀)', 'float_list', 0.0, [], precision=2, unit_type='coefficient',
            note='1.0 means all flow is in the gutter; 0.5 means half is in the gutter and half is in the travel lane')

        self.results['Ratio of grate flow to total flow'] = Variable(
            'Ratio of grate flow to total flow', 'float_list', 0.0, [], precision=2, complexity=1,
            unit_type='coefficient',
            note='1.0 means all water flows over the grate; 0.5 means half flows over the grate.')

        self.results['Gutter flow area'] = Variable('Area of flow in Gutter', 'float_list', 0.0, [], precision=2,
                                                    complexity=1, unit_type=['area'], native_unit='ft^2',
                                                    us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])

        self.results['Grate flow area'] = Variable('Area of flow in Grate', 'float_list', 0.0, [], precision=2,
                                                   complexity=1, unit_type=['area'], native_unit='ft^2',
                                                   us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])

        self.results['Sequent depths'] = Variable(
            'Sequent depth(s)', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['WSE stations'] = Variable(
            'Water surface station(s)', 'list_of_lists', 0.0, [], precision=2,
            limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Critical depth'] = Variable(
            'Critical depth(s)', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Critical elevation'] = Variable(
            'Critical elevation(s)', 'float_list', 0.0, [], precision=2,
            limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # self.critical_wses_stations = [[]]
        # Maybe another day, but this is beyond current computations

        self.results['Flow area'] = Variable('Area of flow', 'float_list', 0.0, [], precision=2, unit_type=['area'],
                                             native_unit='ft^2', us_units=[['yd^2', 'ft^2', 'in^2']],
                                             si_units=[['m^2', 'mm^2']])

        self.results['Wetted perimeter'] = Variable('Wetted perimeter', 'float_list', 0.0, [], precision=2,
                                                    unit_type=['length'], native_unit='ft',
                                                    us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic radius'] = Variable('Hydraulic radius', 'float_list', 0.0, [], precision=2,
                                                    unit_type=['length'], native_unit='ft',
                                                    us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic depth'] = Variable('Hydraulic depth', 'float_list', 0.0, [], precision=2,
                                                   unit_type=['length'], native_unit='ft',
                                                   us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Average velocity'] = Variable('Average velocity', 'float_list', 0.0, [], precision=2,
                                                    unit_type=['length'], native_unit='ft/s',
                                                    us_units=[['mph', 'yd/s', 'ft/s', 'in/s']],
                                                    si_units=[['km/hr', 'm/s', 'mm/s']])

        self.results['Top width'] = Variable('Top width', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                                             native_unit='ft', us_units=self.us_mid_length,
                                             si_units=self.si_mid_length)

        self.results['Froude number'] = Variable('Froude number', 'float_list', 0.0, [], precision=4, unit_type=[],
                                                 native_unit='ft', us_units=[['']], si_units=[[]])

        self.results['Critical velocity'] = Variable('Velocity at critical depth', 'float_list', 0.0, [],
                                                     precision=2, unit_type=['length'], native_unit='ft/s',
                                                     us_units=[['mph', 'yd/s', 'ft/s', 'in/s']],
                                                     si_units=[['km/hr', 'm/s', 'mm/s']])

        self.results['Critical slope'] = Variable('Slope that induces critical depth with this flow', 'float_list',
                                                  0.0, [], precision=6, unit_type=['slope'], native_unit='ft/ft',
                                                  us_units=[['ft/ft']], si_units=[['m/m']])

        self.results['Critical top width'] = Variable('Top width at critical depth', 'float_list', 0.0, [],
                                                      precision=2, unit_type=['length'], native_unit='ft',
                                                      us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Critical flow area'] = Variable('Area of flow at critical depth', 'float_list', 0.0, [],
                                                      precision=2, unit_type=['area'], native_unit='ft^2',
                                                      us_units=[['yd^2', 'ft^2', 'in^2']],
                                                      si_units=[['m^2', 'mm^2']])

        self.results['Max shear stress'] = Variable('Max shear stress', 'float_list', 0.0, [], precision=2,
                                                    unit_type=['shear'], native_unit='lb/ft^2',
                                                    us_units=[['lb/ft^2']], si_units=[['N/m^2']])

        self.results['Average shear stress'] = Variable('Average shear stress', 'float_list', 0.0, [], precision=2,
                                                        unit_type=['shear'], native_unit='lb/ft^2',
                                                        us_units=[['lb/ft^2']], si_units=[['N/m^2']])

        self.results['Composite n computation method'] = Variable('Composite n computation method', 'string_list', '')

        self.results['Composite n value'] = Variable('Composite n value', 'float_list', 0.0, [], precision=4,
                                                     unit_type='coefficient', native_unit='coefficient',
                                                     us_units='coefficient', si_units='coefficient')

        self.results['Reynolds number'] = Variable('Reynolds number', 'float_list', 0.0, [], precision=0,
                                                   unit_type='coefficient', native_unit='coefficient',
                                                   us_units='coefficient', si_units='coefficient', complexity=1)

        self.results['Flow stability'] = Variable('Flow stability', 'string_list', '', complexity=1)

        self.warnings = []

        # plot
        self.plots['Cross-section'] = {}
        self.plots['Cross-section']['Plot name'] = 'Cross-section'
        self.plots['Cross-section']['Legend'] = 'best'

    def add_plot_options(self):
        """Add plot options for a scenario.
        """
        self.station_var = Variable('Station', 'float_list', 0, [0.0], precision=2,
                                    limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                    native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                      limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                      native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input_dict = {}
        self.input_dict['Channel'] = {
            'Station': copy.copy(self.station_var),
            'Elevation': copy.copy(self.elevation_var)
        }

        self.min_y = sys.float_info.max

        # Load theme colors
        self.theme = self.get_theme()
        channel_bank_color = self.theme['Plot channel bank line color']  # Bank lines, not channel lines

        structure_color = self.theme['Plot structure color']
        # structure_fill_color = self.theme['Plot structure fill color']

        # embankment_color = self.theme['Plot embankment color']
        # embankment_fill_color = self.theme['Plot embankment fill color']

        soil_color = self.theme['Plot soil color']
        soil_fill_color = self.theme['Plot soil fill color']

        # wsp_color = self.theme['Plot WSP color']
        # wsp_fill_color = self.theme['Plot WSP fill color']

        normal_color = self.theme['Plot normal depth color']
        normal_fill_color = self.theme['Plot normal depth fill color']

        critical_color = self.theme['Plot critical depth color']
        # critical_fill_color = self.theme['Plot critical depth fill color']

        # hydraulic_jump_color = self.theme['Plot hydraulic jump color']
        # hydraulic_jump_color = self.theme['Plot hydraulic jump fill color']

        # Need plot items for thalweg, point properties (toes, abutment? user-defined?) Auto determine?
        # Need plot items for ineffective flow areas, n values?

        self.input['Plot options']['Cross-section'] = Variable(
            'Plot options', 'class', PlotOptions(
                'Cross-section', input_dict=self.input_dict, show_series=True, app_data=self.app_data,
                model_name=self.model_name, project_uuid=self.project_uuid), complexity=1)

        # Lines
        self.input['Plot options']['Cross-section'].get_val().set_plot_line_options(
            0, 'Channel bank',
            line_intercept=None, line_alignment='vertical', line_color=channel_bank_color, linetype='dotted',
            line_width=1.5, text_color=channel_bank_color, text_alignment='top')

        self.input['Plot options']['Cross-section'].get_val().set_plot_line_options(
            1, 'Critical depth',
            line_intercept=None, line_alignment='horizontal', line_color=critical_color, linetype='dotted',
            line_width=3.0, text_color=critical_color, text_alignment='top')

        # Series
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Normal depth',
            line_color=normal_color, linetype='dotted', line_width=3.0, fill_below_line=None,
            fill_color=normal_fill_color, pattern=None, density=None)
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=1, x_axis='Station', y_axis='Elevation', name='Embedment', line_color=soil_color,
            linetype=None, line_width=2, fill_below_line=None, fill_color=soil_fill_color, pattern='earth',
            density=None)
        self.input['Plot options']['Cross-section'].get_val().set_plot_series_options(
            related_index=0, index=2, x_axis='Station', y_axis='Elevation', name='Channel',
            line_color=structure_color, linetype=None, line_width=4, fill_below_line=None, fill_color=None,
            pattern=None, density=None)

    def set_plotting_data(self, index=None):
        """Sets up the plotting data for the cross section.

        Args:
            index (int): Not used.
        """
        if index is None:
            index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
        self.calculator.set_plotting_data(index)

    def prepare_cross_section(self):
        """Prepares the cross section for calculations."""
        geom = self.input['Geometry'].get_val()
        geom.prepare_cross_section()
        self.input['Composite n'].value.input["Manning's n by station"] = geom.input["Manning's n by station"]

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        if unknown is None:
            unknown = self.unknown
            self.input['Calculate'].set_val(self.unknown)
        calc_index = self.input['Calculate'].value

        # reset calculate variable (it gets variables added with the change event)
        if not self.hide_depth:
            if self.input['Geometry'].get_val().input['Shape'].get_val() == 'curb and gutter':
                self.input['Head'].value_options = ['Depth', 'Elevation', 'Width of spread']
                if not self.prev_curb_head_set:
                    self.input['Head'].set_val('Width of spread')
                    self.prev_curb_head_set = True
            else:
                self.prev_curb_head_set = False
                if self.input['Head'].value == 'Width of spread':
                    self.input['Head'].set_val('Depth')
            calc_uuid = self.input['Calculate'].uuid  # Do not change the uuid
            self.input['Calculate'] = Variable('Calculate', "list", 1, ['Head', 'Flows'])
            self.input['Calculate'].uuid = calc_uuid
            input_vars['Calculate'] = self.input['Calculate']

            input_vars['Head'] = self.input['Head']

        input_vars['Geometry'] = self.input['Geometry']

        # if unknown != 'Composite n':
        #     input_vars['Composite n'] = self.input['Composite n']
        if unknown != 'Composite n':
            self.input['Composite n'].get_val().update_lists()
            input_vars['Composite n'] = self.input['Composite n']

        if unknown != 'Slope':
            input_vars['Slope'] = self.input['Slope']

        if not self.hide_depth:
            if unknown != 'Head':
                shape = input_vars['Geometry'].value.input['Shape'].get_val()
                rise = None
                if shape in input_vars['Geometry'].value.closed_shapes and shape != 'cross-section':
                    rise = input_vars['Geometry'].value.input['Rise'].get_val()
                    if shape == 'circle':
                        rise = input_vars['Geometry'].value.input['Diameter'].get_val()
                if self.input['Head'].get_val() == 'Depth':
                    input_vars['Depths'] = self.input['Depths']
                    if rise is not None:
                        input_vars['Depths'].value.limits = (0.0, rise)
                    else:
                        input_vars['Depths'].value.limits = (0.0, self.max_value)
                elif self.input['Head'].get_val() == 'Width of spread':
                    input_vars['Width of spread'] = self.input['Width of spread']
                else:
                    input_vars['Elevations'] = self.input['Elevations']
                    if rise is not None:
                        thalweg_elev = input_vars['Geometry'].value.input['Thalweg invert elevation'].get_val()
                        input_vars['Elevations'].value.limits = (0.0, rise + thalweg_elev)
                    else:
                        input_vars['Elevations'].value.limits = (0.0, self.max_value)
            if unknown != 'Flows':
                input_vars['Flows'] = self.input['Flows']

            # Add all calculation options
            calculate_list = self.input['Calculate'].get_list()

            # Add geometry variables to the list
            # If that item is selected to calculate, remove from input dictionary (but not in subclasses)
            for geom_option in self.input['Geometry'].get_val().get_input_group():
                if geom_option not in [
                        'Shape', 'Head', 'Compute cross-section', 'Original stations', 'Original elevations',
                        'Channel depth', 'Thalweg invert elevation', 'Embedment', 'Alignment', 'Curvature direction',
                        'Only fill flow areas with thalweg locations', 'Ineffective flow stations',
                        'Define gutter cross-slope', 'Curb height', 'Road width',
                        "Use Manning's equation for flow computations",
                ]:
                    calculate_list.append(geom_option)

            calculate_list.append('Slope')

            if self.input['Composite n'].value.input['n entry'].get_val() == 'specify composite n value':
                calculate_list.append('Composite n')

            # # We have some items that would not make sense to make our 'unknown' (or 'to calculate') variable.
            # # So remove these items from our calculate list
            # never_calc_set = {

            # }
            # calculate_list = [item for item in calculate_list if item not in never_calc_set]

            input_vars['Calculate'].value_options = calculate_list

            if unknown in calculate_list:
                input_vars['Calculate'].set_val(unknown)
            elif calc_index < len(calculate_list):
                input_vars['Calculate'].value = calc_index
            else:
                input_vars['Calculate'].value = 1

            self.input['Calculate'] = input_vars['Calculate']

        if 'Depth' in input_vars:
            shape_is_closed = self.input['Geometry'].get_val().shape_is_closed
            rise_to_crown = self.input['Geometry'].get_val().rise_to_crown
            if shape_is_closed and rise_to_crown > 0.0:
                embed = self.input['Geometry'].get_val().input['Embedment'].get_val().input[
                    'Embedment depth'].get_val()
                input_vars['Depths'].get_val().limits = (0.0, rise_to_crown - embed)

        num_calcs = 0
        if 'Flows' in input_vars:
            num_calcs = len(input_vars['Flows'].get_val().get_result())
        elif 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        elif 'Width of spread' in input_vars:
            num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            self.input['Selected profile'].limits = (1, num_calcs)
            input_vars['Selected profile'] = self.input['Selected profile']
            if not self.plot_index_set:
                self.input['Selected profile'].set_val(num_calcs)  # Default to last index
                self.plot_index_set = True
        else:
            self.input['Selected profile'].limits = (0, num_calcs)
            self.input['Selected profile'].set_val(0)
            self.plot_index_set = False

        return input_vars

    def get_results_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
            result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if unknown is None:
            unknown = self.unknown

        # Create a new variable as float list for all items if there is more than one result.
        if self.can_compute:
            if unknown == 'Head':
                if self.input['Head'].get_val() == 'Depth':
                    result_vars['Depths'] = self.results['Depths']
                else:
                    result_vars['Elevations'] = self.results['WSE']
            elif unknown == 'Flows':
                result_vars['Flows'] = self.results['Flows']
            else:
                if unknown in self.results:
                    result_vars[unknown] = self.results[unknown]
                else:
                    # We have a problem!
                    pass

            if self.input['Geometry'].value.input['Shape'].get_val() == 'curb and gutter' and not self.input[
                    'Geometry'].value.input["Use Manning's equation for flow computations"].get_val():
                # Curb and gutter
                if 'Depths' in result_vars:
                    result_vars.pop('Depths')  # Make sure Width of Spread is before Depths
                result_vars['Width of spread'] = self.results['Width of spread']
                result_vars['Depths'] = self.results['Depths']
                result_vars['Gutter depression'] = self.results['Gutter depression']
                result_vars['Flow area'] = self.results['Flow area']
                result_vars['Ratio of gutter flow to total flow'] = self.results['Ratio of gutter flow to total flow']
                result_vars['Ratio of grate flow to total flow'] = self.results['Ratio of grate flow to total flow']
                result_vars['Gutter flow area'] = self.results['Gutter flow area']
                result_vars['Grate flow area'] = self.results['Grate flow area']

            else:

                # Display the Sequent Depths if there is some valid data (ignore if all depths are sub critical)
                _, null_data = self.get_setting('Null data')
                # Only applies when we have supercritical flow
                froude = self.results['Froude number'].get_val()
                max_froude = max(froude) if len(froude) > 0 else 0.0
                if max_froude > 1.0:
                    if not all(depth == null_data for depth in self.results['Sequent depths'].get_val()):
                        result_vars['Sequent depths'] = self.results['Sequent depths']

                result_vars['Flow area'] = self.results['Flow area']
                result_vars['Wetted perimeter'] = self.results['Wetted perimeter']
                result_vars['Hydraulic radius'] = self.results['Hydraulic radius']
                result_vars['Hydraulic depth'] = self.results['Hydraulic depth']
                result_vars['Average velocity'] = self.results['Average velocity']
                result_vars['Top width'] = self.results['Top width']
                result_vars['Froude number'] = self.results['Froude number']
                if self.input['Head'].get_val() == 'Elevation':
                    result_vars['Critical elevation'] = self.results['Critical elevation']
                result_vars['Critical depth'] = self.results['Critical depth']
                result_vars['Critical velocity'] = self.results['Critical velocity']
                result_vars['Critical slope'] = self.results['Critical slope']
                result_vars['Critical top width'] = self.results['Critical top width']
                result_vars['Max shear stress'] = self.results['Max shear stress']
                result_vars['Average shear stress'] = self.results['Average shear stress']

                result_vars['Composite n computation method'] = self.results['Composite n computation method']
                result_vars['Composite n value'] = self.results['Composite n value']

                result_vars['Reynolds number'] = self.results['Reynolds number']
                result_vars['Flow stability'] = self.results['Flow stability']

        return result_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
                result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        input_vars = self.get_input_group(unknown)
        if 'Flows' in input_vars:
            num_calcs = len(input_vars['Flows'].get_val().get_result())
        elif 'Depths' in input_vars:
            num_calcs = len(input_vars['Depths'].get_val().get_result())
        elif 'WSE' in input_vars:
            num_calcs = len(input_vars['WSE'].get_val().get_result())
        elif 'Width of spread' in input_vars:
            num_calcs = len(input_vars['Width of spread'].get_val().get_result())

        if num_calcs > 1:
            profile_index = self.input['Selected profile'].get_val() - 1  # Index for users is 1-based
            results_group = self.get_results_group(unknown)
            results_vars['Selected profile results'] = {}
            for item in results_group:
                results_vars['Selected profile results'][item] = copy.copy(results_group[item])
                if len(results_vars['Selected profile results'][item].value_options) > profile_index:
                    results_vars['Selected profile results'][item].set_val(
                        results_vars['Selected profile results'][item].get_val()[profile_index])

            results_vars['Full results'] = self.get_results_group(unknown)
        else:
            results_vars['Results'] = self.get_results_group(unknown)

        return results_vars

    def get_input_var_with_name(self, name):
        """Get the input variable by name.

        Args:
            name (string): name of the variable
        """
        for _, cur_variable in self.get_input_group(
                'all').items():  # use 'all' as that won't match any unknown
            if cur_variable.name == name:
                return cur_variable
            if cur_variable.type == 'class':
                input_var = self._get_input_var_with_name(
                    name, cur_variable.get_val())
                if input_var:
                    return input_var
        return None

    def _get_input_var_with_name(self, name, variable_class):
        """Return the input variable by name, even into the children.

        Args:
            name (string): Name of the variable in the input dictionary.
            variable_class (dictionary of variables): dictionary to search
        """
        for _, cur_variable in variable_class.get_input_group().items():
            if cur_variable.name == self.unknown:
                return cur_variable
            if cur_variable.type == 'class':
                input_var = self._get_input_var_with_name(
                    name, cur_variable.get_val())
                if input_var:
                    return input_var
        return None

    def set_stations_and_elevation(self, stations, elevations, n_data=None):
        """Sets the stations and elevations.

        Args:
            stations (list of floats): stations of the channel geometry
            elevations (list of floats): elevations of the channel geometry
            n_data (list of floats): optional manning's n data

        """
        self.input['Geometry'].get_val().input['Cross-section'].value.input['Table options'].value.input[
            'Number of items'].set_val(len(stations))
        self.input['Geometry'].get_val().input['Cross-section'].value.input['Data input'].value.input[
            'Station'].set_val(stations, app_data=self.app_data)
        self.input['Geometry'].get_val().input['Cross-section'].value.input['Data input'].value.input[
            'Elevation'].set_val(elevations, app_data=self.app_data)
        if n_data is not None:
            self.input['Geometry'].get_val().input['Cross-section'].value.input['Data input'].value.input[
                "Manning's n"].set_val(n_data)
        self.input['Geometry'].get_val().input['Cross-section'].value.compute_data()

    def set_shape(self, shape):
        """Sets the shape of the channel.

        Args:
            shape (string): The channel shape
        """
        self.input['Geometry'].get_val().input['Shape'].set_index(shape)

    def check_for_vertical_wall_warning(self):
        """Checks whether there was vertical walls added to the channel.

        Returns:
            (bool): True if there was vertical walls added, False if not
        """
        return self.input['Geometry'].get_val().calculator.check_for_vertical_wall_warning(
            self.normal_wses, self.results['Critical elevation'].get_val())

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        self.warnings.extend(self.check_for_vertical_wall_warning())

        return self.warnings

    def assign_unknown_results(self):
        """Assigns the results for unknown variables."""
        # Remove any results created from calculating an unknown other than depth or flow:
        self.results = {k: v for k, v in self.results.items() if k in self.calculator.original_results}

        input_list = [self.input, self.input['Geometry'].value.input, self.input['Composite n'].value.input]
        for location in input_list:
            if self.unknown in self.results:
                # Should be handled already
                return
            elif self.unknown in location and self.unknown in self.calculator.results:
                self.results[self.unknown] = copy.copy(location[self.unknown])
                self.results[self.unknown].type = 'float_list'
                self.results[self.unknown].value_options = self.calculator.results[self.unknown]
                self.results[self.unknown].value = len(self.calculator.results[self.unknown])
                return
