"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.adcirc.components.adcirc_component import AdcircComponent
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.feedback.bc_worker_thread import BcMappingWorkerThread
from xms.adcirc.feedback.tidal_worker_thread import TidalMappingWorkerThread
from xms.adcirc.gui.model_control_dlg import AdcircModelControlDlg
from xms.adcirc.gui.partial_export_dlg import PartialExportDialog


class SimComponent(AdcircComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = SimData(self.main_file)
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            ('Export...', 'open_partial_export'),
        ]  # [(menu_text, menu_method)...]

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (:obj:`dict`): A dictionary with keys being UUIDs as strings representing the objects being linked
                into this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (:obj:`bool`): True if the component is locked for editing.
                Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param == 'boundCov':
                    self.data.info.attrs['bound_uuid'] = link_uuid
                    action = ActionRequest(
                        main_file=self.main_file,
                        modality='MODAL',
                        class_name=self.class_name,
                        module_name=self.module_name,
                        method_name='map_bc',
                        comp_uuid=self.uuid
                    )
                    actions.append(action)
                elif xml_param == 'tidalCons':
                    self.data.info.attrs['tidal_uuid'] = link_uuid
                    action = ActionRequest(
                        main_file=self.main_file,
                        modality='MODAL',
                        class_name=self.class_name,
                        module_name=self.module_name,
                        method_name='map_tidal',
                        comp_uuid=self.uuid
                    )
                    actions.append(action)
        self.data.commit()
        return messages, actions

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        pe_tree = query.project_tree
        sim_uuid = query.parent_item_uuid()
        sim_node = tree_util.find_tree_node_by_uuid(pe_tree, sim_uuid)
        # Get the linked domain mesh if it exists
        child = tree_util.descendants_of_type(
            sim_node, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, only_first=True
        )
        domain_uuid = child.uuid if child else ''
        # Get the linked wind grid if it exists
        child = tree_util.descendants_of_type(
            sim_node, xms_types=['TI_CGRID2D_PTR'], allow_pointers=True, only_first=True
        )
        grid_uuid = child.uuid if child else ''
        # Get the linked recording stations coverage if it exists
        child = tree_util.descendants_of_type(
            sim_node,
            xms_types=['TI_COVER_PTR'],
            allow_pointers=True,
            only_first=True,
            coverage_type='Recording Stations'
        )
        station_uuid = child.uuid if child and child.coverage_type == 'Recording Stations' else ''
        dlg = AdcircModelControlDlg(
            win_cont, self.data, pe_tree, query.global_time, domain_uuid, grid_uuid, station_uuid
        )
        if dlg.exec():
            self.data.commit()
        return [], []

    def open_partial_export(self, query, params, win_cont):
        """Opens the dialog for exporting individual files of an ADCIRC simulation.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
           (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        dlg = PartialExportDialog(win_cont, query)
        dlg.exec()
        return [], []

    def copy_external_files(self, new_main_file):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_main_file (:obj:`str`): The location of the new component main file in the package.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        new_data = SimData(new_main_file)
        new_data.copy_external_files()
        new_data.commit()
        return ''

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (:obj:`str`): The location of the new main file.
            convert_filepaths (:obj:`bool`): False if only the project directory should be updated.

        Returns:
            (:obj:`str`): Message on failure, empty string on success
        """
        new_data = SimData(new_main_file)
        if not convert_filepaths:
            # This case is to handle opening a package project for the first time.
            comp_folder = os.path.dirname(self.main_file)
            package_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            new_data.info.attrs['proj_dir'] = package_proj_dir
            new_data.commit()  # Save the updated project directory
            return ''
        err_msg = new_data.update_proj_dir()
        # Copy the newly saved file to temp.
        io_util.copyfile(new_main_file, self.main_file)
        return err_msg

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path if
                necessary.
                - messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        if save_type == 'UNLOCK':
            self.data.migrate_relative_paths()

        return super().save_to_location(new_path, save_type)

    def map_bc(self, query, params, win_cont):
        """Action Request method for mapping a source BC coverage.

        Not really a dialog ActionRequest, but we want to lock the GUI

        Args:
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS
            params (:obj:`dict`): The ActionRequest parameter map, unused
            win_cont (:obj:`QWidget`): The parent window, unused

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        note = "Applied data generated by this operation is valid with the simulation's current domain mesh and " \
               "source Boundary Conditions coverage. Any further editing of the domain mesh or source coverage may " \
               "invalidate the applied boundary conditions. To ensure that the data exported by SMS is up to date, " \
               "reapply the source Boundary Conditions coverage after editing the mesh, coverage geometry, or " \
               "boundary condition arc attributes."
        worker = BcMappingWorkerThread(query, self, win_cont)
        display_text = {
            'title': 'Applying Boundary Conditions',
            'working_prompt': 'Applying boundary conditions to ADCIRC simulation. Please wait...',
            'error_prompt':
                'Error(s) encountered applying boundary conditions to simulation. Review log output for '
                'more details.',
            'warning_prompt':
                'Warning(s) encountered applying boundary conditions to simulation. Review log output '
                'for more details.',
            'success_prompt':
                'Successfully applied boundary conditions to simulation. Close this dialog to load the '
                'applied data into SMS.',
            'note': note,
            'auto_load': 'Auto load applied data into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        feedback_dlg.exec()
        return [], []

    def map_tidal(self, query, params, win_cont):
        """Action Request method for mapping source tidal constituents.

        Not really a dialog ActionRequest, but we want to lock the GUI

        Args:
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS
            params (:obj:`dict`): The ActionRequest parameter map, unused
            win_cont (:obj:`QWidget`): The parent window, unused

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        note = "Applied data generated by this operation is valid with the simulation's current domain mesh and " \
               "applied boundary conditions. Any further editing of the domain mesh or source boundary condition " \
               "attributes may invalidate the mapped tidal constituents. To ensure that the data exported by SMS is " \
               "up to date, reapply tidal constituents after editing the domain or source Boundary Conditions coverage."
        worker = TidalMappingWorkerThread(query, self, win_cont)
        display_text = {
            'title': 'Applying Tidal Constituents',
            'working_prompt': 'Applying tidal constituents to ADCIRC simulation. Please wait...',
            'error_prompt':
                'Error(s) encountered applying tidal constituents to simulation. Review log output for '
                'more details.',
            'warning_prompt':
                'Warning(s) encountered applying tidal constituents to simulation. Review log output '
                'for more details.',
            'success_prompt':
                'Successfully applied tidal constituents to simulation. Close this dialog to load the '
                'applied data into SMS.',
            'note': note,
            'auto_load': 'Auto load applied data into SMS when operation is complete',
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        feedback_dlg.exec()
        return [], []
