"""Class for managing interprocess communication with XMS."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules


class XmsData:
    """Class for managing interprocess communication with XMS."""
    def __init__(self, query=None, sim_component=None):
        """Constructor.

        Args:
            query (Query): The XMS interprocess communication object
            sim_component (SimComponent): the simulation component
        """
        self.sim_global_parameters = None
        self._query = query
        # self._logger = util.get_logger()
        self._sim_comp = sim_component
        self._sim_item = None
        self._sim_tree_item = None
        self._sim_uuid = None
        self._sim_name = None
        self._do_ugrid = None
        self._xmugrid = None
        self._cogrid = None
        self._ugrid_item = None

    @property
    def query(self):
        """Returns the xmsapi interprocess communication object.

        Notes:
            If constructed in a component runner script, Query should be passed in on construction. If constructed in
            an export or import script, creation of the Query can happen here. Only one Query/XmsData object should be
            constructed per script.
        """
        if self._query is None:
            self._query = Query()
        return self._query

    @property
    def display_wkt(self):
        """Returns the display projection well known text."""
        if self._query:
            return self._query.display_projection.well_known_text
        return ''

    @property
    def display_projection(self):
        """Returns the display projection well known text."""
        if self._query:
            return self._query.display_projection
        return ''

    @property
    def coordinate_system(self):
        """Returns the display projection coordinate system."""
        if self._query:
            return self._query.display_projection.coordinate_system
        return ''

    @property
    def horizontal_units(self):
        """Returns the display projection horizontal units."""
        if self._query:
            return self._query.display_projection.horizontal_units
        return ''

    def parent_item(self):
        """Returns the display projection horizontal units."""
        if self._query:
            return self._query.parent_item()
        return ''

    def load_component_ids(self, bc_comp, arcs=True):
        """Calls the query load_component_ids function.

        Args:
            component (xms.components.bases.coverage_component_base.CoverageComponentBase): The coverage component to
                load id files for
            arcs (bool): True if you want arc component id maps

        Returns:
            dict: The file dict.
        """
        if self._query:
            return self._query.load_component_ids(bc_comp, arcs=arcs)
        return ''

    def _get_sim_data(self):
        """Get the simulation data from the XMS project tree.

        Raises:
            RuntimeError: If the simulation data cannot be retrieved.
        """
        if self._sim_item is None:
            if self._sim_comp:
                self._sim_item = self.query.parent_item()
            else:
                self._sim_item = self.query.current_item()
            if self._sim_item:
                self._sim_uuid = self._sim_item.uuid
                self._sim_name = self._sim_item.name
                if self._sim_comp is None:
                    do_comp = self.query.item_with_uuid(self._sim_uuid, model_name='RSM', unique_name='SimComponent')
                    from xms.rsm.components.sim_component import SimComponent
                    self._sim_comp = SimComponent(do_comp.main_file)
                self._sim_tree_item = self.get_tree_item(self._sim_uuid)

    @property
    def sim_component(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_comp is None:
            self._get_sim_data()
        return self._sim_comp

    @property
    def sim_tree_item(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_tree_item is None:
            self._get_sim_data()
        return self._sim_tree_item

    @property
    def sim_uuid(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_uuid is None:
            self._get_sim_data()
        return self._sim_uuid

    @property
    def sim_name(self):
        """Returns the name of simulation."""
        if self._sim_name is None:
            self._get_sim_data()
        return self._sim_name

    @property
    def do_ugrid(self):
        """Retrieves the simulation's data_objects UGrid."""
        if self._do_ugrid is None:
            self._logger.info('Retrieving UGrid from SMS...')
            uuid = self.ugrid_item.uuid if self.ugrid_item else ''
            self._do_ugrid = self.query.item_with_uuid(uuid)
            if self._do_ugrid is not None:
                self._cogrid = read_grid_from_file(self._do_ugrid.cogrid_file)
        return self._do_ugrid

    @property
    def xmugrid(self):
        """Retrieves the simulation's data_objects XmUGrid."""
        if self._xmugrid is None and self.cogrid:
            self._xmugrid = self.cogrid.ugrid
        return self._xmugrid

    @property
    def cogrid(self):
        """Retrieves the simulation's domain CoGrid."""
        if self._cogrid is None:
            _ = self.do_ugrid
        return self._cogrid

    @property
    def ugrid_item(self):
        """Returns the linked UGrid tree item, or None if there is none."""
        if self._ugrid_item is None and self.sim_tree_item is not None:
            ugrid_ptr = tree_util.descendants_of_type(
                tree_root=self.sim_tree_item, only_first=True, xms_types=self._grid_tree_types, allow_pointers=True
            )
            if ugrid_ptr is not None:
                self._ugrid_item = self.get_tree_item(ugrid_ptr.uuid)
        return self._ugrid_item

    def get_tree_item(self, item_uuid):
        """Get a project explorer item from UUID.

        Args:
            item_uuid (str): UUID of the item to retrieve

        Returns:
            (TreeNode): The tree item with specified UUID or None if not found
        """
        return tree_util.find_tree_node_by_uuid(self.query.project_tree, item_uuid)

    def add_component_to_xms(self, do_component, actions):
        """Add a component to XMS.

        Args:
            do_component (xms.data_objects.parameters.Component): The component to add
            actions (list): The actions to perform
        """
        self.query.add_component(do_component=do_component, actions=actions)
        self.query.link_item(taker_uuid=self.sim_uuid, taken_uuid=do_component.uuid)
