"""Module to perform xms.api Query operations for the ADCIRC model checker."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util

# 4. Local modules
from xms.adcirc.components.sim_component import SimComponent
from xms.adcirc.data.mapped_bc_data import MappedBcData
from xms.adcirc.data.mapped_flow_data import MappedFlowData
from xms.adcirc.data.mapped_tidal_data import MappedTidalData
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.dmi.sim_component_queries import SimComponentQueries


class ModelCheckQueries:
    """Performs xms.api calls for ADCIRC model checks."""
    def __init__(self, query=None):
        """Initialize the API helper.

        Args:
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS. If provided, Context should be
                at the simulation level
        """
        self._query = query
        if not self._query:
            self._query = Query()
        self._sim_item = None
        self._xms_data = {}

    def _get_wind_dset_time_ranges(self):
        """Get the beginning and ending times of input wind datasets.

        Returns:
            (:obj:`xms.data_object`) or (:obj:`datetime`): Times returned may be xms.data_objects
            durations or datetimes.
        """
        self._xms_data['wind_dsets'] = {}
        nws = self._xms_data['sim_data'].wind.attrs['NWS']
        scalar_uuid = None
        scalar_name = None
        if nws in [1, 2, 4, 5]:  # Get scalar and vector mesh datasets
            scalar_uuid = self._xms_data['sim_data'].wind.attrs['mesh_pressure']
            scalar_name = 'atmospheric pressure on the mesh'
            vector_uuid = self._xms_data['sim_data'].wind.attrs['mesh_wind']
            vector_name = f'{"stress" if nws < 4 else "velocity"} on the mesh'
        elif nws in [3, 6]:  # Get scalar and vector grid datasets
            vector_uuid = self._xms_data['sim_data'].wind.attrs['grid_wind']
            vector_name = 'velocity on a wind grid'
            if nws != 3:
                scalar_name = 'atmospheric pressure on a wind grid'
                scalar_uuid = self._xms_data['sim_data'].wind.attrs['grid_pressure']
        else:
            return  # No datasets to check

        if scalar_uuid:  # Select the scalar dataset by UUID
            self._xms_data['wind_dsets'][scalar_name] = self._query.item_with_uuid(scalar_uuid)
        if vector_uuid:  # Select the vector dataset by UUID
            self._xms_data['wind_dsets'][vector_name] = self._query.item_with_uuid(vector_uuid)

    def get_xms_model_check_data(self):
        """Retrieve data from XMS needed to perform model checks.

        Returns:
            (:obj:`tuple`): First element is dict of XMS data, second element is list of errors encountered.
        """
        errors = []

        # Get the project explorer dump
        self._xms_data['pe_tree'] = self._query.project_tree

        # Get display projection and XMS global time from simulation Context level
        try:
            projection = self._query.display_projection
            self._xms_data['projection'] = projection.coordinate_system
            self._xms_data['horizontal_units'] = projection.horizontal_units
            self._xms_data['vertical_units'] = projection.vertical_units
            self._xms_data['global_time'] = self._query.global_time
        except Exception:
            errors.append(
                (
                    'Unknown error retrieving display projection and SMS global time.',
                    'Running ADCIRC model checks requires the current display projection and global time information.',
                    'Please try running model checks again.'
                )
            )

        # Get the simulation component main file.
        try:
            sim_uuid = self._query.current_item_uuid()
            self._sim_item = tree_util.find_tree_node_by_uuid(self._xms_data['pe_tree'], sim_uuid)
            sim_comp = self._query.item_with_uuid(sim_uuid, model_name='ADCIRC', unique_name='Sim_Component')
            self._xms_data['sim_data'] = SimData(sim_comp.main_file)
            self._xms_data['py_comp'] = SimComponent(sim_comp.main_file)
        except Exception:
            errors.append(
                (
                    'Unknown error retrieving ADCIRC simulation Model Control data.',
                    'Running ADCIRC model checks requires the current ADCIRC Model Control values.',
                    'Please try running model checks again.'
                )
            )

        sim_helper = SimComponentQueries(self._xms_data['py_comp'], self._query)

        # Get the domain mesh dump as a cogrid to see if it has changed.
        self._xms_data['current_cogrid'] = None
        try:
            mesh_item = sim_helper.find_sim_link('TI_MESH2D_PTR')
            self._xms_data['current_cogrid'] = self._query.item_with_uuid(mesh_item.uuid).cogrid_file
        except Exception:
            errors.append(
                (
                    'Unknown error retrieving ADCIRC domain mesh.',
                    'Unable to verify that applied boundary conditions and tidal constituents are up to date.',
                    'Please try running model checks again.'
                )
            )

        # Get the wind coverage dump, if applicable.
        self._xms_data['wind_cov'] = None
        cov_item = sim_helper.find_sim_link('TI_COVER_PTR', coverage_type='WIND')
        if cov_item:
            try:
                self._xms_data['wind_cov'] = self._query.item_with_uuid(cov_item.uuid, generic_coverage=True)
            except Exception:
                errors.append(
                    (
                        'Unknown error retrieving ADCIRC storm track coverage.',
                        'Unable to verify that wind track coverage applied to simulation is valid for the ADCIRC run.',
                        'Please try running model checks again.'
                    )
                )

        # Get the recording stations coverage dump, if applicable.
        self._xms_data['station_uuid'] = None
        cov_item = sim_helper.find_sim_link('TI_COVER_PTR', coverage_type='Recording Stations')
        if cov_item:
            try:
                self._xms_data['station_uuid'] = cov_item.uuid
            except Exception:
                errors.append(
                    (
                        'Unknown error retrieving ADCIRC Recordin Stations coverage.',
                        'Unable to verify that the Recording Stations coverage applied to simulation is valid for the '
                        'ADCIRC run.', 'Please try running model checks again.'
                    )
                )

        # Get the mapped boundary conditions component main file.
        try:
            mapped_bcs = tree_util.descendants_of_type(self._sim_item, unique_name='Mapped_Bc_Component', recurse=False)
            self._xms_data['mapped_bc_data'] = MappedBcData(mapped_bcs[0].main_file)  # Should be one and only one.
        except Exception:
            errors.append(
                (
                    "STOP! Don't continue with this simulation. No boundary conditions have been applied.",
                    'An ADCIRC Boundary Conditions coverage must be applied to this simulation before running.',
                    'Right-click on the "Map Data" tree item in the project explorer and select "New Coverage". '
                    'Under "ADCIRC", choose "Boundary Conditions" and press "OK". Drag and drop the new coverage '
                    'under the simulation after boundary condition specification is complete.'
                )
            )

        # Get the mapped tidal constituent component main file(s).
        try:
            mapped_tides = tree_util.descendants_of_type(
                self._sim_item, unique_name='Mapped_Tidal_Component', recurse=False
            )
            self._xms_data['tidal_data'] = [MappedTidalData(item.main_file) for item in mapped_tides]
        except Exception:
            self._xms_data['tidal_data'] = []  # Not necessarily an error.

        # Get the mapped flow constituent component main file, if there is one.
        try:
            mapped_flows = tree_util.descendants_of_type(
                self._sim_item, unique_name='Mapped_Flow_Component', recurse=False
            )
            self._xms_data['flow_data'] = MappedFlowData(mapped_flows[0].main_file)  # Should be at most one
        except Exception:
            self._xms_data['flow_data'] = None  # Not necessarily an error.

        # Get time ranges of various input wind datasets if applicable.
        try:
            self._get_wind_dset_time_ranges()
        except Exception:
            errors.append(
                (
                    'Unknown error transient wind input dataset time ranges.',
                    'Input data sets cannot be checked to ensure that their time steps overlap the ADCIRC model run.',
                    'Please try running model checks again.'
                )
            )

        return self._xms_data, errors

    def send_model_check_messages(self, errors):
        """Send Model Check warnings and errors back to XMS.

        Args:
            errors (:obj:`list`): List of the xms.api.dmi.ModelCheckError objects to send
        """
        if errors:
            self._query.add_model_check_errors(errors)
            self._query.send()
