"""Provide progress feedback when running an ADCIRC model from SMS."""

# 1. Standard Python modules
import datetime
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.adcirc.data.sim_data import SimData


class AdcircTracker:
    """Class to track the progress of a running ADCIRC model."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    num_ts = 0.0
    first_ts = -1
    nscreen = 0

    @staticmethod
    def calculate_progress(ts):
        """Calculate progress given a timestep.

        Args:
            ts (:obj:`int`): The current timestep number

        Returns:
            (:obj:`float`): Progress as a percent of all timesteps.
        """
        # Cast to float to avoid truncation if int
        return (ts / float(AdcircTracker.num_ts)) * 100.0

    @staticmethod
    def progress_function():
        """Method called inside the progress loop for computing percent done."""
        if not AdcircTracker.echo_file:
            AdcircTracker.echo_file = AdcircTracker.prog.command_line_output_file

        need_progress_update = False
        current_ts = 0
        try:  # Parse the command line output to determine current timestep
            with open(AdcircTracker.echo_file, "r") as f:
                f.seek(AdcircTracker.echo_pos)  # Skip past lines we already read
                echo_line = f.readline()
                while echo_line:  # Have a complete line
                    if (echo_line.endswith('\n') or echo_line.endswith('\r')) and \
                            echo_line.strip().startswith("TIME STEP"):  # This is a time step output line
                        echo_vals = shlex.split(echo_line)
                        try:
                            current_ts = int(echo_vals[3])
                            if AdcircTracker.first_ts == -1:
                                # adjusts the total number of timesteps for hotstart runs
                                AdcircTracker.first_ts = current_ts - AdcircTracker.nscreen
                                AdcircTracker.num_ts -= AdcircTracker.first_ts
                            current_ts -= AdcircTracker.first_ts
                        except ValueError:
                            pass
                        need_progress_update = True
                        AdcircTracker.echo_pos = f.tell()  # Store position in file of lines already read
                    echo_line = f.readline()  # Keep reading until EOF
        except Exception:
            pass  # File might not exist yet

        if need_progress_update:  # Update model process progress in SMS Simulation Run Queue
            percent_done = AdcircTracker.calculate_progress(current_ts)
            AdcircTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def start_tracking():
        """Entry point for the ADCIRC progress script."""
        AdcircTracker.query = Query(progress_script=True)
        AdcircTracker.prog = AdcircTracker.query.xms_agent.session.progress_loop

        # Get the simulation's hidden component
        sim_uuid = AdcircTracker.query.current_item_uuid()  # Get the simulation uuid
        sim_comp = AdcircTracker.query.item_with_uuid(sim_uuid, model_name='ADCIRC', unique_name='Sim_Component')
        data = SimData(sim_comp.main_file)

        # Find out if we are echoing to the command prompt or a file
        AdcircTracker.nscreen = data.general.attrs['NSCREEN']
        if AdcircTracker.nscreen < 0:
            AdcircTracker.nscreen = abs(AdcircTracker.nscreen)
            AdcircTracker.echo_file = "adcirc.log"

        # Find out how many iterations this run has
        # Get the timestep size in seconds
        dtdp = data.timing.attrs['DTDP']
        # Get the run length in days
        run_duration_days = data.timing.attrs['RUNDAY']
        # Convert the run length to seconds
        run_duration_secs = datetime.timedelta(days=run_duration_days) / datetime.timedelta(seconds=1)
        # Find the number of timesteps
        AdcircTracker.num_ts = run_duration_secs / dtdp

        AdcircTracker.prog.set_progress_function(AdcircTracker.progress_function)
        AdcircTracker.prog.start_loop()
