"""Worker thread for the Boundary Conditions coverage unmapping operation."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import read_grid_from_file
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread

# 4. Local modules
from xms.adcirc.dmi.mapped_bc_component_queries import MappedBcComponentQueries
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.bc_unmapper import BcUnMapper


class BcUnMappingWorkerThread(ProcessFeedbackThread):
    """Worker thread class to do computations in another thread."""
    def __init__(self, query, mapped_bc_comp, do_ugrid, parent):
        """Construct the worker.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
            mapped_bc_comp (:obj:`MappedBcComponent`): component for the mapped Boundary Condition
            do_ugrid (:obj:`data_objects.parameters.UGrid`): The data_object for the linked mesh
            parent (:obj:`QWidget`): The Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.query = query
        self.do_ugrid = do_ugrid
        self.mapped_bc_comp = mapped_bc_comp

    def _do_work(self):
        """Worker code to execute in a separate thread."""
        XmLog().instance.info('Retrieving applied boundary conditions data from SMS...')
        query_helper = MappedBcComponentQueries(self.mapped_bc_comp, self.query)
        xms_data = query_helper.get_coverage_convert_data()
        if LogEchoQSignalStream.logged_error:
            return

        # Convert the mapped boundary conditions to a source ADCIRC Boundary Conditions coverage.
        try:
            # Find the linked mesh
            XmLog().instance.info("Reading simulation's ADCIRC domain mesh...")
            xms_data['mesh'] = read_grid_from_file(self.do_ugrid.cogrid_file)

            unmapper = BcUnMapper(
                self.mapped_bc_comp, xms_data['temp_dir'], xms_data['comp_name'], xms_data['mesh'],
                xms_data['flow_data']
            )
            do_cov, do_comp, py_comp = unmapper.unmap_data()

            # Set up the coverage component id maps for the new source BC coverage.
            unmapper.initialize_coverage_comp_ids()

            # Add the built data to Query Context
            if do_cov and do_comp and py_comp:
                XmLog().instance.info('Packaging converted coverage data to send to SMS...')
                # Get any unmapped flow data that goes with the unmapped BC coverage.
                flow_forcing_mesh, flow_forcing_dsets = unmapper.get_unmapped_flow_data()
                query_helper.add_xms_data_for_coverage_convert(
                    do_cov, do_comp, py_comp, flow_forcing_mesh, flow_forcing_dsets
                )

            if not LogEchoQSignalStream.logged_error:  # All done, no errors
                XmLog().instance.info('Boundary Conditions coverage successfully created from applied data.')
        except Exception:
            XmLog().instance.exception('Unable to convert applied boundary condition data to a coverage.')
