"""Worker thread for the fort.14 exporter."""

# 1. Standard Python modules
from io import StringIO
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackThread

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort14_writer import export_geometry_to_fort14, export_mapped_bc_data
from xms.adcirc.mapping.mapping_util import grid_projection_mismatches_display


class Fort14ExportWorkerThread(ProcessFeedbackThread):
    """Worker thread for exporting an ADCIRC fort.14 file."""
    def __init__(self, filename, query, parent):
        """Construct the worker.

        Args:
            filename (:obj:`str`): Path to the export location.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Parent of the QThread
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.filename = filename
        self.query = query

    def _do_work(self):
        """Thread runner that exports the fort.14."""
        XmLog().instance.info('Getting applied boundary conditions of the simulation')
        geom_name = ''
        domain_mesh = None
        try:
            # Get the parent simulation's UUID
            sim_uuid = self.query.parent_item_uuid()
            sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, sim_uuid)

            # Get the domain mesh
            mesh_item = tree_util.descendants_of_type(
                sim_item, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, only_first=True, recurse=False
            )
            if mesh_item:
                geom_name = mesh_item.name
                geom_uuid = mesh_item.uuid
                domain_mesh = self.query.item_with_uuid(geom_uuid)  # Should be one and only one

            if not domain_mesh:
                XmLog().instance.error('Error getting mesh geometry when exporting.')
                return

            # Try to get the mapped BC component, may not exist.
            mapped_bcs = tree_util.descendants_of_type(sim_item, unique_name='Mapped_Bc_Component', recurse=False)
            mapped_bc_comp_files = [mapped_bc.main_file for mapped_bc in mapped_bcs]
        except Exception:
            XmLog().instance.warning(
                'No applied boundary conditions were found in the simulation. Only the domain mesh geometry will '
                'be exported.'
            )
        else:  # Have applied boundary conditions. Try to export them.
            try:
                export_mapped_bc_data(self.filename, domain_mesh.cogrid_file, geom_name, mapped_bc_comp_files)
            except Exception:
                XmLog().instance.exception('Unable to export boundary conditions to fort.14 file.')
            return

        try:  # No applied boundary conditions on the simulation. Try to export just the mesh.
            # Check for inconsistent native and display projections.
            native_proj = domain_mesh.native_projection
            display_proj = domain_mesh.projection
            projection_error = grid_projection_mismatches_display(native_proj, display_proj)
            if projection_error:
                raise RuntimeError(projection_error)

            # Write the mesh geometry to the file
            ss = StringIO()
            export_geometry_to_fort14(ss, domain_mesh.cogrid_file, geom_name)
            ss.write('0\n0\n0\n0')  # Set number of boundary condition nodestrings to zero.
            XmLog().instance.info('Saving in-memory stream to file.')
            with open(self.filename, 'w') as f:
                ss.seek(0)
                shutil.copyfileobj(ss, f, 100000)
            XmLog().instance.info('Successfully exported fort.14 file.')
        except Exception:
            XmLog().instance.exception('Unable to export ADCIRC domain mesh to export fort.14.')
