"""Worker thread for the fort.15 importer."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort15_reader import read_adcirc_simulation
from xms.adcirc.gui.import_process_feedback_dlg import ImportProcessFeedbackDlg


class Fort15ImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing an ADCIRC fort.15 file."""
    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.reader = None

    def _do_work(self):
        """Thread runner that imports the fort.15."""
        try:
            self.reader = read_adcirc_simulation()
        except Exception:
            XmLog().instance.exception('Error(s) encountered while reading fort.15 ADCIRC control file.')


def import_fort15_with_feedback():
    """Driver for the fort.15 import script with a feedback dialog."""
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope

    worker = Fort15ImportWorkerThread(win_cont)
    display_text = {
        'title': 'Reading ADCIRC Simulation',
        'working_prompt': 'Reading fort.15 ADCIRC control file. Please wait...',
        'error_prompt': 'Error(s) encountered reading fort.15. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading fort.15. Review log output for more details.',
        'success_prompt': 'Successfully read ADCIRC simulation.',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ImportProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send()  # Send data back to XMS
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
