"""Worker thread for the tidal constituent mapping operation."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread

# 4. Local modules
from xms.adcirc.dmi.sim_component_queries import SimComponentQueries, unlink_tidal_comp_on_error
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.mapping.tidal_mapper import map_tides


class TidalMappingWorkerThread(ProcessFeedbackThread):
    """Worker thread class to do computations in another thread."""
    def __init__(self, query, sim_comp, parent):
        """Constuct the worker.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
            sim_comp (:obj:`SimComponent`): The simulation data
            parent (:obj:`QWidget`): The Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.query = query
        self.sim_comp = sim_comp

    def _do_work(self):
        """Worker code to execute in a separate thread."""
        XmLog().instance.info('Retrieving source tidal constituent data from SMS.')

        # Request the required source data from XMS
        query_helper = SimComponentQueries(self.sim_comp, self.query)
        tidal_data = query_helper.get_tidal_mapping_data()
        if tidal_data['error']:
            unlink_tidal_comp_on_error(tidal_data, self.query)
            return

        # Perform the mapping
        try:
            from xms.adcirc.components.mapped_bc_component import MappedBcComponent
            mapped_bc_comp = MappedBcComponent(tidal_data['mapped_bc_main_file'])
            # Apply the tidal constituents to the mapped boundary conditions.
            map_tides(self.query, tidal_data, mapped_bc_comp)
            if not LogEchoQSignalStream.logged_error:  # All done, no errors
                XmLog().instance.info('Tidal constituent data successfully applied to ADCIRC simulation.')
            else:
                raise RuntimeError()
        except Exception:
            unlink_tidal_comp_on_error(tidal_data, self.query)
            XmLog().instance.exception('Unable to apply tidal constituents to ADCIRC simulation.')
