"""The ADCIRC simulation model control dialog."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import webbrowser

# 2. Third party modules
from harmonica.resource import ResourceManager
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QMessageBox, QSizePolicy, QVBoxLayout

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util as tr_util
from xms.api.tree.tree_node import TreeNode
from xms.api.tree.tree_util import ProjectExplorerTreeCreator
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import datetime_to_qdatetime, ISO_DATETIME_FORMAT, qdatetime_to_datetime
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.adcirc.data.sim_data import REG_KEY_SUPPRESS_HOTSTART, REG_KEY_SUPPRESS_NONPERIODIC, REG_KEY_SUPPRESS_WIND_FILE
from xms.adcirc.gui import gui_util
from xms.adcirc.gui.harmonics_table import HarmonicsTableWidget
import xms.adcirc.gui.model_control_dlg_consts as const
from xms.adcirc.gui.model_control_dlg_ui import Ui_AdcircModelControlDlg
from xms.adcirc.gui.nws_file_table import WindFileTableWidget
from xms.adcirc.gui.nws_owi_file_table import WindOWIFileTableWidget
from xms.adcirc.gui.output_table import OutputTableWidget


class AdcircModelControlDlg(XmsDlg):
    """A dialog showing the sediment material list and properties."""
    def __init__(self, win_cont, data, pe_tree, global_time, domain_uuid='', grid_uuid='', station_uuid=''):
        """Initializes the sediment material list and properties dialog.

        Args:
            win_cont (:obj:`QWidget`): Parent dialog
            data (:obj:`SimData`): Data class containing the Model Control parameters
            pe_tree (:obj:`TreeNode`): Project explorer dump from Query at root
            global_time (:obj:`datetime.datetime`): The XMS global time
            domain_uuid (:obj:`str`): UUID of the domain mesh, if it exists
            grid_uuid (:obj:`str`): UUID of the wind grid, if it exists
            station_uuid (:obj:`str`): UUID of the recording station coverage, if it exists
        """
        super().__init__(win_cont, 'xmsadcirc.gui.model_control_dlg')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC'
        self.data = data
        self.domain_uuid = domain_uuid
        self.grid_uuid = grid_uuid
        self.station_uuid = station_uuid
        self.global_time = global_time
        self.ui = Ui_AdcircModelControlDlg()
        self.ui.setupUi(self)

        # Available harmonica models for the harmonic analysis tab
        self.available_models = ResourceManager.available_models()

        # Table widgets
        self.harmonics_table = None
        self.nws10_table = None
        self.nws11_table = None
        self.nws12_table = None
        self.nws15_table = None
        self.nws16_table = None
        self.output_table = None

        # Commonly used edit field validators
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_noneg_validator = QxDoubleValidator(bottom=0.001, parent=self)
        self.int_validator = QxIntValidator(parent=self)
        self.int_noneg_validator = QxIntValidator(parent=self)
        self.int_noneg_validator.setBottom(0)
        self.int_pos_validator = QxIntValidator(parent=self)
        self.int_pos_validator.setBottom(1)

        # For dataset selectors that must be on the domain mesh
        self.domain_pe_tree = tr_util.trim_project_explorer(pe_tree, self.domain_uuid)
        # For dataset selectors that must be on the wind CGrid
        self.grid_pe_tree = tr_util.trim_project_explorer(pe_tree, self.grid_uuid)

        # Set up the dialog
        self._setup_ui()

    @staticmethod
    def _add_cbx_opts(cbx):
        """Add a combobox's option texts and user data values.

        Args:
            cbx (:obj:`QCombobox`): The combobox widget.
        """
        cbx_name = cbx.objectName()
        model_vals = const.MC_CBX_OPTS[cbx_name][1]
        for idx, opt in enumerate(const.MC_CBX_OPTS[cbx_name][0]):  # Loop through the list of display option texts
            cbx.addItem(opt, model_vals[idx])  # Add the ADCIRC value as UserData

    @staticmethod
    def _set_cbx_opt_from_data(cbx, value):
        """Set a combobox's  current option index based on its user data value."""
        index = cbx.findData(value)
        if index != -1:
            cbx.setCurrentIndex(index)

    def _setup_ui(self):
        """Setup dialog widgets."""
        self._add_tables()
        self._add_validators()
        self._add_cbx_options()
        self._connect_slots()
        self._load_data()
        self.ui.mc_tabs.setCurrentIndex(0)

    def _add_tables(self):
        """Add pandas DataFrame table widgets for all tabs."""
        self._add_output_table()
        self._add_wind_tables()
        self._add_harmonics_table()

    def _add_validators(self):
        """Add validators on all edit fields."""
        self._add_general_validators()
        self._add_formulation_validators()
        self._add_timing_validators()
        self._add_output_validators()
        self._add_wind_validators()
        self._add_nodal_atts_validators()
        self._add_harmonics_validators()
        self._add_advanced_validators()

    def _add_cbx_options(self):
        """Setup combobox options on all comboboxes."""
        self._add_general_cbx_options()
        self._add_formulation_cbx_options()
        self._add_timing_cbx_options()
        self._add_output_cbx_options()
        self._add_wind_cbx_options()
        self._add_advanced_cbx_options()

    def _add_output_table(self):
        """Add the output table to the 'Output' tab."""
        has_stations = True if self.station_uuid else False
        has_wind = self.data.wind.attrs['NWS'] > 0
        df = self.data.output.to_dataframe()
        gui_util.fix_df_column_names_for_gui(df)
        self.output_table = OutputTableWidget(df, has_stations, has_wind, self.data.info.attrs['proj_dir'], self)
        layout = QVBoxLayout()
        layout.addWidget(self.output_table)
        self.ui.grp_output.setLayout(layout)

        self.ui.grp_netcdf.setTitle('View NetCDF options')
        self.ui.grp_netcdf.setChecked(False)

    def _add_harmonics_table(self):
        """Add the constituent table to the 'Harmonic Analysis' tab."""
        ref_date = self.data.timing.attrs['ref_date']
        if not ref_date:
            ref_date = self.global_time
        else:
            ref_date = datetime.datetime.strptime(ref_date, ISO_DATETIME_FORMAT)
        self.harmonics_table = HarmonicsTableWidget(self.data.harmonics.to_dataframe(), ref_date, self)
        layout = QVBoxLayout()
        layout.addWidget(self.harmonics_table)
        self.ui.grp_nfreq.setLayout(layout)
        size_policy = self.ui.grp_nfreq.sizePolicy()
        size_policy.setVerticalPolicy(QSizePolicy.MinimumExpanding)
        self.ui.grp_nfreq.setSizePolicy(size_policy)

    def _add_wind_tables(self):
        """Add the file selector tables to the 'Wind' tab for NWS=15,16."""
        default_10 = {
            'AVN File': '',
        }
        default_11 = {
            'ETA File': '',
        }
        default_12 = {
            'Pressure File': '',
            'Wind File': '',
        }
        default_15 = {
            'Hours': 0.0,
            'Central Pressure': 0.0,
            'Ramp Multiplier': 0.0,
            'HWND File': '',
        }
        default_16 = {
            'Hours': 0.0,
            'Ramp Multiplier': 0.0,
            'GFDL File': '',
        }
        self.nws10_table = WindFileTableWidget(self.data.nws10_files.to_dataframe(), self.data, default_10, self)
        self.nws11_table = WindFileTableWidget(self.data.nws11_files.to_dataframe(), self.data, default_11, self)
        self.nws15_table = WindFileTableWidget(self.data.nws15_files.to_dataframe(), self.data, default_15, self)
        self.nws16_table = WindFileTableWidget(self.data.nws16_files.to_dataframe(), self.data, default_16, self)
        self.nws12_table = WindOWIFileTableWidget(self.data.nws12_files.to_dataframe(), self.data, default_12, self)
        layout = QVBoxLayout()
        layout.addWidget(self.ui.lbl_nws12_note)  # Add the NWS=12 text label from UI file to layout first.
        layout.addWidget(self.nws10_table)
        layout.addWidget(self.nws11_table)
        layout.addWidget(self.nws12_table)
        layout.addWidget(self.nws15_table)
        layout.addWidget(self.nws16_table)
        self.ui.grp_file_tables.setLayout(layout)
        size_policy = self.ui.grp_file_tables.sizePolicy()
        size_policy.setVerticalPolicy(QSizePolicy.MinimumExpanding)
        self.ui.grp_file_tables.setSizePolicy(size_policy)

    def _add_general_validators(self):
        """Add int and double validators to edit field widgets for the 'General parameters' tab."""
        lon_validator = QxDoubleValidator(bottom=-360.0, top=360.0, parent=self)
        lat_validator = QxDoubleValidator(bottom=-90.0, top=90.0, parent=self)
        tan_ang_validator = QxDoubleValidator(bottom=90.0, top=150.0, parent=self)

        self.ui.edt_comp_proc.setValidator(self.int_pos_validator)
        self.ui.edt_io_proc.setValidator(self.int_noneg_validator)
        self.ui.edt_fric_coeff.setValidator(self.dbl_validator)
        self.ui.edt_hbreak.setValidator(self.dbl_validator)
        self.ui.edt_ftheta.setValidator(self.dbl_validator)
        self.ui.edt_fgamma.setValidator(self.dbl_validator)
        self.ui.edt_cori.setValidator(self.dbl_validator)
        self.ui.edt_slam0.setValidator(lon_validator)
        self.ui.edt_sfea0.setValidator(lat_validator)
        self.ui.edt_anginn.setValidator(tan_ang_validator)
        self.ui.edt_nscreen.setValidator(self.int_validator)

    def _add_formulation_validators(self):
        """Add int and double validators to edit field widgets for the 'Model formulation' tab."""
        h0_validator = QxDoubleValidator(bottom=0.01, top=10.0, parent=self)
        velmin_validator = QxDoubleValidator(bottom=0.01, top=1.0, parent=self)
        tau0_validator = QxDoubleValidator(bottom=0.001, top=0.1, parent=self)
        gwce_validator = QxDoubleValidator(bottom=0.0, top=1.0, parent=self)
        convcr_validator = QxDoubleValidator(bottom=0.0, top=0.001, parent=self)
        itmax_validator = QxIntValidator(parent=self)
        itmax_validator.setBottom(0)
        itmax_validator.setTop(100)

        self.ui.edt_eslm.setValidator(self.dbl_validator)
        self.ui.edt_h0.setValidator(h0_validator)
        self.ui.edt_velmin.setValidator(velmin_validator)
        self.ui.edt_tau0_specified.setValidator(tau0_validator)
        self.ui.edt_tau0_min.setValidator(tau0_validator)
        self.ui.edt_tau0_max.setValidator(tau0_validator)
        self.ui.edt_a00.setValidator(gwce_validator)
        self.ui.edt_b00.setValidator(gwce_validator)
        self.ui.edt_c00.setValidator(gwce_validator)
        self.ui.edt_convcr.setValidator(convcr_validator)
        self.ui.edt_itmax.setValidator(itmax_validator)

    def _add_timing_validators(self):
        """Add double validators to edit field widgets for the 'Timing' tab."""
        self.ui.edt_dtdp.setValidator(self.dbl_noneg_validator)
        self.ui.edt_runday.setValidator(self.dbl_noneg_validator)
        self.ui.edt_dramp.setValidator(self.dbl_validator)
        self.ui.edt_drampextflux.setValidator(self.dbl_validator)
        self.ui.edt_fluxsettlingtime.setValidator(self.dbl_validator)
        self.ui.edt_drampintflux.setValidator(self.dbl_validator)
        self.ui.edt_drampelev.setValidator(self.dbl_validator)
        self.ui.edt_dramptip.setValidator(self.dbl_validator)
        self.ui.edt_drampmete.setValidator(self.dbl_validator)
        self.ui.edt_drampwrad.setValidator(self.dbl_validator)
        self.ui.edt_dunrampmete.setValidator(self.dbl_validator)

    def _add_output_validators(self):
        """Add double validators to edit field widgets for the 'Output' tab."""
        self.ui.edt_nhsinc.setValidator(self.dbl_validator)

    def _add_wind_validators(self):
        """Add double validators to edit field widgets for the 'Wind' tab."""
        self.ui.edt_wtiminc.setValidator(self.dbl_validator)
        self.ui.edt_nwbs.setValidator(self.dbl_validator)
        self.ui.edt_dwm.setValidator(self.dbl_validator)
        self.ui.edt_max_extrap.setValidator(self.dbl_validator)
        self.ui.edt_rstiminc.setValidator(self.dbl_noneg_validator)
        self.ui.edt_cice_timinc.setValidator(self.dbl_validator)

    def _add_nodal_atts_validators(self):
        """Add double validators to edit field widgets for the 'Nodal attributes' tab."""
        offset_validator = QxDoubleValidator(bottom=-10.0, top=10.0, parent=self)
        self.ui.edt_geoidoffset.setValidator(offset_validator)

    def _add_harmonics_validators(self):
        """Add int and double validators to edit field widgets for the 'Harmonic analysis' tab."""
        fmv_validator = QxDoubleValidator(bottom=0.0, top=100.0, parent=self)
        self.ui.edt_thas.setValidator(self.dbl_validator)
        self.ui.edt_thaf.setValidator(self.dbl_validator)
        self.ui.edt_nhainc.setValidator(self.int_validator)
        self.ui.edt_fmv.setValidator(fmv_validator)

    def _add_advanced_validators(self):
        """Add int validators to edit field widgets for the 'Advanced' tab."""
        self.ui.edt_btiminc.setValidator(self.int_validator)
        self.ui.edt_bchgtiminc.setValidator(self.int_validator)

    def _add_general_cbx_options(self):
        """Set up the combobox options for the 'General parameters' tab."""
        self._add_cbx_opts(self.ui.cbx_hot_start)
        self._add_cbx_opts(self.ui.cbx_nolibf)
        self._add_cbx_opts(self.ui.cbx_ncor)
        self._add_cbx_opts(self.ui.cbx_nabout)

    def _add_formulation_cbx_options(self):
        """Set up the combobox options for the 'Model formulation' tab."""
        self._add_cbx_opts(self.ui.cbx_im)
        self._add_cbx_opts(self.ui.cbx_iden)
        self._add_cbx_opts(self.ui.cbx_nolifa)
        self._add_cbx_opts(self.ui.cbx_tau0)
        self._add_cbx_opts(self.ui.cbx_ititer)
        self._add_cbx_opts(self.ui.cbx_isldia)

    def _add_timing_cbx_options(self):
        """Set up the combobox options for the 'Timing' tab."""
        self._add_cbx_opts(self.ui.cbx_nramp)

    def _add_output_cbx_options(self):
        """Set up the combobox options for the 'Output' tab."""
        self._add_cbx_opts(self.ui.cbx_nhstar)

    def _add_wind_cbx_options(self):
        """Set up the combobox options for the 'Wind' tab."""
        self._add_cbx_opts(self.ui.cbx_nws)
        self._add_cbx_opts(self.ui.cbx_wind_pressure_relationship)
        self._add_cbx_opts(self.ui.cbx_geofactor)
        self._add_cbx_opts(self.ui.cbx_wave_radiation)

    def _add_advanced_cbx_options(self):
        """Set up the combobox options for the 'Advanced' tab."""
        self._add_cbx_opts(self.ui.cbx_nddt)

    """
    Slots
    """

    def _connect_slots(self):
        """Connect signals and slots for all widgets."""
        self._connect_general_slots()
        self._connect_formulation_slots()
        self._connect_timing_slots()
        self._connect_output_slots()
        self._connect_wind_slots()
        self._connect_nodal_atts_slots()
        self._connect_advanced_slots()

    def _connect_general_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'General parameters' tab."""
        self.ui.button_box.helpRequested.connect(self.help_requested)
        self.ui.cbx_hot_start.currentIndexChanged.connect(self.on_ihot_changed)
        self.ui.cbx_hot_start.currentIndexChanged.connect(self.on_ihot_changed)
        self.ui.btn_hot_start.clicked.connect(self.on_btn_hot_start)
        self.ui.tog_padcirc.stateChanged.connect(self.on_tog_padcirc)
        self.ui.cbx_nolibf.currentIndexChanged.connect(self.on_nolibf_changed)
        self.ui.btn_dset_nolibf.clicked.connect(self.on_btn_dset_nolibf)
        self.ui.cbx_ncor.currentIndexChanged.connect(self.on_ncor_changed)
        self.ui.tog_calc_coords.stateChanged.connect(self.on_tog_calc_coords)

    def _connect_formulation_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Model formulation' tab."""
        self.ui.cbx_im.currentIndexChanged.connect(self.on_im_changed)
        self.ui.tog_evc.stateChanged.connect(self.on_tog_evc)
        self.ui.btn_evc.clicked.connect(self.on_btn_evc)
        self.ui.cbx_nolifa.currentIndexChanged.connect(self.on_nolifa_changed)
        self.ui.tog_nolica.stateChanged.connect(self.on_tog_nolica)
        self.ui.tog_advection_nodal_att.stateChanged.connect(self.on_tog_advection_nodal_att)
        self.ui.btn_advection_nodal_att.clicked.connect(self.on_btn_advection_nodal_att)
        self.ui.cbx_tau0.currentIndexChanged.connect(self.on_tau0_changed)
        self.ui.btn_tau0_nodal_att.clicked.connect(self.on_btn_tau0_nodal_att)

    def _connect_timing_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Timing' tab."""
        self.ui.cbx_nramp.currentIndexChanged.connect(self.on_nramp_changed)
        self.ui.date_interp_ref.dateTimeChanged.connect(self.harmonics_table.on_ref_date_changed)

    def _connect_output_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Output' tab."""
        # Connect a slot to enable/disable meteorological output when the NWS option changes.
        self.ui.cbx_nhstar.currentIndexChanged.connect(self.on_nhstar_changed)

    def _connect_wind_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Wind' tab."""
        self.ui.cbx_nws.currentIndexChanged.connect(self.on_nws_changed)
        self.ui.tog_nws_file.stateChanged.connect(self.on_tog_nws_file)
        self.ui.btn_wind_pressure.clicked.connect(self.on_btn_wind_pressure)
        self.ui.btn_wind_stress.clicked.connect(self.on_btn_wind_stress)
        self.ui.btn_wind_grid_speed.clicked.connect(self.on_btn_wind_grid_speed)
        self.ui.btn_wind_grid_pressure.clicked.connect(self.on_btn_wind_grid_pressure)
        self.ui.cbx_wave_radiation.currentIndexChanged.connect(self.on_cbx_wave_radiation_changed)
        self.ui.btn_fort23_dset.clicked.connect(self.on_btn_fort23dset)
        self.ui.tog_z0land.stateChanged.connect(self.on_tog_z0land)
        self.ui.tog_vcanopy.stateChanged.connect(self.on_tog_vcanopy)
        self.ui.tog_use_ice.stateChanged.connect(self.on_tog_use_ice)

        # File selectors
        self.ui.btn_nws_file.clicked.connect(self.on_btn_nws_file)

        # Nodal attribute dataset selectors
        self.ui.btn_z0land_000.clicked.connect(self.on_btn_z0land_000)
        self.ui.btn_z0land_030.clicked.connect(self.on_btn_z0land_030)
        self.ui.btn_z0land_060.clicked.connect(self.on_btn_z0land_060)
        self.ui.btn_z0land_090.clicked.connect(self.on_btn_z0land_090)
        self.ui.btn_z0land_120.clicked.connect(self.on_btn_z0land_120)
        self.ui.btn_z0land_150.clicked.connect(self.on_btn_z0land_150)
        self.ui.btn_z0land_180.clicked.connect(self.on_btn_z0land_180)
        self.ui.btn_z0land_210.clicked.connect(self.on_btn_z0land_210)
        self.ui.btn_z0land_240.clicked.connect(self.on_btn_z0land_240)
        self.ui.btn_z0land_270.clicked.connect(self.on_btn_z0land_270)
        self.ui.btn_z0land_300.clicked.connect(self.on_btn_z0land_300)
        self.ui.btn_z0land_330.clicked.connect(self.on_btn_z0land_330)
        self.ui.btn_vcanopy.clicked.connect(self.on_btn_vcanopy)

    def _connect_nodal_atts_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Nodal attributes' tab."""
        self.ui.tog_startdry.stateChanged.connect(self.on_tog_startdry)
        self.ui.tog_geoidoffset.stateChanged.connect(self.on_tog_geoidoffset)
        self.ui.tog_swanwaverefrac.stateChanged.connect(self.on_tog_swanwaverefrac)
        self.ui.tog_bridge_pilings.stateChanged.connect(self.on_tog_bridge_pilings)
        self.ui.tog_slope_limiter.stateChanged.connect(self.on_tog_slope_limiter)
        self.ui.tog_river_elev.stateChanged.connect(self.on_tog_river_elev)
        self.ui.btn_startdry.clicked.connect(self.on_btn_startdry)
        self.ui.btn_swanwaverefrac.clicked.connect(self.on_btn_swanwaverefrac)
        self.ui.btn_bk.clicked.connect(self.on_btn_bk)
        self.ui.btn_balpha.clicked.connect(self.on_btn_balpha)
        self.ui.btn_bdelx.clicked.connect(self.on_btn_bdelx)
        self.ui.btn_poan.clicked.connect(self.on_btn_poan)
        self.ui.btn_slope_limiter.clicked.connect(self.on_btn_slope_limiter)
        self.ui.btn_river_elev.clicked.connect(self.on_btn_river_elev)

    def _connect_advanced_slots(self):
        """Set up the signal/slot connections to handle widget dependencies for the 'Advanced' tab."""
        self.ui.cbx_nddt.currentIndexChanged.connect(self.on_nddt_changed)

    def select_nodal_attribute(self, dlg_title, att_name, label_widget):
        """Display a dataset picker and store the new selection if one is made.

        Args:
            dlg_title (:obj:`str`): Title of the dataset selector dialog
            att_name (:obj:`str`): The ADCIRC parameter name for the nodal attribute being selected
            label_widget (:obj:`QLabel`): The label associated with the dataset selector push button
        """
        # Get the UUID of the domain mesh.
        if not self.domain_uuid:  # No geometry
            msg = QMessageBox(
                QMessageBox.Information, 'SMS', 'A mesh must be linked to the simulation to select a dataset.',
                QMessageBox.Ok, self
            )
            msg.exec()
            return

        previous_uuid = self.data.nodal_atts.attrs[att_name]
        tree_copy = TreeNode(other=self.domain_pe_tree)  # Create a copy of the project explorer tree to filter.
        if tree_copy:
            tr_util.filter_project_explorer(tree_copy, DatasetSelector.is_scalar_and_steady_state_if_dset)
        if not tree_copy.children:  # Have a geometry but no selectable datasets
            # I don't think this could ever happen because the Z dataset is always valid. This is really an assert.
            msg = QMessageBox(
                QMessageBox.Information, 'SMS', 'No selectable datasets found on the ADCIRC mesh. Nodal'
                'attribute datasets must be scalar and steady state.', QMessageBox.Ok, self
            )
            msg.exec()
            return

        dlg = TreeItemSelectorDlg(
            title=dlg_title,
            target_type=xmd.DatasetItem,
            pe_tree=tree_copy,
            previous_selection=previous_uuid,
            show_root=True,
            parent=self
        )
        if dlg.exec():
            dset_uuid = dlg.get_selected_item_uuid()
            if dset_uuid:
                self.data.nodal_atts.attrs[att_name] = dset_uuid
                dset_path = tr_util.build_tree_path(tree_copy, dset_uuid)
                label_widget.setText(dset_path)

    def select_wind_dataset(self, dlg_title, att_name, label_widget, on_mesh, is_scalar):
        """Display a dataset picker and store the new selection if one is made.

        Args:
            dlg_title (:obj:`str`): Title of the dataset selector dialog
            att_name (:obj:`str`): The ADCIRC parameter name for the nodal attribute being selected
            label_widget (:obj:`QLabel`): The label associated with the dataset selector push button
            on_mesh (:obj:`bool`): If True the dataset must be on the domain mesh. If false, must be on the wind grid.
            is_scalar (:obj:`bool`): True if the selected dataset must be scalar
        """
        # Get the UUID of the domain mesh.
        if on_mesh:
            geom_uuid = self.domain_uuid
        else:
            geom_uuid = self.grid_uuid
        if not geom_uuid:  # No geometry
            if on_mesh:
                err_msg = 'A domain mesh must be linked to the simulation to select a wind dataset.'
            else:
                err_msg = 'A wind grid must be linked to the simulation to select a wind dataset.'
            msg = QMessageBox(QMessageBox.Information, 'SMS', err_msg, QMessageBox.Ok, self)
            msg.exec()
            return

        # Create a deep copy of the project explorer tree to filter.
        filtered_tree = None
        creator = ProjectExplorerTreeCreator()
        if on_mesh and self.domain_pe_tree:
            filtered_tree = creator.copy(self.domain_pe_tree)
            if is_scalar:
                tr_util.filter_project_explorer(filtered_tree, DatasetSelector.is_scalar_if_dset)
            else:
                tr_util.filter_project_explorer(filtered_tree, DatasetSelector.is_vector_if_dset)
        elif not on_mesh and self.grid_pe_tree:
            filtered_tree = creator.copy(self.grid_pe_tree)
            if is_scalar:
                tr_util.filter_project_explorer(filtered_tree, DatasetSelector.is_scalar_if_dset)
            else:
                tr_util.filter_project_explorer(filtered_tree, DatasetSelector.is_vector_if_dset)
        if not filtered_tree:  # Have a geometry but no selectable datasets
            msg = QMessageBox(
                QMessageBox.Information, 'SMS', 'No selectable wind datasets found on the geometry.', QMessageBox.Ok,
                self
            )
            msg.exec()
            return

        previous_uuid = self.data.wind.attrs[att_name]
        dlg = TreeItemSelectorDlg(
            title=dlg_title,
            target_type=xmd.DatasetItem,
            pe_tree=filtered_tree,
            previous_selection=previous_uuid,
            show_root=True,
            parent=self
        )
        if dlg.exec():
            dset_uuid = dlg.get_selected_item_uuid()
            if dset_uuid:
                self.data.wind.attrs[att_name] = dset_uuid
                dset_path = tr_util.build_tree_path(filtered_tree, dset_uuid)
                label_widget.setText(dset_path)

    def accept(self):
        """Save dialog data on accepted."""
        self._save_data()
        super().accept()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def on_ihot_changed(self, index):
        """Called when the hot start combobox option is changed."""
        # Enable/disable the hot start file selector.
        enable = False
        if index > 0:  # 0 = cold start
            enable = True
        self.ui.layout_ihot_file.setVisible(enable)

    def on_btn_hot_start(self):
        """Called when the hot start file selector button is clicked."""
        gui_util.select_file(
            self, self.ui.lbl_hot_start_btn, 'Select a hot start file',
            'fort.17 | fort.67 | fort.68 | fort.67.nc | fort.68.nc', self.data.info.attrs['proj_dir']
        )

    def on_tog_padcirc(self, state):
        """Called when the run PADCIRC checkbox is toggled."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.lbl_comp_proc.setVisible(enable)
        self.ui.edt_comp_proc.setVisible(enable)
        self.ui.lbl_io_proc.setVisible(enable)
        self.ui.edt_io_proc.setVisible(enable)

    def on_nolibf_changed(self, index):
        """Called when the NOLIBF combobox option is changed."""
        enable_dset = False
        if index in [3, 4, 5, 6]:  # Nodal attribute types.
            enable_dset = True
        elif index == 0:  # Linear bottom friction
            self.ui.lbl_firc_coef_param.setText('TAU')
        else:  # 1 or 2 (Quadratic bottom friction or Hybrid nonlinear bottom friction)
            self.ui.lbl_firc_coef_param.setText('CF')
        # Hide/show dataset selectors
        self.ui.btn_dset_nolibf.setVisible(enable_dset)
        self.ui.lbl_nolibf_dset.setVisible(enable_dset)
        self.ui.lbl_fric_coef.setVisible(not enable_dset)
        self.ui.edt_fric_coeff.setVisible(not enable_dset)
        self.ui.lbl_firc_coef_param.setVisible(not enable_dset)
        # Hide/show edit fields
        is_hybrid = index == 2  # 2 = Hybrid nonlinear bottom friction
        self.ui.lbl_hbreak.setVisible(is_hybrid)
        self.ui.edt_hbreak.setVisible(is_hybrid)
        self.ui.lbl_hbreak_param.setVisible(is_hybrid)
        self.ui.lbl_ftheta.setVisible(is_hybrid)
        self.ui.edt_ftheta.setVisible(is_hybrid)
        self.ui.lbl_ftheta_param.setVisible(is_hybrid)
        self.ui.lbl_fgamma.setVisible(is_hybrid)
        self.ui.edt_fgamma.setVisible(is_hybrid)
        self.ui.lbl_fgamma_param.setVisible(is_hybrid)

    def on_btn_dset_nolibf(self):
        """Called when the NOLIBF nodal attribute dataset selector button is clicked."""
        nolibf = int(self.ui.cbx_nolibf.itemData(self.ui.cbx_nolibf.currentIndex()))
        if nolibf == 3:
            att_name = 'quadratic_friction_coefficient_at_sea_floor'
        elif nolibf == 4:
            att_name = 'mannings_n_at_sea_floor'
        elif nolibf == 5:
            att_name = 'chezy_friction_coefficient_at_sea_floor'
        else:
            att_name = 'bottom_roughness_length'
        self.select_nodal_attribute('Select a friction dataset', att_name, self.ui.lbl_nolibf_dset)

    def on_ncor_changed(self, index):
        """Called when the NCOR combobox option is changed."""
        enable_cori = False
        if index == 0:  # 0 = Constant
            enable_cori = True
        # Hide/show edit fields
        self.ui.lbl_cori.setVisible(enable_cori)
        self.ui.edt_cori.setVisible(enable_cori)
        self.ui.lbl_cori_param.setVisible(enable_cori)

    def on_tog_calc_coords(self, state):
        """Called when the calculate coordinates checkbox is toggled."""
        enable = False
        if state == Qt.Unchecked:
            enable = True
        self.ui.lbl_slam0.setVisible(enable)
        self.ui.edt_slam0.setVisible(enable)
        self.ui.lbl_slam0_param.setVisible(enable)
        self.ui.lbl_sfea0.setVisible(enable)
        self.ui.edt_sfea0.setVisible(enable)
        self.ui.lbl_sfea0_param.setVisible(enable)

    def on_im_changed(self, index):
        """Called when the NCOR combobox option is changed."""
        im = int(self.ui.cbx_im.itemData(index))
        const_time_weighting = False
        if im in [111112, 611112]:  # Barotropic 2DDI/Barotropic 3D using lumped GWCE
            const_time_weighting = True
        # Constant time weighting factors if using lumped GWCE.
        self.ui.grp_time_weighting_factors_const.setVisible(const_time_weighting)
        self.ui.grp_time_weighting_factors.setVisible(not const_time_weighting)

        show_form = False
        if im == 21:  # Baroclinic 3D
            show_form = True
        self.ui.layout_iden.setVisible(show_form)

        # Enable/disable Z0b_var NOLIBF option if not a 3D run
        is_3d = im not in [0, 111112]
        nolibf = int(self.ui.cbx_nolibf.itemData(self.ui.cbx_nolibf.currentIndex()))
        z0_bar_opt = 6  # Combobox index and data
        num_items = self.ui.cbx_nolibf.count()
        if is_3d:  # Add the Z0b_var nodal attribute option to NOLIBF combobox if not there.
            if num_items == z0_bar_opt:
                self.ui.cbx_nolibf.addItem(const.MC_CBX_OPTS['cbx_nolibf'][0][z0_bar_opt], z0_bar_opt)
        else:  # Remove Z0b_var option
            if num_items > z0_bar_opt:
                self.ui.cbx_nolibf.removeItem(z0_bar_opt)
            if nolibf == z0_bar_opt:  # 6 = Z0b_var nodal attribute. Not valid for 2D runs
                self.ui.cbx_nolibf.setCurrentIndex(0)  # Select the first option

    def on_tog_evc(self, state):
        """Called when the EVC nodal attribute checkbox is toggled."""
        use_nodal_att = False
        if state == Qt.Checked:
            use_nodal_att = True
        self.ui.layout_evc.setVisible(use_nodal_att)
        self.ui.layout_eslm.setVisible(not use_nodal_att)

    def on_btn_evc(self):
        """Called when the EVC dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select an EVC dataset', 'average_horizontal_eddy_viscosity_in_sea_water_wrt_depth', self.ui.lbl_evc
        )

    def on_nolifa_changed(self, index):
        """Called when the NOLIFA combobox option is changed."""
        nolifa = int(self.ui.cbx_nolifa.itemData(index))
        enable_min_vel = False
        if nolifa == 2:  # 2 = With wetting/drying
            enable_min_vel = True
        self.ui.lbl_velmin.setVisible(enable_min_vel)
        self.ui.edt_velmin.setVisible(enable_min_vel)
        self.ui.lbl_velmin_param.setVisible(enable_min_vel)
        # If NOLICA is enabled or NOLIFA is enabled NOLICAT is on
        nolica = self.ui.tog_nolica.checkState()
        enable_nolicat = nolica == Qt.Unchecked and nolifa == 0  # NOLIFA(0) = None
        self.ui.tog_nolicat.setVisible(enable_nolicat)
        self.ui.lbl_nolicat_note.setVisible(not enable_nolicat)

    def on_tog_nolica(self, state):
        """Called when the NOLICA checkbox is toggled."""
        # If NOLICA is enabled or NOLIFA is enabled NOLICAT is on
        nolifa = int(self.ui.cbx_nolifa.itemData(self.ui.cbx_nolifa.currentIndex()))
        enable_nolicat = False
        if state == Qt.Unchecked and nolifa == 0:  # NOLIFA(0) = None
            enable_nolicat = True
        self.ui.tog_nolicat.setVisible(enable_nolicat)
        self.ui.lbl_nolicat_note.setVisible(not enable_nolicat)

    def on_tog_advection_nodal_att(self, state):
        """Called when the advection state nodal attribute checkbox is toggled."""
        use_nodal_att = False
        if state == Qt.Checked:
            use_nodal_att = True
        self.ui.layout_advection_nodal_att.setVisible(use_nodal_att)

    def on_btn_advection_nodal_att(self):
        """Called when the advection state dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select an Advection State dataset', 'advection_state', self.ui.lbl_advection_nodal_att
        )

    def on_tau0_changed(self, index):
        """Called when the TAU0 combobox option is changed."""
        tau0 = int(self.ui.cbx_tau0.itemData(index))
        tau0_note = ''
        enable_specified = False
        enable_nodal_att = False
        enable_local = False
        if tau0 == 1:  # Specified
            tau0_note = const.TAU0_SPECIFIED_NOTE
            enable_specified = True
        elif tau0 == -1:  # Spatially discrete
            tau0_note = const.TAU0_DISCRETE_NOTE
        elif tau0 == -2:  # Spatially variable
            tau0_note = const.TAU0_VARIABLE_NOTE
        elif tau0 == -3:  # Nodal attribute
            tau0_note = const.TAU0_NODAL_NOTE
            enable_nodal_att = True
        elif tau0 == -5:  # Local friction
            enable_local = True
        if tau0_note:
            self.ui.lbl_tau0_guidance.setText(tau0_note)
            self.ui.lbl_tau0_guidance.setVisible(True)
        else:
            self.ui.lbl_tau0_guidance.setVisible(False)
        self.ui.layout_tau0_specified.setVisible(enable_specified)
        self.ui.layout_tau0_local.setVisible(enable_local)
        self.ui.layout_tau0_nodal_att.setVisible(enable_nodal_att)

    def on_btn_tau0_nodal_att(self):
        """Called when the TAU0 nodal attribute dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select an TAU0 dataset', 'primitive_weighting_in_continuity_equation', self.ui.lbl_tau0_nodal_att
        )

    def on_nramp_changed(self, index):
        """Called when the NRAMP combobox option is changed."""
        enable_1 = False
        enable_2 = False
        enable_3 = False
        enable_4 = False
        enable_5 = False
        enable_6 = False
        enable_7 = False
        enable_8 = False
        nramp = int(self.ui.cbx_nramp.itemData(index))
        if nramp > 0:
            enable_1 = True
        if nramp > 1:
            enable_2 = True
        if nramp > 2:
            enable_3 = True
        if nramp > 3:
            enable_4 = True
        if nramp > 4:
            enable_5 = True
        if nramp > 5:
            enable_6 = True
        if nramp > 6:
            enable_7 = True
        if nramp > 7:
            enable_8 = True
        # Hide/show edit fields and labels
        self.ui.lbl_dramp.setVisible(enable_1)
        self.ui.edt_dramp.setVisible(enable_1)
        self.ui.lbl_dramp_param.setVisible(enable_1)
        self.ui.lbl_drampextflux.setVisible(enable_2)
        self.ui.edt_drampextflux.setVisible(enable_2)
        self.ui.lbl_drampextflux_param.setVisible(enable_2)
        self.ui.lbl_fluxsettlingtime.setVisible(enable_2)
        self.ui.edt_fluxsettlingtime.setVisible(enable_2)
        self.ui.lbl_fluxsettlingtime_param.setVisible(enable_2)
        self.ui.lbl_drampintflux.setVisible(enable_3)
        self.ui.edt_drampintflux.setVisible(enable_3)
        self.ui.lbl_drampintflux_param.setVisible(enable_3)
        self.ui.lbl_drampelev.setVisible(enable_4)
        self.ui.edt_drampelev.setVisible(enable_4)
        self.ui.lbl_drampelev_param.setVisible(enable_4)
        self.ui.lbl_dramptip.setVisible(enable_5)
        self.ui.edt_dramptip.setVisible(enable_5)
        self.ui.lbl_dramptip_param.setVisible(enable_5)
        self.ui.lbl_drampmete.setVisible(enable_6)
        self.ui.edt_drampmete.setVisible(enable_6)
        self.ui.lbl_drampmete_param.setVisible(enable_6)
        self.ui.lbl_drampwrad.setVisible(enable_7)
        self.ui.edt_drampwrad.setVisible(enable_7)
        self.ui.lbl_drampwrad_param.setVisible(enable_7)
        self.ui.lbl_dunrampmete.setVisible(enable_8)
        self.ui.edt_dunrampmete.setVisible(enable_8)
        self.ui.lbl_dunrampmete_param.setVisible(enable_8)

    def on_nhstar_changed(self, index):
        """Called when the NHSTAR combobox option changes."""
        nhstar_format = int(self.ui.cbx_nhstar.itemData(index))
        enable = False if nhstar_format == 0 else True
        self.ui.lbl_nhsinc.setVisible(enable)
        self.ui.edt_nhsinc.setVisible(enable)
        self.ui.lbl_nhsinc_param.setVisible(enable)

    def on_nws_changed(self, index):
        """Called when the NWS combobox option is changed."""
        show_mesh = False
        show_grid = False
        show_files_params = False
        show_files_tables = False
        show_wind_track = False
        nws = int(self.ui.cbx_nws.itemData(index))
        have_wind = nws != 0
        show_existing_file = nws not in [0, 10, 11, 12, 15, 16]
        show_wtiminc = nws not in [0, 1, 8, 11, 15, 16, 19, 20]
        show_matches_hot_start = nws in [2, 4, 5, 12, 15, 16]

        # Show a warning if using NWS=1 because it is dumb.
        if nws == 1:
            self.ui.lbl_nws1_note.setVisible(True)
        else:
            self.ui.lbl_nws1_note.setVisible(False)

        if nws in [1, 2, 4, 5]:  # Datasets on the mesh
            show_mesh = True
            # Set the dataset selector text
            self.ui.lbl_wind_stress_param.setText('Wind stress:' if nws in [1, 2] else 'Wind velocity:')
        elif nws in [3, 6]:  # Datasets on a wind grid
            show_grid = True
            show_pressure = False
            if nws == 6:
                self.ui.lbl_wind_grid_speed_param.setText('Wind (X/Y):')
                show_pressure = True
            else:
                self.ui.lbl_wind_grid_speed_param.setText('Wind (speed/direction):')
            self.ui.lbl_wind_grid_pressure_param.setVisible(show_pressure)
            self.ui.btn_wind_grid_pressure.setVisible(show_pressure)
            self.ui.lbl_wind_grid_pressure.setVisible(show_pressure)
        elif nws in [8, 19, 20]:
            show_wind_track = True
            show_geofactor = False
            if nws == 20:
                show_geofactor = True
            self.ui.lbl_geofactor.setVisible(show_geofactor)
            self.ui.cbx_geofactor.setVisible(show_geofactor)
        elif nws in [10, 11, 12, 15, 16]:
            show_files_tables = True
            is_10 = nws == 10
            is_11 = nws == 11
            is_12 = nws == 12
            is_15 = nws == 15
            is_16 = nws == 16
            self.ui.lbl_dwm.setVisible(True)
            self.ui.edt_dwm.setVisible(True)

            # Hide/show specific table for NWS type
            self.nws10_table.setVisible(is_10)
            self.nws11_table.setVisible(is_11)
            self.nws12_table.setVisible(is_12)
            self.nws15_table.setVisible(is_15)
            self.nws16_table.setVisible(is_16)
            show_files_params = not is_10 and not is_11  # Show the file parameters group only if NWS=12,15,16

            # Hide/show NWS=12 specific widgets
            self.ui.lbl_nws12_note.setVisible(is_12)
            self.ui.lbl_nwbs.setVisible(is_12)
            self.ui.edt_nwbs.setVisible(is_12)
            self.ui.lbl_nwbs_param.setVisible(is_12)

            # Hide/show NWS=15,16 specific widgets
            self.ui.lbl_max_extrap.setVisible(is_16)
            self.ui.edt_max_extrap.setVisible(is_16)
            self.ui.lbl_max_extrap_param.setVisible(is_16)
            self.ui.lbl_wind_pressure_relationship.setVisible(is_15)
            self.ui.cbx_wind_pressure_relationship.setVisible(is_15)

        using_existing_file = self.ui.tog_nws_file.checkState() == Qt.Checked
        self.ui.grp_mesh_params.setVisible(show_mesh and not using_existing_file)
        self.ui.grp_grid_params.setVisible(show_grid and not using_existing_file)
        self.ui.grp_wind_track_params.setVisible(show_wind_track)
        self.ui.grp_file_params.setVisible(show_files_params)
        self.ui.grp_file_tables.setVisible(show_files_tables)
        self.ui.tog_nws_file.setVisible(show_existing_file)
        self.ui.layout_nws_file.setVisible(show_existing_file and using_existing_file)
        self.ui.tog_wind_hot_start.setVisible(show_matches_hot_start)
        self.ui.layout_wtiminc.setVisible(show_wtiminc)
        self.ui.grp_wave_radiation.setVisible(have_wind)
        self.ui.grp_wind_nodal_atts.setVisible(have_wind)
        self.ui.grp_wind_ice.setVisible(have_wind)

        # Tell the output table that the NWS option has changed
        self.output_table.on_nws_changed(nws)

    def on_tog_nws_file(self, state):
        """Called when the use existing fort.22 toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.layout_nws_file.setVisible(enable)
        nws = int(self.ui.cbx_nws.itemData(self.ui.cbx_nws.currentIndex()))
        show_mesh = show_grid = show_wind_track = False
        if nws in [1, 2, 4, 5]:
            show_mesh = True
        elif nws in [3, 6]:
            show_grid = True
        elif nws in [8, 19, 20]:
            show_wind_track = True
        self.ui.grp_mesh_params.setVisible(show_mesh and not enable)
        self.ui.grp_grid_params.setVisible(show_grid and not enable)
        self.ui.grp_wind_track_params.setVisible(show_wind_track)

    def on_btn_wind_pressure(self):
        """Called when the mesh pressure dataset selector button is clicked."""
        self.select_wind_dataset(
            'Select an atmospheric pressure dataset', 'mesh_pressure', self.ui.lbl_wind_pressure, True, True
        )

    def on_btn_wind_stress(self):
        """Called when the mesh wind velocity dataset selector button is clicked."""
        self.select_wind_dataset('Select a wind velocity dataset', 'mesh_wind', self.ui.lbl_wind_stress, True, False)

    def on_btn_wind_grid_pressure(self):
        """Called when the grid pressure dataset selector button is clicked."""
        self.select_wind_dataset(
            'Select an atmospheric pressure dataset', 'grid_pressure', self.ui.lbl_wind_grid_pressure, False, True
        )

    def on_btn_wind_grid_speed(self):
        """Called when the grid wind speed dataset selector button is clicked."""
        self.select_wind_dataset(
            'Select a wind velocity dataset', 'grid_wind', self.ui.lbl_wind_grid_speed, False, False
        )

    def on_cbx_wave_radiation_changed(self, index):
        """Called when the wave radiation combobox is changed."""
        wave_opt = int(self.ui.cbx_wave_radiation.itemData(index))
        enable_forcing_dset = wave_opt == 100
        enable_swan = wave_opt == 300
        enable_rstiminc = wave_opt != 0
        self.ui.tog_swan_hot_start.setVisible(enable_swan)
        self.ui.layout_rstiminc.setVisible(enable_rstiminc)
        self.ui.lbl_fort23_dset.setVisible(enable_forcing_dset)
        self.ui.btn_fort23_dset.setVisible(enable_forcing_dset)

    def on_btn_fort23dset(self):
        """Called when the fort23dset button is clicked."""
        self.select_wind_dataset(
            'Select a radiation stress dataset', 'fort23_uuid', self.ui.lbl_fort23_dset, True, False
        )

    def on_tog_z0land(self, state):
        """Called when the Z0Land nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.layout_z0land.setVisible(enable)

    def on_tog_vcanopy(self, state):
        """Called when the VCanopy nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.layout_vcanopy.setVisible(enable)

    def on_tog_use_ice(self, state):
        """Called when the use ice toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.layout_cice_timinc.setVisible(enable)

    def on_btn_nws_file(self):
        """Called when the existing NWS file selector button is clicked."""
        gui_util.select_file(
            self, self.ui.lbl_nws_file, 'Select a fort.22 file', 'fort.22', self.data.info.attrs['proj_dir']
        )

    def on_btn_z0land_000(self):
        """Called when the Z0Land 000 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (000) dataset', 'z0land_000', self.ui.lbl_z0land_000)

    def on_btn_z0land_030(self):
        """Called when the Z0Land 030 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (030) dataset', 'z0land_030', self.ui.lbl_z0land_030)

    def on_btn_z0land_060(self):
        """Called when the Z0Land 060 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (060) dataset', 'z0land_060', self.ui.lbl_z0land_060)

    def on_btn_z0land_090(self):
        """Called when the Z0Land 090 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (090) dataset', 'z0land_090', self.ui.lbl_z0land_090)

    def on_btn_z0land_120(self):
        """Called when the Z0Land 120 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (120) dataset', 'z0land_120', self.ui.lbl_z0land_120)

    def on_btn_z0land_150(self):
        """Called when the Z0Land 150 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (150) dataset', 'z0land_150', self.ui.lbl_z0land_150)

    def on_btn_z0land_180(self):
        """Called when the Z0Land 180 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (180) dataset', 'z0land_180', self.ui.lbl_z0land_180)

    def on_btn_z0land_210(self):
        """Called when the Z0Land 210 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (210) dataset', 'z0land_210', self.ui.lbl_z0land_210)

    def on_btn_z0land_240(self):
        """Called when the Z0Land 240 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (240) dataset', 'z0land_240', self.ui.lbl_z0land_240)

    def on_btn_z0land_270(self):
        """Called when the Z0Land 270 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (270) dataset', 'z0land_270', self.ui.lbl_z0land_270)

    def on_btn_z0land_300(self):
        """Called when the Z0Land 300 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (300) dataset', 'z0land_300', self.ui.lbl_z0land_300)

    def on_btn_z0land_330(self):
        """Called when the Z0Land 330 dataset selector button is clicked."""
        self.select_nodal_attribute('Select a Z0Land (330) dataset', 'z0land_330', self.ui.lbl_z0land_330)

    def on_btn_vcanopy(self):
        """Called when the VCanopy dataset selector button is clicked."""
        self.select_nodal_attribute('Select a VCanopy dataset', 'surface_canopy_coefficient', self.ui.lbl_vcanopy)

    def on_tog_startdry(self, state):
        """Called when the StartDry nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.btn_startdry.setVisible(enable)
        self.ui.lbl_startdry.setVisible(enable)

    def on_tog_geoidoffset(self, state):
        """Called when the GeoidOffset nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.edt_geoidoffset.setVisible(enable)
        self.ui.lbl_geoidoffset.setVisible(enable)

    def on_tog_swanwaverefrac(self, state):
        """Called when the GeoidOffset nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.btn_swanwaverefrac.setVisible(enable)
        self.ui.lbl_swanwaverefrac.setVisible(enable)

    def on_tog_bridge_pilings(self, state):
        """Called when the bridge pilings nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.lbl_bk_param.setVisible(enable)
        self.ui.btn_bk.setVisible(enable)
        self.ui.lbl_bk.setVisible(enable)
        self.ui.lbl_balpha_param.setVisible(enable)
        self.ui.btn_balpha.setVisible(enable)
        self.ui.lbl_balpha.setVisible(enable)
        self.ui.lbl_bdelx_param.setVisible(enable)
        self.ui.btn_bdelx.setVisible(enable)
        self.ui.lbl_bdelx.setVisible(enable)
        self.ui.lbl_poan_param.setVisible(enable)
        self.ui.btn_poan.setVisible(enable)
        self.ui.lbl_poan.setVisible(enable)

    def on_tog_slope_limiter(self, state):
        """Called when the elemental slope limiter nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.btn_slope_limiter.setVisible(enable)
        self.ui.lbl_slope_limiter.setVisible(enable)

    def on_tog_river_elev(self, state):
        """Called when the initial river elevation nodal attribute toggle is clicked."""
        enable = False
        if state == Qt.Checked:
            enable = True
        self.ui.btn_river_elev.setVisible(enable)
        self.ui.lbl_river_elev.setVisible(enable)

    def on_btn_startdry(self):
        """Called when the StartDry dataset selector button is clicked."""
        self.select_nodal_attribute('Select a StartDry dataset', 'surface_submergence_state', self.ui.lbl_startdry)

    def on_btn_swanwaverefrac(self):
        """Called when the SwanWaveRefrac dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select a SwanWaveRefrac dataset', 'wave_refraction_in_swan', self.ui.lbl_swanwaverefrac
        )

    def on_btn_bk(self):
        """Called when the BK bridge pilings dataset selector button is clicked."""
        self.select_nodal_attribute('Select a BK dataset', 'BK', self.ui.lbl_bk)

    def on_btn_balpha(self):
        """Called when the BAlpha bridge pilings dataset selector button is clicked."""
        self.select_nodal_attribute('Select a BAlpha dataset', 'BAlpha', self.ui.lbl_balpha)

    def on_btn_bdelx(self):
        """Called when the BDelX bridge pilings dataset selector button is clicked."""
        self.select_nodal_attribute('Select a BDelX dataset', 'BDelX', self.ui.lbl_bdelx)

    def on_btn_poan(self):
        """Called when the POAN bridge pilings dataset selector button is clicked."""
        self.select_nodal_attribute('Select a POAN dataset', 'POAN', self.ui.lbl_poan)

    def on_btn_slope_limiter(self):
        """Called when the elemental slope limiter dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select an Elemental Slope Limiter dataset', 'elemental_slope_limiter', self.ui.lbl_slope_limiter
        )

    def on_btn_river_elev(self):
        """Called when the initial river elevation dataset selector button is clicked."""
        self.select_nodal_attribute(
            'Select an Initial River Elevation dataset', 'initial_river_elevation', self.ui.lbl_river_elev
        )

    def on_nddt_changed(self, index):
        """Called when the NDDT combobox option changes."""
        enable = False if int(self.ui.cbx_nddt.itemData(index)) == 0 else True
        self.ui.tog_nddt_hot_start.setVisible(enable)
        self.ui.layout_nddt.setVisible(enable)

    """
    File I/O
    """

    def _load_data(self):
        """Populate widget values from persistent data."""
        self._load_general_parameters()
        self._load_formulation()
        self._load_timing()
        self._load_output()
        self._load_wind()
        self._load_nodal_atts()
        self._load_harmonics()
        self._load_advanced()
        self._load_registry_settings()

    def _load_general_parameters(self):
        """Populate widgets in the 'General parameters' tab from persistent data."""
        general_attrs = self.data.general.attrs
        self.ui.edt_rundes.setText(general_attrs['RUNDES'])
        self.ui.edt_runid.setText(general_attrs['RUNID'])
        self._set_cbx_opt_from_data(self.ui.cbx_hot_start, int(general_attrs['IHOT']))
        hot_start_file = str(general_attrs['IHOT_file'])
        if hot_start_file and self.data.does_file_exist(hot_start_file):
            self.ui.lbl_hot_start_btn.setText(hot_start_file)
        self.on_ihot_changed(self.ui.cbx_hot_start.currentIndex())
        self.ui.tog_padcirc.setCheckState(Qt.Checked if int(general_attrs['run_padcirc']) == 1 else Qt.Unchecked)
        self.ui.edt_comp_proc.setText(str(general_attrs['num_comp_proc']))
        self.ui.edt_io_proc.setText(str(general_attrs['num_io_proc']))
        self.on_tog_padcirc(self.ui.tog_padcirc.checkState())
        nolibf = int(general_attrs['NOLIBF'])
        self._set_cbx_opt_from_data(self.ui.cbx_nolibf, nolibf)
        self.ui.edt_fric_coeff.setText(str(general_attrs['CF']))
        self.ui.edt_hbreak.setText(str(general_attrs['HBREAK']))
        self.ui.edt_ftheta.setText(str(general_attrs['FTHETA']))
        self.ui.edt_fgamma.setText(str(general_attrs['FGAMMA']))
        if nolibf > 2:  # Nodal attribute variety
            if nolibf == 3:
                dset_uuid = self.data.nodal_atts.attrs['quadratic_friction_coefficient_at_sea_floor']
            elif nolibf == 4:
                dset_uuid = self.data.nodal_atts.attrs['mannings_n_at_sea_floor']
            elif nolibf == 5:
                dset_uuid = self.data.nodal_atts.attrs['chezy_friction_coefficient_at_sea_floor']
            else:
                dset_uuid = self.data.nodal_atts.attrs['chezy_friction_coefficient_at_sea_floor']
            if dset_uuid:  # The selected dataset still exists
                path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
                self.ui.lbl_nolibf_dset.setText(path if path else gui_util.NULL_SELECTION)
        self.on_nolibf_changed(self.ui.cbx_nolibf.currentIndex())
        self._set_cbx_opt_from_data(self.ui.cbx_ncor, int(general_attrs['NCOR']))
        self.ui.edt_cori.setText(str(general_attrs['CORI']))
        self.on_ncor_changed(self.ui.cbx_ncor.currentIndex())
        self.ui.edt_slam0.setText(str(general_attrs['SLAM0']))
        self.ui.edt_sfea0.setText(str(general_attrs['SFEA0']))
        self.ui.tog_calc_coords.setCheckState(
            Qt.Checked if int(general_attrs['calc_center_coords']) == 1 else Qt.Unchecked
        )
        self.on_tog_calc_coords(self.ui.tog_calc_coords.checkState())
        self.ui.edt_anginn.setText(str(general_attrs['ANGINN']))
        self.ui.tog_nfover.setCheckState(Qt.Checked if int(general_attrs['NFOVER']) == 1 else Qt.Unchecked)
        self._set_cbx_opt_from_data(self.ui.cbx_nabout, int(general_attrs['NABOUT']))
        self.ui.edt_nscreen.setText(str(general_attrs['NSCREEN']))

    def _load_formulation(self):
        """Populate widgets in the 'Model formulation' tab from persistent data."""
        formulation_attrs = self.data.formulation.attrs
        nodal_attrs = self.data.nodal_atts.attrs
        self._set_cbx_opt_from_data(self.ui.cbx_im, int(formulation_attrs['IM']))
        self._set_cbx_opt_from_data(self.ui.cbx_iden, int(formulation_attrs['IDEN']))
        evc_enabled = int(nodal_attrs['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on']) == 1
        self.ui.tog_evc.setCheckState(Qt.Checked if evc_enabled else Qt.Unchecked)
        if evc_enabled:
            evc_uuid = str(nodal_attrs['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth'])
            path = tr_util.build_tree_path(self.domain_pe_tree, evc_uuid)
            self.ui.lbl_evc.setText(path if path else gui_util.NULL_SELECTION)
        self.ui.edt_eslm.setText(str(formulation_attrs['ESLM']))
        self._set_cbx_opt_from_data(self.ui.cbx_nolifa, int(formulation_attrs['NOLIFA']))
        self.ui.edt_h0.setText(str(formulation_attrs['H0']))
        self.ui.edt_velmin.setText(str(formulation_attrs['VELMIN']))
        self.ui.tog_nolica.setCheckState(Qt.Checked if int(formulation_attrs['NOLICA']) == 1 else Qt.Unchecked)
        self.ui.tog_nolicat.setCheckState(Qt.Checked if int(formulation_attrs['NOLICAT']) == 1 else Qt.Unchecked)
        advection_nodal_att_enabled = int(nodal_attrs['advection_state_on']) == 1
        self.ui.tog_advection_nodal_att.setCheckState(Qt.Checked if advection_nodal_att_enabled else Qt.Unchecked)
        if advection_nodal_att_enabled:
            adv_uuid = str(nodal_attrs['advection_state'])
            path = tr_util.build_tree_path(self.domain_pe_tree, adv_uuid)
            self.ui.lbl_advection_nodal_att.setText(path if path else gui_util.NULL_SELECTION)
        tau0 = int(formulation_attrs['TAU0'])
        self._set_cbx_opt_from_data(self.ui.cbx_tau0, tau0)
        self.ui.edt_tau0_specified.setText(str(formulation_attrs['TAU0_specified']))
        self.ui.edt_tau0_min.setText(str(formulation_attrs['Tau0FullDomainMin']))
        self.ui.edt_tau0_max.setText(str(formulation_attrs['Tau0FullDomainMax']))
        if tau0 == -3:  # TAU0(-3) = nodal attribute
            tau0_uuid = str(nodal_attrs['primitive_weighting_in_continuity_equation'])
            path = tr_util.build_tree_path(self.domain_pe_tree, tau0_uuid)
            self.ui.lbl_tau0_nodal_att.setText(path if path else gui_util.NULL_SELECTION)
        self.ui.edt_a00.setText(str(formulation_attrs['A00']))
        self.ui.edt_b00.setText(str(formulation_attrs['B00']))
        self.ui.edt_c00.setText(str(formulation_attrs['C00']))
        self._set_cbx_opt_from_data(self.ui.cbx_ititer, int(formulation_attrs['ITITER']))
        self._set_cbx_opt_from_data(self.ui.cbx_isldia, int(formulation_attrs['ISLDIA']))
        self.ui.edt_convcr.setText(str(formulation_attrs['CONVCR']))
        self.ui.edt_itmax.setText(str(formulation_attrs['ITMAX']))

        self.on_im_changed(self.ui.cbx_im.currentIndex())
        self.on_tog_evc(self.ui.tog_evc.checkState())
        self.on_nolifa_changed(self.ui.cbx_nolifa.currentIndex())
        self.on_tog_advection_nodal_att(self.ui.tog_advection_nodal_att.checkState())
        self.on_tau0_changed(self.ui.cbx_tau0.currentIndex())

    def _load_timing(self):
        """Populate widgets in the 'Timing' tab."""
        timing_attrs = self.data.timing.attrs
        ref_date = timing_attrs['ref_date']
        if not ref_date:
            ref_date = datetime_to_qdatetime(self.global_time)
        else:
            ref_date = datetime_to_qdatetime(datetime.datetime.strptime(ref_date, ISO_DATETIME_FORMAT))
        self.ui.date_interp_ref.setDateTime(ref_date)
        self.ui.edt_dtdp.setText(str(timing_attrs['DTDP']))
        self.ui.edt_runday.setText(str(timing_attrs['RUNDAY']))
        self._set_cbx_opt_from_data(self.ui.cbx_nramp, int(timing_attrs['NRAMP']))
        self.ui.edt_dramp.setText(str(timing_attrs['DRAMP']))
        self.ui.edt_drampextflux.setText(str(timing_attrs['DRAMPExtFlux']))
        self.ui.edt_fluxsettlingtime.setText(str(timing_attrs['FluxSettlingTime']))
        self.ui.edt_drampintflux.setText(str(timing_attrs['DRAMPIntFlux']))
        self.ui.edt_drampelev.setText(str(timing_attrs['DRAMPElev']))
        self.ui.edt_dramptip.setText(str(timing_attrs['DRAMPTip']))
        self.ui.edt_drampmete.setText(str(timing_attrs['DRAMPMete']))
        self.ui.edt_drampwrad.setText(str(timing_attrs['DRAMPWRad']))
        self.ui.edt_dunrampmete.setText(str(timing_attrs['DUnRampMete']))
        self.on_nramp_changed(self.ui.cbx_nramp.currentIndex())

    def _load_output(self):
        """Populate widgets in the 'Output' tab."""
        output_attrs = self.data.output.attrs
        # Hot start output
        self._set_cbx_opt_from_data(self.ui.cbx_nhstar, int(output_attrs['NHSTAR']))
        self.ui.edt_nhsinc.setText(str(output_attrs['NHSINC']))
        self.on_nhstar_changed(self.ui.cbx_nhstar.currentIndex())
        # NetCDF options
        self.ui.edt_ncproj.setText(str(output_attrs['NCPROJ']))
        self.ui.edt_ncinst.setText(str(output_attrs['NCINST']))
        self.ui.edt_ncsour.setText(str(output_attrs['NCSOUR']))
        self.ui.edt_nchist.setText(str(output_attrs['NCHIST']))
        self.ui.edt_ncref.setText(str(output_attrs['NCREF']))
        self.ui.edt_nccom.setText(str(output_attrs['NCCOM']))
        self.ui.edt_nchost.setText(str(output_attrs['NCHOST']))
        self.ui.edt_ncconv.setText(str(output_attrs['NCCONV']))
        self.ui.edt_nccont.setText(str(output_attrs['NCCONT']))

    def _load_wind(self):
        """Populate widgets in the 'Wind' tab."""
        wind_attrs = self.data.wind.attrs
        self._set_cbx_opt_from_data(self.ui.cbx_nws, int(wind_attrs['NWS']))
        self.ui.tog_nws_file.setCheckState(Qt.Checked if int(wind_attrs['use_existing']) == 1 else Qt.Unchecked)
        existing_file = str(wind_attrs['existing_file'])
        if existing_file and self.data.does_file_exist(existing_file):
            self.ui.lbl_nws_file.setText(existing_file)
        self.ui.tog_wind_hot_start.setCheckState(
            Qt.Checked if int(wind_attrs['first_matches_hot_start']) == 1 else Qt.Unchecked
        )
        self.ui.edt_wtiminc.setText(str(wind_attrs['WTIMINC']))

        # Dataset selectors
        pressure_uuid = str(wind_attrs['mesh_pressure'])
        path = tr_util.build_tree_path(self.domain_pe_tree, pressure_uuid)
        self.ui.lbl_wind_pressure.setText(path if path else gui_util.NULL_SELECTION)
        wind_uuid = str(wind_attrs['mesh_wind'])
        path = tr_util.build_tree_path(self.domain_pe_tree, wind_uuid)
        self.ui.lbl_wind_stress.setText(path if path else gui_util.NULL_SELECTION)
        pressure_uuid = str(wind_attrs['grid_pressure'])
        path = tr_util.build_tree_path(self.grid_pe_tree, pressure_uuid)
        self.ui.lbl_wind_grid_pressure.setText(path if path else gui_util.NULL_SELECTION)
        wind_uuid = str(wind_attrs['grid_wind'])
        path = tr_util.build_tree_path(self.grid_pe_tree, wind_uuid)
        self.ui.lbl_wind_grid_speed.setText(path if path else gui_util.NULL_SELECTION)
        wind_uuid = str(wind_attrs['fort23_uuid'])
        path = tr_util.build_tree_path(self.domain_pe_tree, wind_uuid)
        self.ui.lbl_fort23_dset.setText(path if path else gui_util.NULL_SELECTION)

        self.ui.edt_nwbs.setText(str(wind_attrs['NWBS']))
        self.ui.edt_dwm.setText(str(wind_attrs['DWM']))
        self.ui.edt_max_extrap.setText(str(wind_attrs['max_extrap']))
        self._set_cbx_opt_from_data(self.ui.cbx_wind_pressure_relationship, str(wind_attrs['wind_pressure']))

        storm_start = str(wind_attrs['storm_start'])
        if not storm_start:
            storm_start = datetime_to_qdatetime(self.global_time)
        else:
            storm_start = datetime_to_qdatetime(datetime.datetime.strptime(storm_start, ISO_DATETIME_FORMAT))
        self.ui.date_storm_start.setDateTime(storm_start)
        self.ui.edt_bladj.setText(str(wind_attrs['bladj']))
        self._set_cbx_opt_from_data(self.ui.cbx_geofactor, int(wind_attrs['geofactor']))

        self._set_cbx_opt_from_data(self.ui.cbx_wave_radiation, int(wind_attrs['wave_radiation']))
        self.ui.tog_swan_hot_start.setCheckState(
            Qt.Checked if int(wind_attrs['swan_hot_start']) == -1 else Qt.Unchecked
        )

        # Wind nodal attribute datasets.
        nodal_atts = self.data.nodal_atts.attrs
        z0land_on = int(nodal_atts['surface_directional_effective_roughness_length_on']) == 1
        self.ui.tog_z0land.setCheckState(Qt.Checked if z0land_on else Qt.Unchecked)
        if z0land_on:
            dset_uuid = str(nodal_atts['z0land_000'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_000.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_030'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_030.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_060'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_060.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_090'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_090.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_120'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_120.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_150'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_150.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_180'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_180.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_210'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_210.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_240'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_240.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_270'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_270.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_300'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_300.setText(path if path else gui_util.NULL_SELECTION)
            dset_uuid = str(nodal_atts['z0land_330'])
            path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
            self.ui.lbl_z0land_330.setText(path if path else gui_util.NULL_SELECTION)
        self.ui.tog_vcanopy.setCheckState(
            Qt.Checked if int(nodal_atts['surface_canopy_coefficient_on']) == 1 else Qt.Unchecked
        )
        dset_uuid = str(nodal_atts['surface_canopy_coefficient'])
        path = tr_util.build_tree_path(self.domain_pe_tree, dset_uuid)
        self.ui.lbl_vcanopy.setText(path if path else gui_util.NULL_SELECTION)

        self.ui.edt_rstiminc.setText(str(wind_attrs['RSTIMINC']))
        self.ui.tog_use_ice.setCheckState(Qt.Checked if int(wind_attrs['use_ice']) == 1 else Qt.Unchecked)
        self.ui.edt_cice_timinc.setText(str(wind_attrs['CICE_TIMINC']))

        self.on_tog_nws_file(self.ui.tog_nws_file.checkState())
        self.on_nws_changed(self.ui.cbx_nws.currentIndex())
        self.on_cbx_wave_radiation_changed(self.ui.cbx_wave_radiation.currentIndex())
        self.on_tog_z0land(self.ui.tog_z0land.checkState())
        self.on_tog_vcanopy(self.ui.tog_vcanopy.checkState())
        self.on_tog_use_ice(self.ui.tog_use_ice.checkState())

    def _load_nodal_atts(self):
        """Populate widgets in the 'Nodal attributes' tab from persistent data."""
        nodal_attrs = self.data.nodal_atts.attrs
        startdry_enabled = int(nodal_attrs['surface_submergence_state_on'])
        self.ui.tog_startdry.setCheckState(Qt.Checked if startdry_enabled == 1 else Qt.Unchecked)
        if startdry_enabled:  # Set the tree path in the label
            startdry_uuid = str(nodal_attrs['surface_submergence_state'])
            path = tr_util.build_tree_path(self.domain_pe_tree, startdry_uuid)
            self.ui.lbl_startdry.setText(path if path else gui_util.NULL_SELECTION)
        self.on_tog_startdry(self.ui.tog_startdry.checkState())
        offset_on = int(nodal_attrs['sea_surface_height_above_geoid_on'])
        self.ui.tog_geoidoffset.setCheckState(Qt.Checked if offset_on == 1 else Qt.Unchecked)
        self.ui.edt_geoidoffset.setText(str(nodal_attrs['sea_surface_height_above_geoid']))
        self.on_tog_geoidoffset(self.ui.tog_geoidoffset.checkState())
        swanrefrac_enabled = int(nodal_attrs['wave_refraction_in_swan_on'])
        self.ui.tog_swanwaverefrac.setCheckState(Qt.Checked if swanrefrac_enabled == 1 else Qt.Unchecked)
        if swanrefrac_enabled:  # Set the tree path in the label
            swanrefrac_uuid = str(nodal_attrs['wave_refraction_in_swan'])
            path = tr_util.build_tree_path(self.domain_pe_tree, swanrefrac_uuid)
            self.ui.lbl_swanwaverefrac.setText(path if path else gui_util.NULL_SELECTION)
        self.on_tog_swanwaverefrac(self.ui.tog_swanwaverefrac.checkState())
        bridge_pilings_enabled = int(nodal_attrs['bridge_pilings_friction_paramenters_on'])
        self.ui.tog_bridge_pilings.setCheckState(Qt.Checked if bridge_pilings_enabled == 1 else Qt.Unchecked)
        if bridge_pilings_enabled:  # Set the tree path in the label
            bk_uuid = str(nodal_attrs['BK'])
            path = tr_util.build_tree_path(self.domain_pe_tree, bk_uuid)
            self.ui.lbl_bk.setText(path if path else gui_util.NULL_SELECTION)
            balpha_uuid = str(nodal_attrs['BAlpha'])
            path = tr_util.build_tree_path(self.domain_pe_tree, balpha_uuid)
            self.ui.lbl_balpha.setText(path if path else gui_util.NULL_SELECTION)
            bdelx_uuid = str(nodal_attrs['BDelX'])
            path = tr_util.build_tree_path(self.domain_pe_tree, bdelx_uuid)
            self.ui.lbl_bdelx.setText(path if path else gui_util.NULL_SELECTION)
            poan_uuid = str(nodal_attrs['POAN'])
            path = tr_util.build_tree_path(self.domain_pe_tree, poan_uuid)
            self.ui.lbl_poan.setText(path if path else gui_util.NULL_SELECTION)
        self.on_tog_bridge_pilings(self.ui.tog_bridge_pilings.checkState())
        slope_limiter_enabled = int(nodal_attrs['elemental_slope_limiter_on'])
        self.ui.tog_slope_limiter.setCheckState(Qt.Checked if slope_limiter_enabled == 1 else Qt.Unchecked)
        if slope_limiter_enabled:  # Set the tree path in the label
            slope_limiter_uuid = str(nodal_attrs['elemental_slope_limiter'])
            path = tr_util.build_tree_path(self.domain_pe_tree, slope_limiter_uuid)
            self.ui.lbl_slope_limiter.setText(path if path else gui_util.NULL_SELECTION)
        self.on_tog_slope_limiter(self.ui.tog_slope_limiter.checkState())
        river_elev_enabled = int(nodal_attrs['initial_river_elevation_on'])
        self.ui.tog_river_elev.setCheckState(Qt.Checked if river_elev_enabled == 1 else Qt.Unchecked)
        if river_elev_enabled:  # Set the tree path in the label
            river_elev_uuid = str(nodal_attrs['initial_river_elevation'])
            path = tr_util.build_tree_path(self.domain_pe_tree, river_elev_uuid)
            self.ui.lbl_river_elev.setText(path if path else gui_util.NULL_SELECTION)
        self.on_tog_river_elev(self.ui.tog_river_elev.checkState())

    def _load_harmonics(self):
        """Populate widgets in the 'Harmonic analysis' tab from persistent data."""
        harmonics_attrs = self.data.harmonics.attrs
        self.ui.edt_thas.setText(str(harmonics_attrs['THAS']))
        self.ui.edt_thaf.setText(str(harmonics_attrs['THAF']))
        self.ui.edt_nhainc.setText(str(harmonics_attrs['NHAINC']))
        self.ui.edt_fmv.setText(str(harmonics_attrs['FMV']))
        self.ui.tog_nhase.setCheckState(Qt.Checked if int(harmonics_attrs['NHASE']) == 1 else Qt.Unchecked)
        self.ui.tog_nhasv.setCheckState(Qt.Checked if int(harmonics_attrs['NHASV']) == 1 else Qt.Unchecked)
        self.ui.tog_nhage.setCheckState(Qt.Checked if int(harmonics_attrs['NHAGE']) == 1 else Qt.Unchecked)
        self.ui.tog_nhagv.setCheckState(Qt.Checked if int(harmonics_attrs['NHAGV']) == 1 else Qt.Unchecked)

    def _load_advanced(self):
        """Populate widgets in the 'Advanced' tab."""
        advanced_attrs = self.data.advanced.attrs
        self._set_cbx_opt_from_data(self.ui.cbx_nddt, int(advanced_attrs['NDDT']))
        self.ui.tog_nddt_hot_start.setCheckState(
            Qt.Unchecked if int(advanced_attrs['NDDT_hot_start']) == 1 else Qt.Checked
        )  # -1 if checked, 1 if unchecked. Multiplier for NDDT
        self.ui.edt_btiminc.setText(str(advanced_attrs['BTIMINC']))
        self.ui.edt_bchgtiminc.setText(str(advanced_attrs['BCHGTIMINC']))
        self.on_nddt_changed(self.ui.cbx_nddt.currentIndex())

    def _load_registry_settings(self):
        """Read settings from registry for suppressing error messages."""
        settings = SettingsManager()
        self.ui.tog_suppress_missing_hotstart.setChecked(
            settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_HOTSTART, 0) == 1
        )
        self.ui.tog_suppress_missing_wind.setChecked(
            settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_WIND_FILE, 0) == 1
        )
        self.ui.tog_suppress_missing_nonperiodic.setChecked(
            settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_NONPERIODIC, 0) == 1
        )

    def _save_data(self):
        """Saves the dialog data to the SimData.

        Does not flush data to disk, just updates in memory xarray.Datasets. Call flush() on the
        SimData to write to disk.
        """
        self._save_general_parameters()
        self._save_formulation()
        self._save_timing()
        self._save_output()
        self._save_wind()
        self._save_nodal_atts()
        self._save_harmonics()
        self._save_advanced()
        self._save_registry_settings()

    def _save_general_parameters(self):
        """Save data from the 'General parameters' tab to in-memory Datasets."""
        general_attrs = self.data.general.attrs
        general_attrs['RUNDES'] = self.ui.edt_rundes.text()
        general_attrs['RUNID'] = self.ui.edt_runid.text()
        general_attrs['IHOT'] = self.ui.cbx_hot_start.itemData(self.ui.cbx_hot_start.currentIndex())
        ihot_file = self.ui.lbl_hot_start_btn.text()
        if ihot_file != gui_util.NULL_SELECTION:
            general_attrs['IHOT_file'] = ihot_file
        general_attrs['run_padcirc'] = 1 if self.ui.tog_padcirc.checkState() == Qt.Checked else 0
        general_attrs['num_comp_proc'] = int(self.ui.edt_comp_proc.text())
        general_attrs['num_io_proc'] = int(self.ui.edt_io_proc.text())
        general_attrs['NOLIBF'] = int(self.ui.cbx_nolibf.itemData(self.ui.cbx_nolibf.currentIndex()))
        general_attrs['CF'] = float(self.ui.edt_fric_coeff.text())
        general_attrs['HBREAK'] = float(self.ui.edt_hbreak.text())
        general_attrs['FTHETA'] = float(self.ui.edt_ftheta.text())
        general_attrs['FGAMMA'] = float(self.ui.edt_fgamma.text())
        general_attrs['NCOR'] = self.ui.cbx_ncor.itemData(self.ui.cbx_ncor.currentIndex())
        general_attrs['CORI'] = float(self.ui.edt_cori.text())
        general_attrs['SLAM0'] = float(self.ui.edt_slam0.text())
        general_attrs['SFEA0'] = float(self.ui.edt_sfea0.text())
        general_attrs['calc_center_coords'] = int(1 if self.ui.tog_calc_coords.checkState() == Qt.Checked else 0)
        general_attrs['ANGINN'] = float(self.ui.edt_anginn.text())
        general_attrs['NFOVER'] = int(1 if self.ui.tog_nfover.checkState() == Qt.Checked else 0)
        general_attrs['NABOUT'] = self.ui.cbx_nabout.itemData(self.ui.cbx_nabout.currentIndex())
        general_attrs['NSCREEN'] = int(self.ui.edt_nscreen.text())

    def _save_formulation(self):
        """Save data from the 'Model formulation' tab to in-memory Datasets."""
        formulation_attrs = self.data.formulation.attrs
        nodal_attrs = self.data.nodal_atts.attrs
        im = self.ui.cbx_im.itemData(self.ui.cbx_im.currentIndex())
        formulation_attrs['IM'] = im
        formulation_attrs['IDEN'] = self.ui.cbx_iden.itemData(self.ui.cbx_iden.currentIndex())
        nodal_attrs['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on'] = (
            1 if self.ui.tog_evc.checkState() == Qt.Checked else 0
        )
        formulation_attrs['ESLM'] = float(self.ui.edt_eslm.text())
        formulation_attrs['NOLIFA'] = self.ui.cbx_nolifa.itemData(self.ui.cbx_nolifa.currentIndex())
        formulation_attrs['H0'] = float(self.ui.edt_h0.text())
        formulation_attrs['VELMIN'] = float(self.ui.edt_velmin.text())
        formulation_attrs['NOLICA'] = 1 if self.ui.tog_nolica.checkState() == Qt.Checked else 0
        # NOLICAT is on if explicitly checked, NOLICA toggle is on, or NOLICA option is enabled
        nolicat_on = self.ui.tog_nolicat.checkState() == Qt.Checked or formulation_attrs['NOLIFA'] != 0 \
            or formulation_attrs['NOLICA'] == 1
        formulation_attrs['NOLICAT'] = 1 if nolicat_on else 0
        nodal_attrs['advection_state_on'] = 1 if self.ui.tog_advection_nodal_att.checkState() == Qt.Checked else 0
        formulation_attrs['TAU0'] = self.ui.cbx_tau0.itemData(self.ui.cbx_tau0.currentIndex())
        formulation_attrs['TAU0_specified'] = float(self.ui.edt_tau0_specified.text())
        formulation_attrs['Tau0FullDomainMin'] = float(self.ui.edt_tau0_min.text())
        formulation_attrs['Tau0FullDomainMax'] = float(self.ui.edt_tau0_max.text())
        # Time weighting factors are not specified lumped GWCE is being used
        a00 = float(self.ui.edt_a00.text())
        b00 = float(self.ui.edt_b00.text())
        c00 = float(self.ui.edt_c00.text())
        if im in [111112, 611112]:
            a00 = 0.0
            b00 = 1.0
            c00 = 0.0
        formulation_attrs['A00'] = a00
        formulation_attrs['B00'] = b00
        formulation_attrs['C00'] = c00
        formulation_attrs['ITITER'] = self.ui.cbx_ititer.itemData(self.ui.cbx_ititer.currentIndex())
        formulation_attrs['ISLDIA'] = self.ui.cbx_isldia.itemData(self.ui.cbx_isldia.currentIndex())
        formulation_attrs['CONVCR'] = float(self.ui.edt_convcr.text())
        formulation_attrs['ITMAX'] = int(self.ui.edt_itmax.text())

    def _save_timing(self):
        """Save data from the 'Timing' tab to in-memory Datasets."""
        timing_attrs = self.data.timing.attrs
        timing_attrs['ref_date'] = qdatetime_to_datetime(self.ui.date_interp_ref.dateTime()
                                                         ).strftime(ISO_DATETIME_FORMAT)
        timing_attrs['DTDP'] = float(self.ui.edt_dtdp.text())
        timing_attrs['RUNDAY'] = float(self.ui.edt_runday.text())
        timing_attrs['NRAMP'] = self.ui.cbx_nramp.itemData(self.ui.cbx_nramp.currentIndex())
        timing_attrs['DRAMP'] = float(self.ui.edt_dramp.text())
        timing_attrs['DRAMPExtFlux'] = float(self.ui.edt_drampextflux.text())
        timing_attrs['FluxSettlingTime'] = float(self.ui.edt_fluxsettlingtime.text())
        timing_attrs['DRAMPIntFlux'] = float(self.ui.edt_drampintflux.text())
        timing_attrs['DRAMPElev'] = float(self.ui.edt_drampelev.text())
        timing_attrs['DRAMPTip'] = float(self.ui.edt_dramptip.text())
        timing_attrs['DRAMPMete'] = float(self.ui.edt_drampmete.text())
        timing_attrs['DRAMPWRad'] = float(self.ui.edt_drampwrad.text())
        timing_attrs['DUnRampMete'] = float(self.ui.edt_dunrampmete.text())

    def _save_wind(self):
        """Save data from the 'Wind' tab to in-memory Datasets."""
        self.data.nws10_files = self.nws10_table.model.data_frame.to_xarray()
        self.data.nws11_files = self.nws11_table.model.data_frame.to_xarray()
        self.data.nws12_files = self.nws12_table.model.data_frame.to_xarray()
        self.data.nws15_files = self.nws15_table.model.data_frame.to_xarray()
        self.data.nws16_files = self.nws16_table.model.data_frame.to_xarray()
        wind_attrs = self.data.wind.attrs
        wind_attrs['NWS'] = self.ui.cbx_nws.itemData(self.ui.cbx_nws.currentIndex())
        wind_attrs['use_existing'] = 1 if self.ui.tog_nws_file.checkState() == Qt.Checked else 0
        existing_file = self.ui.lbl_nws_file.text()
        wind_attrs['existing_file'] = existing_file if existing_file != gui_util.NULL_SELECTION else ''
        wind_attrs['first_matches_hot_start'] = 1 if self.ui.tog_wind_hot_start.checkState() == Qt.Checked else 0
        wind_attrs['WTIMINC'] = float(self.ui.edt_wtiminc.text())

        wind_attrs['NWBS'] = float(self.ui.edt_nwbs.text())
        wind_attrs['DWM'] = float(self.ui.edt_dwm.text())
        wind_attrs['max_extrap'] = float(self.ui.edt_max_extrap.text())
        wind_attrs['wind_pressure'] = self.ui.cbx_wind_pressure_relationship.itemData(
            self.ui.cbx_wind_pressure_relationship.currentIndex()
        )

        wind_attrs['storm_start'] = qdatetime_to_datetime(self.ui.date_storm_start.dateTime()
                                                          ).strftime(ISO_DATETIME_FORMAT)
        wind_attrs['bladj'] = float(self.ui.edt_bladj.text())
        wind_attrs['geofactor'] = self.ui.cbx_geofactor.itemData(self.ui.cbx_geofactor.currentIndex())

        wind_attrs['wave_radiation'] = self.ui.cbx_wave_radiation.itemData(self.ui.cbx_wave_radiation.currentIndex())
        wind_attrs['swan_hot_start'] = -1 if self.ui.tog_swan_hot_start.checkState() == Qt.Checked else 1
        wind_attrs['RSTIMINC'] = float(self.ui.edt_rstiminc.text())

        wind_attrs['use_ice'] = 1 if self.ui.tog_use_ice.checkState() == Qt.Checked else 0
        wind_attrs['CICE_TIMINC'] = float(self.ui.edt_cice_timinc.text())

        nodal_attrs = self.data.nodal_atts.attrs
        nodal_attrs['surface_directional_effective_roughness_length_on'] = (
            1 if self.ui.tog_z0land.checkState() == Qt.Checked else 0
        )
        nodal_attrs['surface_canopy_coefficient_on'] = (1 if self.ui.tog_vcanopy.checkState() == Qt.Checked else 0)

    def _save_output(self):
        """Save data from the 'Output' tab to in-memory Datasets."""
        df = self.output_table.model.data_frame
        gui_util.fix_df_column_names_for_io(df)
        dset = df.to_xarray()
        dset.attrs['NHSTAR'] = self.ui.cbx_nhstar.itemData(self.ui.cbx_nhstar.currentIndex())
        dset.attrs['NHSINC'] = float(self.ui.edt_nhsinc.text())
        dset.attrs['NCPROJ'] = self.ui.edt_ncproj.text()
        dset.attrs['NCINST'] = self.ui.edt_ncinst.text()
        dset.attrs['NCSOUR'] = self.ui.edt_ncsour.text()
        dset.attrs['NCHIST'] = self.ui.edt_nchist.text()
        dset.attrs['NCREF'] = self.ui.edt_ncref.text()
        dset.attrs['NCCOM'] = self.ui.edt_nccom.text()
        dset.attrs['NCHOST'] = self.ui.edt_nchost.text()
        dset.attrs['NCCONV'] = self.ui.edt_ncconv.text()
        dset.attrs['NCCONT'] = self.ui.edt_nccont.text()
        self.data.output = dset

    def _save_nodal_atts(self):
        """Save data from the 'Nodal attributes' tab to in-memory Datasets."""
        nodal_atts = self.data.nodal_atts.attrs
        nodal_atts['surface_submergence_state_on'] = int(1 if self.ui.tog_startdry.checkState() == Qt.Checked else 0)
        nodal_atts['sea_surface_height_above_geoid_on'] = int(
            1 if self.ui.tog_geoidoffset.checkState() == Qt.Checked else 0
        )
        nodal_atts['sea_surface_height_above_geoid'] = float(self.ui.edt_geoidoffset.text())
        nodal_atts['wave_refraction_in_swan_on'] = int(
            1 if self.ui.tog_swanwaverefrac.checkState() == Qt.Checked else 0
        )
        nodal_atts['bridge_pilings_friction_paramenters_on'] = int(
            1 if self.ui.tog_bridge_pilings.checkState() == Qt.Checked else 0
        )
        nodal_atts['elemental_slope_limiter_on'] = int(1 if self.ui.tog_slope_limiter.checkState() == Qt.Checked else 0)
        nodal_atts['initial_river_elevation_on'] = int(1 if self.ui.tog_river_elev.checkState() == Qt.Checked else 0)

    def _save_harmonics(self):
        """Save data from the 'Nodal attributes' tab to in-memory Datasets."""
        dset = self.harmonics_table.model.data_frame.to_xarray()
        dset.attrs['THAS'] = float(self.ui.edt_thas.text())
        dset.attrs['THAF'] = float(self.ui.edt_thaf.text())
        dset.attrs['NHAINC'] = int(self.ui.edt_nhainc.text())
        dset.attrs['FMV'] = float(self.ui.edt_fmv.text())
        dset.attrs['NHASE'] = int(1 if self.ui.tog_nhase.checkState() == Qt.Checked else 0)
        dset.attrs['NHASV'] = int(1 if self.ui.tog_nhasv.checkState() == Qt.Checked else 0)
        dset.attrs['NHAGE'] = int(1 if self.ui.tog_nhage.checkState() == Qt.Checked else 0)
        dset.attrs['NHAGV'] = int(1 if self.ui.tog_nhagv.checkState() == Qt.Checked else 0)
        self.data.harmonics = dset

    def _save_advanced(self):
        """Save data from the 'Advanced' tab to in-memory Datasets."""
        advanced_attrs = self.data.advanced.attrs
        advanced_attrs['NDDT'] = self.ui.cbx_nddt.itemData(self.ui.cbx_nddt.currentIndex())
        # Store -1 if toggled checked. Multiplier of NDDT.
        advanced_attrs['NDDT_hot_start'] = int(-1 if self.ui.tog_nddt_hot_start.checkState() == Qt.Checked else 1)
        advanced_attrs['BTIMINC'] = int(self.ui.edt_btiminc.text())
        advanced_attrs['BCHGTIMINC'] = int(self.ui.edt_bchgtiminc.text())

    def _save_registry_settings(self):
        """Read settings from registry for suppressing error messages."""
        settings = SettingsManager()
        suppress_hotstart = 1 if self.ui.tog_suppress_missing_hotstart.isChecked() else 0
        suppress_wind = 1 if self.ui.tog_suppress_missing_wind.isChecked() else 0
        suppress_nonperiodic = 1 if self.ui.tog_suppress_missing_nonperiodic.isChecked() else 0
        settings.save_setting('xmsadcirc', REG_KEY_SUPPRESS_HOTSTART, suppress_hotstart)
        settings.save_setting('xmsadcirc', REG_KEY_SUPPRESS_WIND_FILE, suppress_wind)
        settings.save_setting('xmsadcirc', REG_KEY_SUPPRESS_NONPERIODIC, suppress_nonperiodic)
