"""Table of OWI file selectors where rows are moveable."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QAbstractItemView, QHeaderView

# 3. Aquaveo modules
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget
from xms.adcirc.gui.nws_file_table import WindFileDisableEditModel

PRESSURE_FILE_COL = 0
WIND_FILE_COL = 1


class WindOWIFileTableWidget(AdcircTableWidget):
    """The NWS OWI file selector table widget (for NWS=12)."""
    def __init__(self, data_frame, data, default_values, parent=None):
        """Construct the widget.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The model data.
            data (:obj:`SimData`): SimData object. Needed for resolving relative paths
            default_values (:obj:`dict`): Default DataFrame values for added rows
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        if len(data_frame.index) > 0 and data_frame.index[0] == 0:
            data_frame.index += 1  # Make vertical header 1-based
        super().__init__(parent, data_frame, PRESSURE_FILE_COL, default_values)
        self._file_pressure_selector_delegate = FileSelectorButtonDelegate(
            data.info.attrs['proj_dir'], 'Select a pressure file', None, self.parent()
        )
        self._file_wind_selector_delegate = FileSelectorButtonDelegate(
            data.info.attrs['proj_dir'], 'Select a wind file', None, self.parent()
        )
        self._filter_model = WindFileDisableEditModel(PRESSURE_FILE_COL, self)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        # Assuming the last column is file selector. Change if need to support multiple columns of file selectors in
        # a single table.
        super()._setup_ui(
            {
                PRESSURE_FILE_COL: self._file_pressure_selector_delegate,
                WIND_FILE_COL: self._file_wind_selector_delegate,
            },
            True,
            False,
            filter_model=self._filter_model,
            movable_rows=True
        )
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
