"""The solution output options table."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt, Signal
from PySide2.QtGui import QColor, QPen
from PySide2.QtWidgets import QAbstractItemView, QHeaderView, QLabel, QStyledItemDelegate, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.delegates.file_selector_delegate import FileSelectorButtonDelegate
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.qx_table_view import QxTableView

# 4. Local modules

TBL_COL_NAME = 0
TBL_COL_DEFAULT = 1
TBL_COL_OUTPUT = 2
TBL_COL_START = 3
TBL_COL_END = 4
TBL_COL_INC = 5
TBL_COL_APPEND = 6
TBL_COL_HOT_START_FILE = 7
TBL_COL_HOT_START_FILE_WIND = 8  # Extra hot start for meteorological outputs

TBL_ROW_DEFAULT = 0
TBL_ROW_ELEV = 1
TBL_ROW_VEL = 2
TBL_ROW_WIND = 3
TBL_ROW_STATION_ELEV = 4
TBL_ROW_STATION_VEL = 5
TBL_ROW_STATION_WIND = 6

OUT_FORMATS = [
    'Off',
    'ASCII',
    'Binary',
    'NetCDF',
    'Sparse ASCII',
    'NetCDF4',
]

SPARSE_ASCII_OPT = 4

HOT_START_OPTS = [
    'Append',
    'Overwrite',
]

WIND_ROWS = [
    TBL_ROW_WIND,
    TBL_ROW_STATION_WIND,
]

STATION_ROWS = [
    TBL_ROW_STATION_ELEV,
    TBL_ROW_STATION_VEL,
    TBL_ROW_STATION_WIND,
]

FILE_COLS = [
    TBL_COL_HOT_START_FILE,
    TBL_COL_HOT_START_FILE_WIND,
]


def get_hot_start_filter(index):
    """Get a file extension filter string for output hot start file selectors based on table model index.

    Args:
        index (:obj:`QModelIndex`): Model index of the file selector item that was pressed

    Returns:
        (:obj:`str`): An appropriate filename extension filter for the clicked item
    """
    nc_ext = ''  # Append .nc extension to expected output filename if format is NetCDF
    row = index.row()
    col = index.column()
    if index.model().index(row, TBL_COL_OUTPUT).data() in ['NetCDF', 'NetCDF4']:
        nc_ext = '.nc'

    if row == TBL_ROW_ELEV and col == TBL_COL_HOT_START_FILE:
        return f'fort.63{nc_ext}'
    elif row == TBL_ROW_VEL and col == TBL_COL_HOT_START_FILE:
        return f'fort.64{nc_ext}'
    elif row == TBL_ROW_WIND:
        if col == TBL_COL_HOT_START_FILE:
            return f'fort.73{nc_ext}'
        elif col == TBL_COL_HOT_START_FILE_WIND:
            return f'fort.74{nc_ext}'
    elif row == TBL_ROW_STATION_ELEV and col == TBL_COL_HOT_START_FILE:
        return f'fort.61{nc_ext}'
    elif row == TBL_ROW_STATION_VEL and col == TBL_COL_HOT_START_FILE:
        return f'fort.62{nc_ext}'
    elif row == TBL_ROW_STATION_WIND:
        if col == TBL_COL_HOT_START_FILE:
            return f'fort.71{nc_ext}'
        elif col == TBL_COL_HOT_START_FILE_WIND:
            return f'fort.72{nc_ext}'
    return '*.*'


class DefaultRowBorderDelegate(QStyledItemDelegate):
    """Delegate for drawing a border around the default row."""
    def __init__(self, col_delegates, parent=None):
        """Construct the delegate."""
        super().__init__(parent)
        self.col_delegates = col_delegates

    def paint(self, painter, option, index):
        """Paint thicker, colored border around the default row."""
        # Let parent delegate draw column as it pleases.
        col = index.column()
        if self.col_delegates[col]:
            self.col_delegates[col].paint(painter, option, index)
        else:
            super().paint(painter, option, index)

        # Draw border around row
        old_pen = painter.pen()
        rect = option.rect
        pen = QPen()
        pen.setColor(QColor('blue'))
        pen.setStyle(Qt.SolidLine)
        pen.setWidth(2)
        painter.setPen(pen)
        painter.drawLine(rect.topLeft(), rect.topRight())
        painter.drawLine(rect.bottomLeft(), rect.bottomRight())
        # Draw left edge of left-most cell
        if col == 0:
            painter.drawLine(rect.topLeft(), rect.bottomLeft())
        # Draw right edge of right-most cell
        if col == index.model().columnCount() - 1:
            painter.drawLine(rect.topRight(), rect.bottomRight())
        painter.setPen(old_pen)

    def createEditor(self, parent, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            parent (:obj:`QWidget`): the parent widget
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.

        Returns:
            (:obj:`None`): Returns None to prevent default behavior
        """
        col = index.column()
        if self.col_delegates[col]:
            return self.col_delegates[col].createEditor(parent, option, index)
        else:
            return super().createEditor(parent, option, index)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.

        Returns:
            (:obj:`bool`): The whether the event was consumed.
        """
        col = index.column()
        if self.col_delegates[col]:
            return self.col_delegates[col].editorEvent(event, model, option, index)
        else:
            return super().editorEvent(event, model, option, index)

    def setEditorData(self, editor, index):  # noqa: N802
        """Sets the data to be displayed and edited by the editor from the data model item specified by the model index.

        Args:
            editor (:obj:`QWidget`): The editor.
            index (:obj:`QModelIndex`): The index.

        """
        col = index.column()
        if self.col_delegates[col]:
            return self.col_delegates[col].setEditorData(editor, index)
        else:
            return super().setEditorData(editor, index)

    def setModelData(self, editor, model, index):  # noqa: N802
        """Gets data from the editor widget and stores it in the specified model at the item index.

        Args:
            editor (:obj:`QWidget`): The editor.
            model (:obj:`QAbstractItemModel`): The model.
            index (:obj:`QModelIndex`): The index
        """
        col = index.column()
        if self.col_delegates[col]:
            return self.col_delegates[col].setModelData(editor, model, index)
        else:
            return super().setModelData(editor, model, index)


class OutputTableDisableEditModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, has_stations, has_wind, parent=None):
        """Initializes the filter model.

        Args:
            has_stations (:obj:`bool`): If True, recording station output can be specified
            has_wind (:obj:`bool`): If True, meteorological output can be specified
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.has_stations = has_stations
        self.has_wind = has_wind

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        row = index.row()
        col = index.column()
        enable = True

        if row in WIND_ROWS and not self.has_wind:
            enable = False  # No wind defined
        elif row in STATION_ROWS and not self.has_stations:
            enable = False  # No recording stations defined
        if col == TBL_COL_NAME:  # Disable editing of name column but make it still enabled
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
            return ret_flags
        elif row != TBL_ROW_DEFAULT:  # Nothing disabled in the default row except file selector
            ret_flags = ret_flags & (~Qt.ItemIsEnabled)
            use_default = index.model().index(row, TBL_COL_DEFAULT).data(Qt.CheckStateRole) == Qt.Unchecked
            output_opt = index.model().index(row, TBL_COL_OUTPUT).data()
            output_off = output_opt == 'Off'
            hotstart_allowed = output_opt != 'Sparse ASCII'
            if col in FILE_COLS:
                # Enable the file selector if output enabled and appending
                ret_flags = ret_flags & (~Qt.ItemIsEditable)  # But make sure the push button isn't editable.
                if output_off or not hotstart_allowed or index.model().index(row, TBL_COL_APPEND).data() == 'Overwrite':
                    enable = False
                elif col == TBL_COL_HOT_START_FILE_WIND and (not self.has_wind or row not in WIND_ROWS):
                    enable = False
            elif (use_default and col > TBL_COL_DEFAULT) or (output_off and col > TBL_COL_OUTPUT):
                enable = False  # Disable edit fields and combobox if using defaults. Disable edit fields if off.
            elif col == TBL_COL_APPEND and not hotstart_allowed:
                enable = False
        elif col in FILE_COLS:  # Disable file selectors on default row
            enable = False

        if enable:
            ret_flags |= Qt.ItemIsEnabled
            if col not in FILE_COLS:  # Don't make push button editable
                ret_flags |= Qt.ItemIsEditable
        else:
            ret_flags = ret_flags & (~Qt.ItemIsEnabled)
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
        return ret_flags

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (:obj:`int`): The role.

        Returns:
            The data at index, or None.
        """
        if not index.isValid():
            return None

        if role == Qt.BackgroundColorRole and index.row() == TBL_ROW_DEFAULT:
            return QColor('lightBlue')
        return self.sourceModel().data(index, role)


class OutputTableWidget(QWidget):
    """The output table widget."""
    format_changed = Signal(bool)

    def __init__(self, data_frame, has_stations, has_wind, proj_dir, parent=None):
        """Construct the widget.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The model data.
            has_stations (:obj:`bool`): If True, recording station output can be specified
            has_wind (:obj:`bool`): If True, meteorological output can be specified
            proj_dir (:obj:`str`): File path to the project save location, if it exists.
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.proj_dir = proj_dir
        self.table_view = None
        self.toolbar = None
        self.btn_actions = None
        self.has_stations = has_stations
        self.has_wind = has_wind
        self.model = QxPandasTableModel(data_frame, self)
        self.enable_model = None
        self.format_delegate = None
        self.hot_start_delegate = None
        self.file_selector_delegate = None
        self.tog_delegate = None
        self.valid_delegate = None
        self.border_delegate = None
        self.wind_label = None
        self.station_label = None

        # Setting format to 'Off' looks better in the dialog, but we would lose any previous setting the
        # user had.
        # If recording stations are not linked to the simulation, set their outputs to 'Off'
        # if not self.has_stations:
        #     self.model.data_frame.at['NOUTE', 'Output'] = 0
        #     self.model.data_frame.at['NOUTV', 'Output'] = 0
        #     self.model.data_frame.at['NOUTW', 'Output'] = 0

        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        self.enable_model = OutputTableDisableEditModel(self.has_stations, self.has_wind, self)
        self.format_delegate = QxCbxDelegate(self)
        self.format_delegate.set_strings(OUT_FORMATS)
        self.hot_start_delegate = QxCbxDelegate(self)
        self.hot_start_delegate.set_strings(HOT_START_OPTS)
        self.file_selector_delegate = FileSelectorButtonDelegate(
            self.proj_dir, 'Select output file to append to', get_hot_start_filter, self
        )
        self.tog_delegate = CheckBoxNoTextDelegate(self)
        self.tog_delegate.state_changed.connect(self.use_default_changed)
        dbl_validator = QxDoubleValidator(parent=self)
        self.valid_delegate = EditFieldValidator(dbl_validator, self)

        col_delegates = [
            None,
            self.tog_delegate,
            self.format_delegate,
            self.valid_delegate,
            self.valid_delegate,
            self.valid_delegate,
            self.hot_start_delegate,
            self.file_selector_delegate,
            self.file_selector_delegate,
        ]
        self.border_delegate = DefaultRowBorderDelegate(col_delegates, self)

        self.table_view = QxTableView(self)
        self.model.dataChanged.connect(self.default_option_changed)
        self.model.set_read_only_columns({TBL_COL_NAME})
        self.model.set_checkbox_columns([TBL_COL_DEFAULT])
        self.table_view.setModel(self.model)
        self.table_view.setItemDelegateForRow(TBL_ROW_DEFAULT, self.border_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_DEFAULT, self.tog_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_OUTPUT, self.format_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_START, self.valid_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_END, self.valid_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_INC, self.valid_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_APPEND, self.hot_start_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_HOT_START_FILE, self.file_selector_delegate)
        self.table_view.setItemDelegateForColumn(TBL_COL_HOT_START_FILE_WIND, self.file_selector_delegate)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        self.table_view.filter_model = self.enable_model
        self.table_view.filter_model.setSourceModel(self.model)
        self.table_view.setModel(self.table_view.filter_model)
        self.table_view.horizontalHeader().setSectionResizeMode(TBL_COL_HOT_START_FILE, QHeaderView.Stretch)
        self.table_view.horizontalHeader().setSectionResizeMode(TBL_COL_HOT_START_FILE_WIND, QHeaderView.Stretch)
        # self.table_view.horizontalHeader().setStretchLastSection(True)
        self.table_view.verticalHeader().setVisible(False)
        self.table_view.resizeColumnsToContents()
        self.table_view.resize_height_to_contents()
        layout = QVBoxLayout()
        layout.addWidget(self.table_view)
        if not self.has_wind:
            self.wind_label = QLabel(
                'To enable meteorological output, NWS must be greater than 0 (specified in the "Wind" tab).'
            )
            self.wind_label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            layout.addWidget(self.wind_label)
        if not self.has_stations:
            self.station_label = QLabel(
                'To enable recording station output, a recording station coverage must be linked to the simulation.'
            )
            self.station_label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            layout.addWidget(self.station_label)
        self.setLayout(layout)

    def use_default_changed(self, index):
        """Called when the use default checkbox is toggled for a row."""
        if not index.isValid():
            return
        row = index.row()
        if row > TBL_ROW_DEFAULT:
            if index.data(Qt.CheckStateRole) == Qt.Unchecked:
                default_format_idx = self.model.index(TBL_ROW_DEFAULT, TBL_COL_OUTPUT)
                default_format = self.model.data(default_format_idx)
                new_idx = self.model.index(row, TBL_COL_OUTPUT)
                self.model.setData(new_idx, default_format)
                default_start_idx = self.model.index(TBL_ROW_DEFAULT, TBL_COL_START)
                default_start = self.model.data(default_start_idx)
                new_idx = self.model.index(row, TBL_COL_START)
                self.model.setData(new_idx, default_start)
                default_end_idx = self.model.index(TBL_ROW_DEFAULT, TBL_COL_END)
                default_end = self.model.data(default_end_idx)
                new_idx = self.model.index(row, TBL_COL_END)
                self.model.setData(new_idx, default_end)
                default_inc_idx = self.model.index(TBL_ROW_DEFAULT, TBL_COL_INC)
                default_inc = self.model.data(default_inc_idx)
                new_idx = self.model.index(row, TBL_COL_INC)
                self.model.setData(new_idx, default_inc)
                default_append_idx = self.model.index(TBL_ROW_DEFAULT, TBL_COL_APPEND)
                default_append = self.model.data(default_append_idx)
                new_idx = self.model.index(row, TBL_COL_APPEND)
                self.model.setData(new_idx, default_append)
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_OUTPUT))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_START))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_END))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_INC))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_APPEND))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_HOT_START_FILE))
        self.table_view.update(self.table_view.filter_model.index(index.row(), TBL_COL_HOT_START_FILE_WIND))

    def default_option_changed(self, index):
        """Handles updating rows that are using the default options when a default option changes.

        Args:
            index (:obj:`QModelIndex`): Model index of the item that changed
        """
        if not index.isValid() or index.row() != TBL_ROW_DEFAULT:
            return

        col = index.column()
        default_toggle = col == TBL_COL_DEFAULT
        if default_toggle:
            default_val = 1 if self.model.data(index, Qt.CheckStateRole) == Qt.Checked else 0
        else:
            default_val = self.model.data(index)

        for i in range(TBL_ROW_ELEV, self.model.rowCount()):
            if (i in WIND_ROWS and not self.has_wind) or (i in STATION_ROWS and not self.has_stations):
                continue  # Don't update disabled outputs
            row_idx = self.model.index(i, TBL_COL_DEFAULT)
            if not default_toggle and self.model.data(row_idx, Qt.CheckStateRole) == Qt.Checked:
                # Skip rows that aren't using the default options, unless this is the default toggle on default row.
                continue
            row_idx = self.model.index(i, col)
            self.model.setData(row_idx, default_val)
            self.table_view.update(row_idx)
            if default_toggle:
                self.use_default_changed(row_idx)

    def on_nws_changed(self, nws):
        """Called when the NWS combobox option is changed.

        Args:
            nws (:obj:`int`): The new NWS option
        """
        self.has_wind = nws > 0
        self.enable_model.has_wind = self.has_wind
        # Enable/disable the warning label
        if self.wind_label:
            self.wind_label.setVisible(not self.has_wind)

        # If wind has been disabled, turn off meteorological output.
        if not self.has_wind:
            pass
            # Setting format to 'Off' looks better in the dialog, but we would lose any previous setting the user had.
            # self.model.data_frame.at['NOUTGW', 'Output'] = 0
            # self.model.data_frame.at['NOUTW', 'Output'] = 0
        else:
            # If we have wind and are using the default options, make sure the wind rows are in
            # sync with the default row.
            using_default_idx = self.model.index(TBL_ROW_WIND, TBL_COL_DEFAULT)
            using_default = (self.model.data(using_default_idx, Qt.CheckStateRole) == Qt.Unchecked)
            if using_default:
                self.use_default_changed(using_default_idx)
            if self.has_stations:
                using_default_idx = self.model.index(TBL_ROW_STATION_WIND, TBL_COL_DEFAULT)
                using_default = (self.model.data(using_default_idx, Qt.CheckStateRole) == Qt.Unchecked)
                if using_default:
                    self.use_default_changed(using_default_idx)

        # Enable/disable the meteorological rows
        for i in range(TBL_COL_DEFAULT, TBL_COL_HOT_START_FILE_WIND + 1):
            row_idx = self.model.index(TBL_ROW_WIND, i)
            self.table_view.update(row_idx)
            if self.has_stations:
                row_idx = self.model.index(TBL_ROW_STATION_WIND, i)
                self.table_view.update(row_idx)
