"""Dialog to export individual files of an ADCIRC simulation."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QPushButton, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.file_selector_dialogs import get_save_filename
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.adcirc.feedback.fort13_worker_thread import Fort13ExportWorkerThread
from xms.adcirc.feedback.fort14_worker_thread import Fort14ExportWorkerThread
from xms.adcirc.feedback.fort15_worker_thread import export_fort15_with_feedback
from xms.adcirc.feedback.fort20_worker_thread import Fort20ExportWorkerThread
from xms.adcirc.feedback.fort22_worker_thread import Fort22ExportWorkerThread


class PartialExportDialog(XmsDlg):
    """A dialog that appears for selecting individual files to export."""
    def __init__(self, parent, query):
        """Initializes the class, sets up the ui.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS

        """
        super().__init__(parent, 'xmsadcirc.gui.partial_export_dlg')
        self.setWindowTitle('Export ADCIRC Files to Location')
        self.query = query
        self.widgets = {}
        self.setup_ui()
        self.filenames = []  # For testing

    def _get_save_filename(self, selected_filter, file_filter):
        """Get a filename to use for writing a file.

        Args:
            selected_filter (:obj:`str`): The file filter that should be selected in the file browser dialog
            file_filter (:obj:`str`): The file filter list that should be in the file browser dialog. Filters separated
                by ';;'

        Returns:
            (:obj:`str`): The selected filename (or next hard-coded testing filename). Empty string if user cancels.
        """
        if self.filenames:
            return self.filenames.pop()
        return get_save_filename(self, selected_filter, file_filter)

    def setup_ui(self):
        """Add widgets to the dialog."""
        self.widgets['btn_fort13'] = QPushButton('fort.13...')
        self.widgets['btn_fort13'].clicked.connect(self.on_btn_fort13)
        self.widgets['btn_fort14'] = QPushButton('fort.14...')
        self.widgets['btn_fort14'].clicked.connect(self.on_btn_fort14)
        self.widgets['btn_fort15'] = QPushButton('fort.15...')
        self.widgets['btn_fort15'].clicked.connect(self.on_btn_fort15)
        self.widgets['btn_fort20'] = QPushButton('fort.20...')
        self.widgets['btn_fort20'].clicked.connect(self.on_btn_fort20)
        self.widgets['btn_fort22'] = QPushButton('fort.22...')
        self.widgets['btn_fort22'].clicked.connect(self.on_btn_fort22)
        self.widgets['layout_vert_files'] = QVBoxLayout()
        self.widgets['layout_vert_files'].addWidget(self.widgets['btn_fort13'])
        self.widgets['layout_vert_files'].addWidget(self.widgets['btn_fort14'])
        self.widgets['layout_vert_files'].addWidget(self.widgets['btn_fort15'])
        self.widgets['layout_vert_files'].addWidget(self.widgets['btn_fort20'])
        self.widgets['layout_vert_files'].addWidget(self.widgets['btn_fort22'])

        self.widgets['lbl_fort13'] = QLabel('Export ADCIRC nodal attributes to a specified fort.13 file.')
        self.widgets['lbl_fort14'] = QLabel(
            'Export ADCIRC domain mesh and applied boundary conditions to a specified fort.14 file.'
        )
        self.widgets['lbl_fort15'] = QLabel('Export ADCIRC control file to a specified fort.15 file.')
        self.widgets['lbl_fort20'] = QLabel('Export ADCIRC hydrographs to a specified fort.20 file.')
        self.widgets['lbl_fort22'] = QLabel('Export ADCIRC wind data to a specified fort.22 file.')
        self.widgets['layout_vert_lbls'] = QVBoxLayout()
        self.widgets['layout_vert_lbls'].addWidget(self.widgets['lbl_fort13'])
        self.widgets['layout_vert_lbls'].addWidget(self.widgets['lbl_fort14'])
        self.widgets['layout_vert_lbls'].addWidget(self.widgets['lbl_fort15'])
        self.widgets['layout_vert_lbls'].addWidget(self.widgets['lbl_fort20'])
        self.widgets['layout_vert_lbls'].addWidget(self.widgets['lbl_fort22'])

        self.widgets['layout_horiz_files'] = QHBoxLayout()
        self.widgets['layout_horiz_files'].addLayout(self.widgets['layout_vert_files'])
        self.widgets['layout_horiz_files'].addLayout(self.widgets['layout_vert_lbls'])
        self.widgets['grp_files'] = QGroupBox('ADCIRC files')
        self.widgets['grp_files'].setLayout(self.widgets['layout_horiz_files'])
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setStandardButtons(QDialogButtonBox.Ok)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['layout_vert_main'] = QVBoxLayout()
        self.widgets['layout_vert_main'].addWidget(self.widgets['grp_files'])
        self.widgets['layout_vert_main'].addWidget(self.widgets['btn_box'])
        self.setLayout(self.widgets['layout_vert_main'])

    def on_btn_fort14(self):
        """Slot for when the user clicks the export fort.14 button."""
        # Prompt the user for a save location
        selected_filter = 'ADCIRC native (fort.14)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter)
        if not filename:
            return  # User cancelled
        note = (
            "If the simulation contains applied boundary condition nodestrings, they will be exported to the "
            "fort.14. Otherwise, only the domain mesh will be written to the file."
        )
        worker = Fort14ExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting fort.14',
            'working_prompt': f'Exporting fort.14 to \"{filename}\" Please wait...',
            'error_prompt': 'Error(s) encountered exporting fort.14. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting fort.14. Review log output for more details.',
            'success_prompt': f'Successfully exported fort.14 to  \"{filename}\"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()

    def on_btn_fort13(self):
        """Slot for when the user clicks the export fort.13 button."""
        # Prompt the user for a save location
        selected_filter = 'ADCIRC native (fort.13)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter)
        if not filename:
            return  # User cancelled
        note = 'Only currently enabled nodal attributes of the simulation will be exported.'
        worker = Fort13ExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting fort.13',
            'working_prompt': f'Exporting fort.13 to \"{filename}\" Please wait...',
            'error_prompt': 'Error(s) encountered exporting fort.13. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting fort.13. Review log output for more details.',
            'success_prompt': f'Successfully exported fort.13 to  \"{filename}\"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()

    def on_btn_fort15(self):
        """Slot for when the user clicks the export fort.15 button."""
        # Prompt the user for a save location
        selected_filter = 'ADCIRC native (fort.15)'
        file_filter = f'{selected_filter};;All types (*.*)'
        filename = self._get_save_filename(selected_filter, file_filter)
        if not filename:
            return  # User cancelled
        note = 'Only the fort.15 ADCIRC control file will be exported to the specified location.'
        try:
            export_fort15_with_feedback(filename=filename, query=self.query, xms_data=None, note=note, parent=self)
        except Exception:
            pass
        LogEchoQSignalStream.reset_flags()

    def on_btn_fort20(self):
        """Slot for when the user clicks the export fort.20 button."""
        # Prompt the user for a save location
        selected_filter = 'ADCIRC native (fort.20)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter)
        if not filename:
            return  # User cancelled
        note = 'Only currently enabled nodal attributes of the simulation will be exported.'
        worker = Fort20ExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting fort.20',
            'working_prompt': f'Exporting fort.20 to \"{filename}\" Please wait...',
            'error_prompt': 'Error(s) encountered exporting fort.20. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting fort.20. Review log output for more details.',
            'success_prompt': f'Successfully exported fort.20 to  \"{filename}\"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()

    def on_btn_fort22(self):
        """Slot for when the user clicks the export fort.22 button."""
        # Prompt the user for a save location
        selected_filter = 'ADCIRC native (fort.22)'
        file_filter = f'{selected_filter};;All types (*.*)'
        testing = 'testing' if self.filenames else ''
        filename = self._get_save_filename(selected_filter, file_filter)
        if not filename:
            return  # User cancelled
        note = 'fort.22 will be exported with format currently specified in the simulation "Model Control" dialog.'
        worker = Fort22ExportWorkerThread(filename, self.query, self)
        display_text = {
            'title': 'Exporting fort.22',
            'working_prompt': f'Exporting fort.22 to \"{filename}\" Please wait...',
            'error_prompt': 'Error(s) encountered exporting fort.22. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered exporting fort.22. Review log output for more details.',
            'success_prompt': f'Successfully exported fort.22 to  \"{filename}\"',
            'note': note,
            'auto_load': testing,
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adcirc', worker, self)
        feedback_dlg.testing = True if testing else False
        feedback_dlg.exec()
        LogEchoQSignalStream.reset_flags()
