"""A component for hydrodynamic material data and commands."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import os
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.api.tree import tree_util
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.components.display import fix_duplicated_display_opts, MAT_INITIAL_ATT_ID_FILE, MAT_INITIAL_COMP_ID_FILE
from xms.adh.data.materials_io import MaterialsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.file_io import io_util
from xms.adh.gui.assign_poly_material_dialog import AssignPolyMaterialDialog
from xms.adh.gui.material_dialog import MaterialDialog


class MaterialConceptualComponent(AdHComponent):
    """A hidden Dynamic Model Interface (DMI) component for the AdH model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.is_sediment = False
        self.data = MaterialsIO(self.main_file)
        self.cov_uuid = self.data.cov_uuid
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Material List and Properties', 'open_material_properties'),
        ]
        self.polygon_commands = [('Assign Material', 'open_assign_poly_materials')]
        # Copy default display options if needed
        comp_dir = os.path.dirname(self.main_file)
        self.disp_opts_file = os.path.join(comp_dir, 'material_display_options.json')
        if not os.path.exists(self.disp_opts_file):
            # Copy over the OFF material category id file.
            default_id_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                f'mat_{MaterialsIO.UNASSIGNED_MAT}.matid'
            )
            io_util.copyfile(default_id_file, os.path.join(comp_dir, os.path.basename(default_id_file)))
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            json_dict = default_display_options()
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            categories.from_dict(json_dict)
            categories.comp_uuid = self.uuid
            write_display_options_to_json(self.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.display_uuid = categories.uuid
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            # Change the display UUID to be something different.
            new_mat_comp = MaterialConceptualComponent(new_main_file)

            # Correct the component UUID and change the display UUID to something new
            json_dict = fix_duplicated_display_opts(new_path, new_mat_comp.disp_opts_file)
            new_mat_comp.data.display_uuid = json_dict['uuid']
            new_mat_comp.data.cov_uuid = ""

            new_mat_comp.data.commit()
        return new_main_file, messages, action_requests

    def project_open_event(self, _new_path):
        """Called when an XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            _new_path (str): Path to the new save location.

        """
        return  # This was a legacy thing, but we didn't have a release until this became unnecessary.

    def create_event(self, _lock_state):
        """This will be called when the component is created from nothing.

        Args:
            _lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        messages = []
        action_requests = [self.get_display_refresh_action()]
        return messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        if not self.cov_uuid:
            self.cov_uuid = query.parent_item_uuid()
            self.data.cov_uuid = self.cov_uuid
            self.data.commit()

        initial_att_file = os.path.join(os.path.dirname(self.main_file), MAT_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), MAT_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            io_util.remove(initial_att_file)
            io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.polygon, att_id, comp_id)
            mat_ids = list(self.data.materials.material_display.keys())
            self.update_display_id_files(mat_ids, mat_ids)
            self.update_display_options_file()

        # Send the default material list to XMS.
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))
        return [], []

    def open_assign_poly_materials(self, query, params, win_cont):
        """Opens the Assign Materials dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        # Get the XMS attribute ids of the selected polygons (if any)
        poly_ids = []
        params = params[0]
        if 'selection' in params:
            poly_ids = params['selection']
        num_polys = len(poly_ids)
        if num_polys == 0:
            return [('INFO', 'No polygons selected. Select one or more polygons to assign materials.')], []

        # Get the component id (material id) map of the selected polygons (if any).
        current_mat_id = MaterialsIO.UNASSIGNED_MAT
        if 'id_files' in params:
            if params['id_files'] and params['id_files'][0]:
                files_dict = {'POLYGON': (params['id_files'][0], params['id_files'][1])}
                self.load_coverage_component_id_map(files_dict)
                # Delete the id dumped by xms files.
                if poly_ids:
                    current_mat_id = self.get_comp_id(TargetType.polygon, poly_ids[0])  # First if multiple selected.
                    if current_mat_id < 0:
                        current_mat_id = MaterialsIO.UNASSIGNED_MAT

        title = 'Assign Material' if num_polys == 1 else 'Assign Materials'
        multi_select_lbl = ''
        if num_polys > 1:
            multi_select_lbl = 'Material assignment will be applied to all selected polygons.'
        mat_names = [material.material_name for material in self.data.materials.material_properties.values()]
        mat_ids = list(self.data.materials.material_properties.keys())
        current_mat_idx = MaterialsIO.UNASSIGNED_MAT
        for idx, mat_id in enumerate(mat_ids):
            if mat_id == current_mat_id:
                current_mat_idx = idx
                break
        dlg = AssignPolyMaterialDialog(win_cont, title, multi_select_lbl, mat_names, current_mat_idx)
        if dlg.exec():
            # Set the material coverage display list out-of-date
            self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))

            # Update the polygon material assignment for the selected polygons.
            selected_mat_idx = dlg.get_selected_material()
            mat_id = mat_ids[selected_mat_idx]
            for poly_id in poly_ids:
                self.update_component_id(TargetType.polygon, int(poly_id), int(mat_id))
            self.update_display_id_files(mat_ids, mat_ids)
            self.data.commit()

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)
        return [], []

    def unassign_materials(self, query, delete_ids):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            delete_ids (:obj:`list'): List of the deleted material ids.

        Returns:
            Empty message and ActionRequest lists

        """
        files_dict = query.load_component_ids(self, polygons=True, delete_files=False)
        if self.cov_uuid in self.comp_to_xms and TargetType.polygon in self.comp_to_xms[self.cov_uuid]:
            poly_map = self.comp_to_xms[self.cov_uuid][TargetType.polygon]
            for mat in delete_ids:
                if mat in poly_map:
                    for att_id in poly_map[mat]:
                        self.update_component_id(TargetType.polygon, att_id, MaterialsIO.UNASSIGNED_MAT)

        self.remove_id_files(files_dict)

    def update_display_options_file(self):
        """Update the XMS display options JSON file to match what we have in memory."""
        category_list = CategoryDisplayOptionList()
        category_list.target_type = TargetType.polygon
        category_list.uuid = str(self.data.display_uuid)
        category_list.comp_uuid = self.uuid
        for mat_id, material in self.data.materials.material_properties.items():
            category = CategoryDisplayOption()
            category.id = mat_id
            category.description = material.material_name
            category.file = f'mat_{category.id}.matid'
            category.options = PolygonOptions()
            category.options.color = self.data.materials.material_display[mat_id].color
            category.options.texture = self.data.materials.material_display[mat_id].texture
            # Make OFF material the default category.
            if category.id == 0:  # Should always have 0 as "material id"
                category.is_unassigned_category = True
            category_list.categories.append(category)
        write_display_options_to_json(self.disp_opts_file, category_list)

    def open_material_properties(self, query, params, win_cont):
        """Opens the Material Properties dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # get original material ids
        ids = list(self.data.materials.material_properties.keys())

        transport_comp, transport_name, use_transport = self._get_transport(query.project_tree)

        dlg = MaterialDialog(win_cont, 'Material List and Properties', self.data, query.project_tree)
        dlg.set_transport(self.data.transport_uuid, transport_comp, transport_name, use_transport)
        if dlg.exec():
            # Check for materials removed from the list.
            self.data = dlg.data
            new_ids = self.data.materials.material_properties.keys()
            deleted_ids = self.update_display_id_files(ids, list(new_ids))
            self.unassign_materials(query, deleted_ids)
            # Write the display options file.
            self.update_display_options_file()
            self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))
            self.data.delete_materials_for_commit(deleted_ids)
            self.data.commit()

        return [], []

    def update_display_id_files(self, old_ids, new_ids):
        """Update the display files.

        Args:
            old_ids (list): list of ids before editing materials
            new_ids (list): list of current material ids

        Returns:
            (list) : deleted ids
        """
        deleted_ids = copy.deepcopy(old_ids)
        path = os.path.dirname(self.main_file)
        for mat_id in new_ids:
            id_file = f'mat_{mat_id}.matid'
            filename = os.path.join(path, id_file)
            id_list = [mat_id]
            write_display_option_ids(filename, id_list)
            if mat_id in deleted_ids:
                deleted_ids.remove(mat_id)

        for mat_id in deleted_ids:
            id_file = f'mat_{mat_id}.matid'
            filename = os.path.join(path, id_file)
            io_util.remove(filename)
        return deleted_ids

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list` of :obj:`tuple`): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        ATT_FILE = 0  # noqa N806
        COMP_FILE = 1  # noqa N806
        MAIN_FILE_IDX = 0  # noqa N806
        ID_FILE_IDX = 1  # noqa N806
        used_att_ids = []  # prevent a double assignment
        for m in merge_list:
            other_mat = MaterialConceptualComponent(m[MAIN_FILE_IDX])
            old_mat_to_new_mat = {MaterialsIO.UNASSIGNED_MAT: MaterialsIO.UNASSIGNED_MAT}
            for mat_id in other_mat.data.materials.material_properties:
                if mat_id == MaterialsIO.UNASSIGNED_MAT:
                    continue
                # make a new material id
                new_id = max(self.data.materials.material_properties, key=int) + 1
                self.data.materials.material_properties[new_id] = other_mat.data.materials.material_properties[mat_id]
                self.data.materials.material_display[new_id] = other_mat.data.materials.material_display[mat_id]
                old_mat_to_new_mat[mat_id] = new_id
            att_ids = read_display_option_ids(m[ID_FILE_IDX]['POLYGON'][ATT_FILE])
            comp_ids = read_display_option_ids(m[ID_FILE_IDX]['POLYGON'][COMP_FILE])
            for att_id, comp_id in zip(att_ids, comp_ids):
                if att_id not in used_att_ids:
                    self.update_component_id(TargetType.polygon, att_id, old_mat_to_new_mat[comp_id])
            used_att_ids.extend(att_ids)
        # Write the display options file.
        self.update_display_options_file()
        all_mat_ids = list(self.data.materials.material_properties.keys())
        self.update_display_id_files(all_mat_ids, all_mat_ids)
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))
        self.data.commit()

        action = ActionRequest(
            modality='NO_DIALOG',
            method_name='update_component_ids_after_merge',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            parameters={
                'uuid': self.uuid,
                'ids': self.update_ids
            }
        )
        return [], [action]

    def update_component_ids_after_merge(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters.

        Returns:
            Empty message and ActionRequest lists

        """
        params = params[0]
        if 'uuid' in params and 'ids' in params:
            self.uuid = params['uuid']
            self.update_ids = {}
            for uuid_key, uuid_dict in params['ids'].items():  # uuid
                new_type_dict = {}
                uuid_str = uuid_key
                if uuid_str == '':
                    uuid_str = self.cov_uuid
                for type_key, type_dict in uuid_dict.items():  # feature type
                    type_int = type_key
                    new_type_dict[type_int] = {}
                    for xms_key, comp_id in type_dict.items():  # xms id
                        xms_id = xms_key
                        new_type_dict[type_int][xms_id] = comp_id
                self.update_ids[uuid_str] = new_type_dict
        return [], []

    def _get_transport(self, pe_tree):
        """Gets the transport constituents component data.

        Args:
            pe_tree (TreeNode): A root TreeNode of the Project Explorer.

        Returns:
            A tuple with: (a TransportConstituentsIO of the transport constituent component, the transport constituent
            component name as a string, a bool that is True if transport constituents are current used)
        """
        # check for the transport constituents component
        transport_comp = None
        use_transport = self.data.use_transport
        transport_name = ''
        if self.data.transport_uuid:
            # get the name of the transport constituent
            tree_node = tree_util.find_tree_node_by_uuid(pe_tree, self.data.transport_uuid)
            if tree_node:
                transport_name = tree_node.name
                main_file = tree_node.main_file

                # get the transport constituent
                if main_file:
                    transport_comp = TransportConstituentsIO(main_file)
        return transport_comp, transport_name, use_transport


def default_display_options() -> dict:
    """
    Retrieves default display options.

    Returns:
        dict: A dictionary containing the display options.
    """
    default_file = os.path.join(
        os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
        'material_default_display_options.json'
    )
    json_dict = read_display_options_from_json(default_file)
    return json_dict
