"""Module for the get_model function."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules


def get_model() -> GenericModel:
    """Create the hard bottom data definition."""
    gm = GenericModel(exclusive_arc_conditions=True)
    ap = gm.arc_parameters

    group_names = ['first_segment', 'segment']
    labels = ['First Segment', 'Segment']
    for i in range(2):
        sdef = ap.add_group(group_name=group_names[i], label=labels[i])

        segment_type = sdef.add_option('segment_type', 'Type of Segment', default='Arc', options=['Line', 'Arc'])
        sdef.add_float('time_of_arrival', 'Vessel time of arrival (seconds)', default=0.00)
        flags = {'Line': False, 'Arc': True}
        sdef.add_float('x-center', 'x-coordinate of arc center', default=0.00).add_dependency(segment_type, flags)
        sdef.add_float('y-center', 'y-coordinate of arc center', default=0.00).add_dependency(segment_type, flags)
        sdef.add_option(
            'turn_direction',
            'Direction of turn',
            default='Left',
            options=['Left', 'Right']
        ).add_dependency(segment_type, flags)

    vessel = gm.model_parameters.add_group('vessel', 'Vessel')

    vessel.add_float('initial_speed', 'Initial speed (FDEF)', default=0.0)
    vessel.add_float('vessel_draft', 'Vessel draft (DRFT)', default=1.0)
    vessel.add_float('vessel_length', 'Vessel length (BLEN)', default=0.1, low=0.0)
    vessel.add_float('vessel_width', 'Vessel width (BWID)', default=0.0, low=0.0)
    vessel.add_float('bow_length_ratio', 'Bow length ratio (PBOW)', default=0.1, low=0.0)
    vessel.add_float('stern_length_ratio', 'Stern length ratio (PSTR)', default=0.1, low=0.0)
    vessel.add_float('bow_draft_ratio', 'Bow draft ratio (CBOW)', default=1.0, low=0.0)
    vessel.add_float('stern_draft_ratio', 'Stern draft ratio (CSTR)', default=1.0, low=0.0)

    propeller = gm.model_parameters.add_group('propeller', 'Propeller (PROP)')

    options = ['Open wheel', 'Kort nozzle']
    propeller.add_option('propeller_type', 'Propeller type', default='Open wheel', options=options)
    propeller.add_float('propeller_diameter', 'Propeller diameter', default=0.0)
    propeller.add_float('center_distance', 'Propeller center distance', default=0.0)
    propeller.add_float('tow_boat_length', 'Tow boat length', default=0.0)
    propeller.add_float('distance_to_stern', 'Distance to stern', default=0.0)

    return gm
