"""Reads the boundary conditions file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from dataclasses import dataclass
import io
import logging
import os
from typing import Optional
from uuid import uuid4

# 2. Third party modules
from adhparam import file_io
from adhparam.hot_start_data_set import HotStartDataSet
from adhparam.material_transport_properties import MaterialTransportProperties
from adhparam.simulation import AdhSimulation
from adhparam.vessels import Vessel, VesselList
import numpy as np
import pandas as pd
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.coverage_component_builder import CoverageComponentBuilder
from xms.components.display.display_options_io import write_display_option_ids
from xms.constraint.ugrid_builder import UGridBuilder
from xms.core.filesystem import filesystem as io_util
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import Arc, Component, Coverage, Point, Projection, Simulation, UGrid
from xms.datasets.dataset_writer import DatasetWriter
from xms.grid.ugrid import UGrid as XmUGrid
from xms.guipy.data.target_type import TargetType

# 4. Local modules
import xms.adh.components.display
from xms.adh.components.vessel_component import VesselComponent
from xms.adh.data.bc_io import BcIO
from xms.adh.data.materials_io import MaterialsIO
from xms.adh.data.model import get_model
from xms.adh.data.model_control import ModelControl
from xms.adh.data.output_data import OutputData
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.sediment_materials_io import SedimentMaterialsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.gui.widgets.color_list import ColorList


@dataclass
class ImportInfo:
    """Data from import."""
    adh_simulation: AdhSimulation | None = None
    hot_starts: list[HotStartDataSet] | None = None
    vessel_list: VesselList | None = None


def concat_df(dataframe_1: pd.DataFrame, dataframe_2: pd.DataFrame) -> pd.DataFrame:
    """Concatenates two pandas DataFrames.

    Args:
        dataframe_1: The first DataFrame to be concatenated.
        dataframe_2: The second DataFrame to be concatenated.

    Returns:
        A new DataFrame that is the result of the concatenation of dataframe_1 and dataframe_2.

    Notes:
        Should really be building the dataframe all at once instead of concatenating a row at a time.
    """
    if dataframe_1.empty and dataframe_2.empty:
        return dataframe_1.copy()
    elif dataframe_1.empty:
        return dataframe_2.copy()
    elif dataframe_2.empty:
        return dataframe_1.copy()
    return pd.concat([dataframe_1, dataframe_2])


def param_unit_to_combobox_text(param_units):
    """Convert units string used by adhparam.time_control.TimeControl class to equivalent combobox text.

    DMI interface does not support weeks for units, so day text will be returned instead.

    Args:
        param_units (str): Units string as represented in TimeControl param object. One of: '0 - seconds',
            '1 - minutes', '2 - hours', '3 - days', '4 - weeks'

    Returns:
        (str): One of: 'Days', 'Hours', 'Minutes', or 'Seconds' for use in combobox.

    """
    if param_units == '0 - seconds':
        return 'Seconds'
    elif param_units == '1 - minutes':
        return 'Minutes'
    elif param_units == '2 - hours':
        return 'Hours'
    else:
        return 'Days'


def param_to_xms_time(time_val, param_units):
    """Convert units string used by adhparam.time_control.TimeControl class to equivalent xmsapi keyword.

    DMI interface does not support weeks for units, so those values will be converted to days.

    Args:
        time_val (float): The time value
        param_units (str): Units string as represented in TimeControl param object. One of: '0 - seconds',
            '1 - minutes', '2 - hours', '3 - days', '4 - weeks'

    Returns:
        (:obj:`tuple` of :obj:`float`, :obj:`str`):
            Tuple containing the converted time value (if it was in weeks) and one of the following: '#TIME_WEEKS',
            '#TIME_DAYS', '#TIME_HOURS', '#TIME_MINUTES', '#TIME_SECONDS', or '' if unrecognized units.

    """
    if param_units == '4 - weeks':
        return time_val * 7, '#TIME_DAYS'
    elif param_units == '3 - days':
        return time_val, '#TIME_DAYS'
    elif param_units == '2 - hours':
        return time_val, '#TIME_HOURS'
    elif param_units == '1 - minutes':
        return time_val, '#TIME_MINUTES'
    elif param_units == '0 - seconds':
        return time_val, '#TIME_SECONDS'
    else:
        return time_val, ''


def adh_read(query: Query | None = None, logger: logging.Logger | None = None, bc_file_path: str = None):
    """
    Reads an AdH bc file and converts it to an XMS AdH model.

    Args:
        query: The object that communicates with XMS. Value can be None for testing.
        logger: An instance of the logging.Logger class to log any error messages.
        bc_file_path: The file path to the AdH bc file.
    """
    if query is None:
        query = Query()
    if bc_file_path is None:
        bc_file_path = query.read_file

    if not os.path.isfile(bc_file_path):
        raise RuntimeError('AdH bc file not found: {}'.format(bc_file_path))

    import_info = adh_read_simulation(bc_file_path, logger)
    if import_info is not None:
        xms_adh_model = AdhModelToXmsData(import_info.adh_simulation, query, import_info.hot_starts,
                                          import_info.vessel_list)
        xms_adh_model.convert_and_send()


def adh_read_simulation(filename: str, logger: Optional[logging.Logger]) -> ImportInfo | None:
    """Reads the AdH simulation and sends it back to XMS.

    Args:
        filename (str): Path and name of the bc file to read.
        logger: An instance of the logging.Logger class to log any error messages.

    Returns:
        An ImportInfo object that has the values from the files to be converted into components and geometry.
    """
    param_adh_model = AdhSimulation()
    param_adh_model.model_name = os.path.splitext(os.path.basename(filename))[0]
    adh_directory = os.path.dirname(filename)
    file_io.read_bc_file(filename, param_adh_model.boundary_conditions, param_adh_model.model_control, logger)
    mesh_filename = os.path.join(adh_directory, param_adh_model.model_name + '.3dm')
    if os.path.exists(mesh_filename):
        file_io.read_mesh_file(mesh_filename, param_adh_model.mesh, logger)
        if len(param_adh_model.mesh.nodes) == 0:
            logger.error('Mesh file does not contain any nodes.')
            return None
        if param_adh_model.mesh.elements is None and param_adh_model.mesh.elements_3d is None:
            logger.error('Mesh file does not contain any elements.')
            return None
        if param_adh_model.mesh.name == '':
            # The mesh file didn't have a 'MESHNAME' line in it, so give it a default name
            param_adh_model.mesh.name = os.path.splitext(os.path.basename(mesh_filename))[0]
    hotstart_filename = os.path.join(adh_directory, param_adh_model.model_name + '.hot')
    hot_starts = []
    if os.path.exists(hotstart_filename):
        hot_starts = file_io.read_hot_start_file(hotstart_filename, logger)
    # Read vessel file
    vessel_list = VesselList()
    vessel_filename = os.path.join(adh_directory, param_adh_model.model_name + '.bt')
    if os.path.exists(vessel_filename):
        file_io.read_bt_file(vessel_filename, vessel_list, logger)

    return ImportInfo(adh_simulation=param_adh_model, hot_starts=hot_starts, vessel_list=vessel_list)


class AdhModelToXmsData:
    """Convert an AdhSimulation to XMS data and send it to SMS."""

    def __init__(self, adh_model: AdhSimulation, query: Query, hotstarts, vessel_list=None):
        """Construct the converter.

        Args:
            adh_model (:obj:`adhparam.adh_model.AdhModel`): Param representation of the model.
            query: Query object for communicating with SMS.
            hotstarts (:obj:`list` of :obj:`adhparam.hot_start_data_set`): The initial conditions datasets.
            vessel_list (:obj: `adhparam.vessels.VesselList`): VesselList object containing vessel data.

        """
        self.adh_model = adh_model
        self.query = query
        self.hotstarts = hotstarts
        self.vessel_list = vessel_list
        # key = string id, value = mat id (from .3dm)
        self.mat_strings = {}
        # key = string id, value = mesh node id
        self.node_strings = {}
        # Arc definitions organized as follows to allow for unordered cards in the file:
        # key = string id, value = {seg1: seg2}
        self.mid_strings = {}
        self.edge_strings = {}
        self.old_mid_string_ids = []
        self.bc_points = []
        self.bc_points_ts_id = []
        self.bc_points_ts_id_to_node_id = {}
        # key = mesh id, value = coverage point
        self.bc_nodes = {}
        self.next_bc_node_id = 1
        # key = string id, value = coverage arc/point/material id
        self.string_to_cov_arc_id = {}
        self.string_to_cov_pt_id = {}
        self.string_to_cov_mat_id = {}
        self.nodestring_id_to_arc_string_id = {}
        self.have_mesh = self.adh_model.mesh.nodes is not None
        self.sim_data = None
        self.sim_uuid = ''
        # Remaining members only filled in if we read the mesh.
        self.element_to_mat_id = {}
        self.bc_cov = None
        self.bc_cov_uuid = ''
        self.output_cov = None
        self.output_cov_uuid = ''
        self.vessel_cov = None
        self.vessel_cov_uuid = ''
        # Get the XMS temp directory
        self.comp_dir = ''
        self.temp_mesh_file = ''
        self.cogrid = None
        self.point_string_snap = {}
        self._file_to_fullpath = {}

    def get_bc_node(self, mesh_id, is_node=False) -> Point:
        """Get the boundary condition node.

        Args:
            mesh_id (int): The id of the mesh node.
            is_node (bool): True if the point should be a node of the coverage.

        Returns:
            A Point of the coverage correlating to the mesh node.
        """
        if self.have_mesh:
            if mesh_id in self.bc_nodes:
                return self.bc_nodes[mesh_id]
            result = self.adh_model.mesh.nodes.query(f'ID == {mesh_id}')
            if not result.empty:
                pt = Point(x=result.iloc[0]['X'], y=result.iloc[0]['Y'], z=result.iloc[0]['Z'])
                if is_node:
                    pt.id = self.next_bc_node_id
                    self.next_bc_node_id += 1
                self.bc_nodes[mesh_id] = pt
                return pt
        return Point(0.0, 0.0, 0.0)

    def add_node_boundaries(self, cov_arcs):
        """Add the disjoint node boundary conditions.

        Args:
            cov_arcs (list): A list of created arc objects that will be compared against and added to.

        Returns:
            A list of single point objects that will be turned into disjoint points of the coverage.
        """
        # build up sets of arc points
        string_to_unique_points = {}
        for string_id, string_points in self.edge_strings.items():
            string_to_unique_points[string_id] = set()
            for pt1, pt2 in string_points.items():
                string_to_unique_points[string_id].add(pt1)
                string_to_unique_points[string_id].add(pt2)
        for string_id, string_points in self.mid_strings.items():
            string_to_unique_points[string_id] = set()
            for pt1, pt2 in string_points.items():
                string_to_unique_points[string_id].add(pt1)
                string_to_unique_points[string_id].add(pt2)
        # figure out whether we need to build a set of points or not
        cov_pts = []
        for string_id, string in self.node_strings.items():
            is_existing_arc = False
            self.point_string_snap[string_id] = 'Point snap'
            if len(string) > 1:
                string_set = set(string)
                for arc_string_id, arc_string_points in string_to_unique_points.items():
                    # check for same arc point locations
                    if string_set == arc_string_points:
                        is_existing_arc = True
                        self.string_to_cov_arc_id[string_id] = self.string_to_cov_arc_id[arc_string_id]
                        self.nodestring_id_to_arc_string_id[string_id] = arc_string_id
                        if arc_string_id in self.edge_strings:
                            self.point_string_snap[string_id] = 'Edgestring snap'
                        else:
                            self.point_string_snap[string_id] = 'Midstring snap'
                        break
                if is_existing_arc:
                    continue
                last_idx = len(string) - 1
                # Build the API arc
                arc_id = len(cov_arcs) + 1
                start_node = self.get_bc_node(string[0], True)
                end_node = self.get_bc_node(string[last_idx], True)
                vertices = [self.get_bc_node(idx) for idx in range(1, last_idx)]
                arc = Arc(feature_id=arc_id, start_node=start_node, end_node=end_node, vertices=vertices)
                self.string_to_cov_arc_id[string_id] = arc_id
                cov_arcs.append(arc)
            else:
                self.string_to_cov_pt_id[string_id] = []
                # Build the API Point
                pt = self.get_bc_node(string[0], True)
                self.string_to_cov_pt_id[string_id].append(pt.id)
                cov_pts.append(pt)
        return cov_pts

    def add_arc_string_boundaries(self, cov_arcs, strings):
        """Add the edge/mid string arc boundary conditions.

        This is to support arc definition cards that are not in order (EGS, MDS). The AdH documentation
        strongly discourages writing the arc segments out of order, but it is allowed. Note the loops
        required to find start/end node if order is not determinate.

        Returns:
            (:obj:`list` of :obj:`xms.data_objects.parameters.Arc`): Created arcs

        """
        for string_id, string in strings.items():
            # Find the start node
            start_node = None
            pt2s = string.values()
            for pt1 in string:
                # Look for a node on this string that is not the destination of any segment definition. If we find
                #   one of those, we know it is the start node. If that condition is never true, I am assuming a loop.
                if pt1 not in pt2s:
                    start_node = pt1
                    break  # Found the start node
                if start_node is None:
                    # If this is a closed loop, the edges better be in order. So, by default we will assume the first
                    #   encountered point is the start node.
                    start_node = pt1

            # Find the end node
            end_node = start_node  # Best starting place, in case this arc is a loop.
            for pt2 in pt2s:
                # Look for destination nodes that never appear as segment start locations. If we find one, that is the
                #   arc's end node. Otherwise, I am assuming a loop, and the start node is the end node.
                if pt2 not in string:
                    end_node = pt2
                    break

            # Build the API arc
            arc_id = len(cov_arcs) + 1
            self.string_to_cov_arc_id[string_id] = arc_id
            do_start_node = self.get_bc_node(start_node, True)
            do_end_node = self.get_bc_node(end_node, True)
            arc = Arc(feature_id=arc_id, start_node=do_start_node, end_node=do_end_node)
            vert_id = string[start_node]
            vertices = []
            while vert_id != end_node:
                vertices.append(self.get_bc_node(vert_id))
                if vert_id in string:
                    vert_id = string[vert_id]
                else:
                    vert_id = end_node  # Done, exit the loop
            arc.vertices = vertices
            cov_arcs.append(arc)
        return cov_arcs

    def build_bc_coverage(self):
        """Build the boundary condition coverage geometry and populate dialog widgets.

        """
        # Load all the boundary condition and friction cards
        _TYPE_IDX = 0  # noqa
        _ID_IDX = 1  # noqa
        _GEOM_IDX_1 = 2  # noqa
        _GEOM_IDX_2 = 3  # noqa
        for row in self.adh_model.boundary_conditions.boundary_strings.itertuples(index=False, name=None):
            # row[_TYPE_IDX] == boundary string type
            if row[_TYPE_IDX] == 'EGS':  # Edge string
                # row = (type, string id, segment_pt1, segment_pt2
                if row[_ID_IDX] not in self.edge_strings:
                    self.edge_strings[row[_ID_IDX]] = {}
                self.edge_strings[row[_ID_IDX]][row[_GEOM_IDX_1]] = row[_GEOM_IDX_2]
            elif row[_TYPE_IDX] == 'MTS':  # Material string
                # row = (type, string id, material id (from .3dm))
                self.mat_strings[row[_ID_IDX]] = row[_GEOM_IDX_1]
            elif row[_TYPE_IDX] == 'NDS':
                # row = (type, string id, node id (from .3dm))
                if row[_ID_IDX] not in self.node_strings:
                    self.node_strings[row[_ID_IDX]] = []
                self.node_strings[row[_ID_IDX]].append(row[_GEOM_IDX_1])
            elif row[_TYPE_IDX] == 'MDS':
                # row = (type, string id, segment_pt1, segment_pt2
                if row[_ID_IDX] not in self.mid_strings:
                    self.mid_strings[row[_ID_IDX]] = {}
                self.mid_strings[row[_ID_IDX]][row[_GEOM_IDX_1]] = row[_GEOM_IDX_2]
                self.old_mid_string_ids.append(row[_ID_IDX])
            # else:
            #     if row[_ID_IDX] not in self.bc_points:
            #         self.bc_points[row[_ID_IDX]] = {}

        cov_arcs = []
        self.add_arc_string_boundaries(cov_arcs, self.edge_strings)
        self.add_arc_string_boundaries(cov_arcs, self.mid_strings)  # Edge and mid strings are all the same to SMS.
        cov_pts = self.add_node_boundaries(cov_arcs)
        self._add_wind_boundaries()
        cov_pts.extend(self.bc_points)
        cov_name = self.adh_model.mesh.name + ' - Boundary Conditions'
        self.bc_cov_uuid = str(uuid4())
        self.bc_cov = Coverage(name=cov_name, uuid=self.bc_cov_uuid)
        self.bc_cov.set_points(cov_pts)
        self.bc_cov.arcs = cov_arcs
        self.bc_cov.complete()

    def build_output_coverage(self, locations):
        """Build the output coverage geometry.

        Args:
            locations: Mesh node locations.
        """
        nodal_output = self.adh_model.model_control.output_control.nodal_output
        next_point_id = 1
        output_points = []
        for node_id in nodal_output['NODE']:
            node_index = node_id - 1
            if 0 <= node_index < len(locations):
                location = locations[node_index]
                output_points.append(Point(location[0], location[1], 0.0, next_point_id))
                next_point_id += 1

        if len(output_points) != 0:
            cov_name = self.adh_model.mesh.name + ' - Output'
            self.output_cov_uuid = str(uuid4())
            self.output_cov = Coverage(name=cov_name, uuid=self.output_cov_uuid)
            self.output_cov.set_points(output_points)
            self.output_cov.complete()

    def _get_cell_stream(self):
        """Returns the cell stream.

        Returns:
            Returns a tuple with the cell stream as a list and the material ids as a list and a bool that is True if 3D.
        """
        cell_stream = []
        is_3d = False
        if self.adh_model.mesh.elements is not None:
            all_elems = self.adh_model.mesh.elements.sort_values('ID')
            elem_id = all_elems['ID'].values.tolist()
            node_0 = all_elems['NODE_0'].values.tolist()
            node_1 = all_elems['NODE_1'].values.tolist()
            node_2 = all_elems['NODE_2'].values.tolist()
            mat_2d = all_elems['MATERIAL_ID'].values.tolist()
        else:
            elem_id = []
            node_0 = []
            node_1 = []
            node_2 = []
            mat_2d = []
        if self.adh_model.mesh.elements_3d is not None:
            all_3d_elems = self.adh_model.mesh.elements_3d.sort_values('ID')
            elem_id_3d = all_3d_elems['ID'].values.tolist()
            node_0_3d = all_3d_elems['NODE_0'].values.tolist()
            node_1_3d = all_3d_elems['NODE_1'].values.tolist()
            node_2_3d = all_3d_elems['NODE_2'].values.tolist()
            node_3_3d = all_3d_elems['NODE_3'].values.tolist()
            mat_3d = all_3d_elems['MATERIAL_ID'].values.tolist()
        else:
            elem_id_3d = []
            node_0_3d = []
            node_1_3d = []
            node_2_3d = []
            node_3_3d = []
            mat_3d = []
        if len(elem_id_3d) > 0 and len(elem_id) > 0:
            is_3d = True
            mat = []
            index_2d = 0
            index_3d = 0
            for _ in range(len(elem_id) + len(elem_id_3d)):
                if elem_id[index_2d] < elem_id_3d[index_3d]:
                    cell_stream.extend(
                        [
                            XmUGrid.cell_type_enum.TRIANGLE, 3, node_0[index_2d] - 1, node_1[index_2d] - 1,
                            node_2[index_2d] - 1
                        ]
                    )
                    mat.append(mat_2d[index_2d])
                    index_2d += 1
                else:
                    cell_stream.extend(
                        [
                            XmUGrid.cell_type_enum.TETRA, 4, node_0_3d[index_3d] - 1, node_1_3d[index_3d] - 1,
                            node_2_3d[index_3d] - 1, node_3_3d[index_3d] - 1
                        ]
                    )
                    mat.append(mat_3d[index_3d])
                    index_3d += 1
        elif len(elem_id_3d) > 0:
            is_3d = True
            # Currently, this assumes that all elements are tetra.
            for n0, n1, n2, n3 in zip(node_0_3d, node_1_3d, node_2_3d, node_3_3d):
                # change node ids to 0-based indexes
                cell_stream.extend([XmUGrid.cell_type_enum.TETRA, 4, n0 - 1, n1 - 1, n2 - 1, n3 - 1])
            mat = mat_3d
        else:
            # Currently, this assumes that all elements are triangles.
            for n0, n1, n2 in zip(node_0, node_1, node_2):
                # change node ids to 0-based indexes
                cell_stream.extend([XmUGrid.cell_type_enum.TRIANGLE, 3, n0 - 1, n1 - 1, n2 - 1])
            mat = mat_2d
        return cell_stream, mat, is_3d

    def build_mesh(self):
        """Builds a UGrid of the mesh.

        Returns:
            A list of material ids parallel to the cells.
        """
        cell_stream, materials, is_3d = self._get_cell_stream()
        all_nodes = self.adh_model.mesh.nodes.sort_values('ID')
        node_x = all_nodes['X'].values.tolist()
        node_y = all_nodes['Y'].values.tolist()
        node_z = all_nodes['Z'].values.tolist()
        nodes = [(x, y, z) for x, y, z in zip(node_x, node_y, node_z)]
        xmugrid = XmUGrid(nodes, cell_stream)
        co_builder = UGridBuilder()
        if is_3d:
            co_builder.set_is_3d()
        else:
            co_builder.set_is_2d()
        co_builder.set_ugrid(xmugrid)
        self.cogrid = co_builder.build_grid()
        self.temp_mesh_file = os.path.join(XmEnv.xms_environ_process_temp_directory(), str(uuid4()) + '.xmc')
        self.cogrid.write_to_file(self.temp_mesh_file, True)
        return materials, is_3d

    def add_hotstarts(self, mesh_uuid, temp_files):
        """
        Add the initial conditions datasets to the mesh and link up dataset selectors in the Model Control dialog.
        """
        datasets = []
        hot_starts = {}
        for idx, hotstart in enumerate(self.hotstarts):
            dset_uuid = str(uuid4())
            if 'S' in hotstart.values.columns:  # Scalar depth initial conditions
                num_components = 1
                values = [hotstart.values.S.values]
            else:  # Vector velocity initial conditions
                num_components = 2
                values = np.dstack((hotstart.values.VX.values, hotstart.values.VY.values))
            dataset = DatasetWriter(
                h5_filename=temp_files[idx],
                dset_uuid=dset_uuid,
                name=hotstart.name,
                geom_uuid=mesh_uuid,
                num_components=num_components
            )
            dataset.write_xmdf_dataset([0.0], values)
            datasets.append(dataset)
            hot_starts[hotstart.name.lower()] = (dset_uuid, 0)
        self.sim_data.set_hot_starts(hot_starts)
        return datasets

    def convert_and_send(self):
        """
        Convert the Param AdHModel to a format that can be sent to SMS.
        """
        self.comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components')
        self.sim_uuid = str(uuid4())
        sim = Simulation(model='AdH', name=str(self.adh_model.model_name), sim_uuid=self.sim_uuid)

        # Create the hidden simulation component
        comp_uuid, sim_mainfile, sim_comp_dir = self._create_component_mainfile('sim_comp.nc')
        self.sim_data = ModelControl(sim_mainfile)  # This will create a default model control
        self.sim_data.param_control = copy.deepcopy(self.adh_model.model_control)
        # Clear out flux strings since they are in the boundary conditions coverage component.
        self.sim_data.param_control.output_control.output_flow_strings = pd.DataFrame(data=[], columns=['CARD', 'S_ID'])
        self._move_unsupported_to_advanced()
        comp = Component(main_file=sim_mainfile, model_name='AdH', unique_name='Sim_Manager', comp_uuid=comp_uuid)

        if self.query is not None:
            self.query.add_simulation(sim, components=[comp])

        materials = []
        proj = None
        sim_series = []
        is_3d = False
        if self.have_mesh:
            materials, is_3d = self.build_mesh()
            mesh_uuid = str(uuid4())
            proj = Projection(horizontal_units='METERS', vertical_units='METERS')
            mesh = UGrid(self.temp_mesh_file, name=self.adh_model.mesh.name, uuid=mesh_uuid, projection=proj)
            if not is_3d:
                # Need to have node locations for these
                self.build_bc_coverage()
                self.build_output_coverage(self.cogrid.ugrid.locations)

            # add the mesh to the Query
            if self.query is not None:
                self.query.add_ugrid(mesh)
                self.query.link_item(self.sim_uuid, mesh_uuid)
                temp_files = [io_util.temp_filename() for _ in self.hotstarts]
                datasets = self.add_hotstarts(mesh_uuid, temp_files)
                for dataset in datasets:
                    self.query.add_dataset(dataset)

            self.sim_data.domain_uuid = mesh_uuid
            oc_ts_id = self.adh_model.model_control.output_control.oc_time_series_id
            max_time_ts_id = self.adh_model.model_control.time_control.max_time_step_size_time_series
            if oc_ts_id > 0:
                sim_series.append(oc_ts_id)
                self.sim_data.time_series[oc_ts_id] = self.adh_model.boundary_conditions.time_series[oc_ts_id]
            if max_time_ts_id > 0:
                sim_series.append(max_time_ts_id)
                # self.sim_data.time_series[max_time_ts_id] = \
                #     self.adh_model.boundary_conditions.time_series[max_time_ts_id]
            for old_series_id, old_series in self.adh_model.boundary_conditions.time_series.items():
                if old_series.series_type == 'SERIES AWRITE':
                    sim_series.append(old_series_id)
                    self.sim_data.time_series[old_series_id] = \
                        self.adh_model.boundary_conditions.time_series[old_series_id]
                    self.sim_data.info.attrs['os_time_series'] = old_series_id
                    # break
                if old_series.series_type == 'SERIES DT':
                    sim_series.append(old_series_id)
                    self.sim_data.time_series[old_series_id] = \
                        self.adh_model.boundary_conditions.time_series[old_series_id]
                    self.sim_data.info.attrs['dt_time_series'] = old_series_id
        self.sim_data.commit()

        # Duplicate the transport material information to the sediment transport material information
        # Then we can remove the ids that don't belong and preserve the data that belongs at two different places
        for mat_id in self.adh_model.boundary_conditions.material_properties:
            self.adh_model.boundary_conditions.material_properties[mat_id].sediment_transport_properties = \
                self.adh_model.boundary_conditions.material_properties[mat_id].transport_properties

        trans_comp_uuid, constituent_old_to_new = self._get_transport_component()
        sed_trans_comp_uuid, sed_constituent_old_to_new = self._get_sediment_transport_component(constituent_old_to_new)

        if is_3d:
            mat_cov = mat_comp = mat_series = None
        else:
            # Next line changes our TRT, transport properties from a list of 6 to 1...
            mat_cov, mat_comp, mat_series = self._get_material_coverage(
                materials, proj, trans_comp_uuid, constituent_old_to_new
            )
        if mat_cov and self.query is not None:
            self.query.add_coverage(mat_cov, model_name='AdH', coverage_type='Materials', components=[mat_comp])
            self.query.link_item(self.sim_uuid, mat_cov.uuid)

        if is_3d:
            sed_mat_cov = sed_mat_comp = None
        else:
            sed_mat_cov, sed_mat_comp = self._get_sediment_material_coverage(
                materials, proj, sed_trans_comp_uuid, sed_constituent_old_to_new
            )
        if sed_mat_cov and self.query is not None:
            self.query.add_coverage(
                sed_mat_cov, model_name='AdH', coverage_type='Sediment Materials', components=[sed_mat_comp]
            )
            self.query.link_item(self.sim_uuid, sed_mat_cov.uuid)

        # Add the boundary condition coverage to the Context if we built it.
        if self.bc_cov:
            filter_frame = self.adh_model.boundary_conditions.boundary_strings.CARD.ne('MTS')
            bc_strings = self.adh_model.boundary_conditions.boundary_strings[filter_frame].copy(deep=True)
            if not isinstance(bc_strings, pd.DataFrame):
                raise Exception(f"Wrong type! {type(bc_strings)}")
            bc_strings.sort_values('ID', inplace=True)
            id_list = list(bc_strings.ID.unique())
            id_dict = {bc_id: bc_id for idx, bc_id in enumerate(id_list)}  # string id to component id
            # change the component ids for nodestring that got turned into arcs
            for ns_id, arc_string_id in self.nodestring_id_to_arc_string_id.items():
                id_dict[ns_id] = id_dict[arc_string_id]
            next_comp_id = max(id_dict.values()) + 1 if len(id_dict) else 0 + 1
            # bc_strings.replace({'ID': id_dict}, inplace=True)
            bc_strings['ID_0'] = -1
            bc_strings['ID_1'] = -1
            bc_strings.fillna(-1, inplace=True)
            bc_strings.drop_duplicates(inplace=True)

            bc_comp_uuid, bc_mainfile, bc_comp_dir = self._create_component_mainfile('bc_comp.nc')
            bc_data = BcIO(bc_mainfile)  # This will create a default model control
            bc_data.bc.boundary_strings = bc_strings
            self._set_bc_transport(bc_data, id_dict, constituent_old_to_new, sed_constituent_old_to_new)
            bc_data.info.attrs['transport_uuid'] = trans_comp_uuid
            bc_data.info.attrs['sediment_uuid'] = sed_trans_comp_uuid
            bc_data.info.attrs['next_transport_id'] = next_comp_id
            bc_data.info.attrs['next_diversion_id'] = next_comp_id
            tran_ids = bc_data.transport_assignments.TRAN_ID.tolist()
            tran_ids.extend(bc_data.sediment_assignments.TRAN_ID.tolist())
            div_ids = bc_data.sediment_diversions.DIV_ID.tolist()

            next_bc_id = self._set_bc_solution_controls(bc_data, id_dict, id_list, next_comp_id)
            filter_sd = self.adh_model.boundary_conditions.stage_discharge_boundary.S_ID.isin(id_list)
            sd = self.adh_model.boundary_conditions.stage_discharge_boundary[filter_sd].copy(deep=True)
            # sd.replace({'S_ID': id_dict}, inplace=True)
            bc_data.bc.stage_discharge_boundary = sd
            bc_time_series = copy.deepcopy(self.adh_model.boundary_conditions.time_series)
            for mat_time_series in mat_series:
                bc_time_series.pop(mat_time_series)
            for sim_time_series in sim_series:
                if sim_time_series in bc_time_series:
                    bc_time_series.pop(sim_time_series)
            bc_data.bc.time_series = bc_time_series

            # This is for the Wind TS points only
            wind_comp_ids = []
            for ts_id in self.bc_points_ts_id:
                pt_id = self.bc_points_ts_id_to_node_id[ts_id]
                id_dict[next_comp_id] = next_comp_id
                wind_comp_ids.append(next_comp_id)
                self.string_to_cov_pt_id[next_bc_id] = pt_id

                values = ['DB', 'WND', next_bc_id, ts_id, -1, -1]

                # pt_id is the feature object id (from xms coverage)
                # bc_id is used in "STRING_ID"
                # comp_id is the index of the pandas dataframe
                #   (a comp_id of 2, means this row of data (in df2) will go
                #    into the solution_controls dataframe at index 2)
                df2 = pd.DataFrame(
                    [values],
                    index=[next_comp_id],
                    columns=["CARD", "CARD_2", "STRING_ID", "XY_ID_0", "XY_ID_1", "XY_ID_2"]
                )
                bc_data.bc.solution_controls = concat_df(bc_data.bc.solution_controls, df2)

                next_comp_id += 1
                next_bc_id += 1

            bc_data.info.attrs['next_comp_id'] = next_comp_id
            bc_data.info.attrs['next_bc_id'] = next_bc_id
            nb_out_bc_ids = bc_data.nb_out.BC_ID.tolist()
            nb_out_out_ids = bc_data.nb_out.OUT_COMP_ID.tolist()
            nb_out_in_ids = bc_data.nb_out.IN_COMP_ID.tolist()
            comp_to_nb_out_bc = {comp_id: bc_id for bc_id, comp_id in zip(nb_out_bc_ids, nb_out_out_ids)}
            comp_to_nb_out_bc.update({comp_id: bc_id for bc_id, comp_id in zip(nb_out_bc_ids, nb_out_in_ids)})

            filter_fric = self.adh_model.boundary_conditions.friction_controls.STRING_ID.isin(id_list)
            fric = self.adh_model.boundary_conditions.friction_controls[filter_fric].copy(deep=True)
            # Set whether the friction is for a mid-string or an edge string
            old_mid_strings = pd.Series(data=self.old_mid_string_ids)
            fric.loc[fric['REAL_05'].isin(old_mid_strings), 'REAL_05'] = 1
            fric.replace({'REAL_05': {None: 0}}, inplace=True)
            # fric.replace({'STRING_ID': id_dict}, inplace=True)
            fric.fillna(float('NaN'), inplace=True)
            bc_data.bc.friction_controls = fric
            bc_data.info.attrs['next_friction_id'] = next_comp_id
            bc_data.info.attrs['next_flux_id'] = next_comp_id
            comp_id_rows = []
            fric_ids = bc_data.bc.friction_controls.STRING_ID.tolist()
            flux_string_ids = self.adh_model.model_control.output_control.output_flow_strings.S_ID.tolist()
            flux = []
            flux_ids = []
            for flux_string in flux_string_ids:
                mid = 0
                edge = 1
                if flux_string in self.old_mid_string_ids:
                    mid = 1
                    edge = 0
                flux.append([id_dict[flux_string], 1, edge, mid])
                flux_ids.append(id_dict[flux_string])
            bc_data.flux = pd.DataFrame(flux, columns=['ID', 'IS_FLUX', 'EDGESTRING', 'MIDSTRING'])

            solution_control_strings = set(self.adh_model.boundary_conditions.solution_controls.STRING_ID.values)
            snap_map = {'NDS': 'Point snap', 'EGS': 'Edgestring snap', 'MDS': 'Midstring snap'}  # MTS filtered out
            bc_string_cards = bc_strings.CARD.values.tolist()
            snap_type = []
            snap_bc_ids = []
            processed_comp_ids = set()

            for idx, (string_id, comp_id) in enumerate(id_dict.items()):
                if comp_id in processed_comp_ids:
                    continue
                if comp_id in wind_comp_ids:
                    bc_id = string_id
                elif string_id in solution_control_strings:
                    bc_id = comp_to_nb_out_bc[comp_id] if comp_id in comp_to_nb_out_bc else comp_id
                elif comp_id in flux_ids:
                    bc_id = string_id
                else:
                    continue
                fric_id = comp_id if comp_id in fric_ids else -1
                flux_id = comp_id if comp_id in flux_ids else -1
                tran_id = comp_id if comp_id in tran_ids else -1
                div_id = comp_id if comp_id in div_ids else -1
                comp_id_rows.append([comp_id, fric_id, flux_id, tran_id, div_id, bc_id])
                processed_comp_ids.add(comp_id)
                if len(bc_string_cards) > idx:
                    snap_type.append(snap_map[bc_string_cards[idx]])
                    snap_bc_ids.append(bc_id)
            bc_data.comp_id_to_ids = pd.DataFrame(comp_id_rows, columns=bc_data.comp_id_columns)
            bc_data.info.attrs['cov_uuid'] = self.bc_cov_uuid
            bc_data.snap_types = pd.DataFrame(data={'ID': snap_bc_ids, 'SNAP': snap_type})
            bc_data.commit()

            bc_comp = Component(
                main_file=bc_mainfile, unique_name='BcConceptual', comp_uuid=bc_comp_uuid, model_name='AdH'
            )
            bc_comp.uuid = bc_comp_uuid

            if self.query is not None:
                self.query.add_coverage(
                    self.bc_cov, model_name='AdH', coverage_type='Boundary Conditions', components=[bc_comp]
                )
                self.query.link_item(taker_uuid=self.sim_uuid, taken_uuid=self.bc_cov_uuid)

            # fix string IDs to start at 1 and count up
            # old_string_id_to_new = {}

            # rebuild id_dict with new string IDs

            # bc_strings.replace({'ID': id_dict}, inplace=True)
            # sd.replace({'S_ID': id_dict}, inplace=True)
            # nb_out_rows.replace({'STRING_ID': id_dict}, inplace=True)
            # nb_out_rows.replace({'XY_ID_0': id_dict}, inplace=True)
            # other_rows.replace({'STRING_ID': id_dict}, inplace=True)

            bc_arc_att_ids, bc_arc_comp_ids = self._get_component_and_attribute_ids(id_dict, self.string_to_cov_arc_id)
            bc_pt_att_ids, bc_pt_comp_ids = self._get_component_and_attribute_ids(id_dict, self.string_to_cov_pt_id)
            next_id = max(bc_arc_comp_ids) + 1 if len(bc_arc_comp_ids) else 1
            arc_comp_ids = set(bc_arc_comp_ids)
            for i in range(len(bc_pt_comp_ids)):
                if bc_pt_comp_ids[i] in arc_comp_ids:
                    bc_pt_comp_ids[i] = next_id
                    next_id += 1

            id_file = os.path.join(bc_comp_dir, xms.adh.components.display.BC_ARC_INITIAL_ATT_ID_FILE)
            write_display_option_ids(id_file, bc_arc_att_ids)
            id_file = os.path.join(bc_comp_dir, xms.adh.components.display.BC_ARC_INITIAL_COMP_ID_FILE)
            write_display_option_ids(id_file, bc_arc_comp_ids)

            id_file = os.path.join(bc_comp_dir, xms.adh.components.display.BC_POINT_INITIAL_ATT_ID_FILE)
            write_display_option_ids(id_file, bc_pt_att_ids)
            id_file = os.path.join(bc_comp_dir, xms.adh.components.display.BC_POINT_INITIAL_COMP_ID_FILE)
            write_display_option_ids(id_file, bc_pt_comp_ids)

        self._add_output_coverage()
        self._add_vessel_coverage()

        # Add the model control widget values to the Context.
        if self.query is not None:
            self.query.send()

    def _get_transport_component(self):
        """Creates a transport constituents component if transport constituents were found on import.

        Returns:
            A tuple consisting of a string of the new component UUID and a dictionary mapping old constituent ids as
            found in the file to new constituent ids.
        """
        # Get transport constituents, if applicable
        trans_comp_uuid = ''
        constituent_old_to_new = {}
        con_props = self.adh_model.model_control.constituent_properties
        if con_props.salinity or con_props.temperature or con_props.vorticity or \
                len(con_props.general_constituents.index) > 0:
            trans_comp_uuid, trans_mainfile, trans_comp_dir = \
                self._create_component_mainfile('transport_constituents_comp.nc')
            trans_data = TransportConstituentsIO(trans_mainfile)  # This will create a default component data
            trans_data.param_control = con_props
            if con_props.salinity:
                constituent_old_to_new[con_props.salinity_id] = 1
            if con_props.temperature:
                constituent_old_to_new[con_props.temperature_id] = 2
            if con_props.vorticity:
                constituent_old_to_new[con_props.vorticity_id] = 3
            user_cons = []
            old_ids = con_props.general_constituents['ID'].tolist()
            concentrations = con_props.general_constituents['CONC'].tolist()
            # Adding 4 to index to skip ids for salinity, temperature, and vorticity
            for idx, (old_id, conc) in enumerate(zip(old_ids, concentrations)):
                new_id = idx + 4
                constituent_old_to_new[old_id] = new_id
                user_cons.append([new_id, f'Constituent {int(old_id)}', conc])
            trans_data.user_constituents = pd.DataFrame(data=user_cons, columns=['ID', 'NAME', 'CONC']).to_xarray()
            trans_data.commit()
            trans_comp = Component(
                main_file=trans_mainfile,
                name='Transport Constituents',
                unique_name='TransportConstituents',
                comp_uuid=trans_comp_uuid,
                model_name='AdH'
            )
            if self.query is not None:
                self.query.add_component(trans_comp)
        return trans_comp_uuid, constituent_old_to_new

    def _get_sediment_transport_component(self, trans_constituent_old_to_new):
        """Creates a sediment transport constituents component, if sediment transport constituents were found on import.

        Args:
            trans_constituent_old_to_new (dict): maps old constituent ids as found in the file; needed to ensure that
            the new ids used for sediment do not collide

        Returns:
            A tuple consisting of a string of the new component UUID and a dictionary mapping old constituent ids as
            found in the file to new constituent ids.
        """
        # Get sediment transport constituents, if applicable
        trans_comp_uuid = ''
        constituent_old_to_new = {}
        con_props = self.adh_model.model_control.sediment_constituent_properties
        if len(con_props.sand.index) > 0 or len(con_props.clay.index) > 0:
            trans_comp_uuid, trans_mainfile, trans_comp_dir = \
                self._create_component_mainfile('sediment_constituents_comp.nc')
            trans_data = SedimentConstituentsIO(trans_mainfile)  # This will create a default component data
            trans_data.param_control = con_props
            sand_old_ids = con_props.sand['ID'].tolist()
            clay_old_ids = con_props.clay['ID'].tolist()
            new_id = self._get_new_sed_id(0, trans_constituent_old_to_new)
            for old_id in sand_old_ids:
                constituent_old_to_new[old_id] = new_id
                new_id = self._get_new_sed_id(new_id, trans_constituent_old_to_new)
            for old_id in clay_old_ids:
                constituent_old_to_new[old_id] = new_id
                new_id = self._get_new_sed_id(new_id, trans_constituent_old_to_new)
            trans_data.param_control.sand.replace({'ID': constituent_old_to_new}, inplace=True)
            trans_data.param_control.clay.replace({'ID': constituent_old_to_new}, inplace=True)
            trans_data.param_control.sand['ID'] = trans_data.param_control.sand['ID'].astype(dtype='Int64')
            trans_data.param_control.clay['ID'] = trans_data.param_control.clay['ID'].astype(dtype='Int64')
            # For each row in Clay and Sand DataFrames, set NAME column if empty
            con_props.clay['NAME'] = con_props.clay.apply(
                lambda row: row['NAME'] if row['NAME'] else f"Clay {int(row['ID'])}", axis=1
            )
            con_props.sand['NAME'] = con_props.sand.apply(
                lambda row: row['NAME'] if row['NAME'] else f"Sand {int(row['ID'])}", axis=1
            )
            trans_data.info.attrs['next_constituent_id'] = new_id
            trans_data.commit()
            trans_comp = Component(
                main_file=trans_mainfile,
                name='Sediment Constituents',
                unique_name='SedimentConstituents',
                comp_uuid=trans_comp_uuid,
                model_name='AdH'
            )
            if self.query is not None:
                self.query.add_component(trans_comp)
        return trans_comp_uuid, constituent_old_to_new

    def _get_new_sed_id(self, current_id, constituent_old_to_new):
        """Creates a sediment transport constituents component, if sediment transport constituents were found on import.

        Args:
            current_id (int): current id that was last used
            constituent_old_to_new (dict): maps old constituent ids as found in the file; needed to ensure that the
            new ids used for sediment do not collide

        Returns:
            new_id (int): next available id that is not used
        """
        new_id = current_id + 1
        used_id_list = list(constituent_old_to_new.values())
        while new_id in used_id_list:
            new_id += 1
        return new_id

    def _set_bc_solution_controls(self, bc_data, id_dict, id_list, next_comp_id):
        """Sets the solution controls section of the boundary conditions component.

        Args:
            bc_data (BcIO): The boundary condition component data to set the solutions controls section on.
            id_dict (dict): A dictionary with string id the file to the new string id that will be the component id.
            id_list (list): A list of integer string ids used in boundary conditions.
            next_comp_id (int): The next component id.

        Returns:
            The next bc id.
        """
        off_filter = self.adh_model.boundary_conditions.solution_controls.CARD.eq('OFF')
        out_filter = self.adh_model.boundary_conditions.solution_controls.CARD_2.eq('OUT')
        trn_filter = self.adh_model.boundary_conditions.solution_controls.CARD_2.eq('TRN')
        id_filter = self.adh_model.boundary_conditions.solution_controls.STRING_ID.isin(id_list)
        out_id2_filter = self.adh_model.boundary_conditions.solution_controls.XY_ID_0.isin(id_list)
        nb_out_filter = out_filter & id_filter & out_id2_filter
        nb_out_rows = self.adh_model.boundary_conditions.solution_controls[nb_out_filter].copy(deep=True)
        other_filter = ~off_filter & ~out_filter & ~trn_filter & id_filter
        other_rows = self.adh_model.boundary_conditions.solution_controls[other_filter].copy(deep=True)
        # nb_out_rows.replace({'STRING_ID': id_dict}, inplace=True)
        # nb_out_rows.replace({'XY_ID_0': id_dict}, inplace=True)
        # other_rows.replace({'STRING_ID': id_dict}, inplace=True)
        nb_out_rows.rename(
            columns={
                'STRING_ID': 'OUT_COMP_ID',
                'XY_ID_0': 'IN_COMP_ID',
                'XY_ID_1': 'SERIES_ID'
            }, inplace=True
        )
        nb_out_rows.drop(columns=['CARD', 'CARD_2', 'XY_ID_2'])
        next_bc_id = next_comp_id + len(nb_out_rows['OUT_COMP_ID'].tolist())
        nb_out_rows['BC_ID'] = [bc_id for bc_id in range(next_comp_id, next_bc_id)]
        bc_data.nb_out = nb_out_rows
        bc_data.bc.solution_controls = concat_df(bc_data.bc.solution_controls, other_rows)
        bc_data.bc.solution_controls.fillna(-1, inplace=True)
        labels = ['STRING_ID', 'XY_ID_0', 'XY_ID_1', 'XY_ID_2']
        for x in range(1, len(labels)):
            bc_data.bc.solution_controls[labels[x]] = \
                bc_data.bc.solution_controls[labels[x]].astype(dtype='Int64')
        return next_bc_id

    def _set_bc_transport(self, bc_data, id_dict, constituent_id_dict, sediment_constituent_id_dict):
        """Sets the solution controls section of the boundary conditions component.

        Args:
            bc_data (BcIO): The transport boundary condition component data to set the transport values on.
            id_dict (dict): A dictionary with string id in the file to the new string id that will be the component id.
            constituent_id_dict (dict): A dictionary with constituent id in the file to the new constituent id.
            sediment_constituent_id_dict (dict): A dictionary with constituent id in the file to the new sediment
            constituent id.
        """
        trn_filter = self.adh_model.boundary_conditions.solution_controls.CARD_2.eq('TRN')
        assignments = []
        sediment_assignments = []
        used_transport = []
        used_sediment = []
        for row_data in self.adh_model.boundary_conditions.solution_controls[trn_filter].itertuples():
            card_type = row_data.CARD
            string_id = int(row_data.STRING_ID)
            con_id = int(row_data.XY_ID_0)
            series_id = int(row_data.XY_ID_1)
            if card_type == 'DB':
                row_type = 'Dirichlet'
            elif card_type == 'NB':
                row_type = 'Natural'
            elif card_type == 'EQ':
                row_type = 'Equilibrium'
            else:
                row_type = 'Natural'
            snap_type = 'Point snap'
            if string_id in self.point_string_snap:
                snap_type = self.point_string_snap[string_id]
            if con_id in constituent_id_dict:
                used_transport.append(string_id)
                assignments.append(
                    [
                        id_dict[string_id], constituent_id_dict[con_id], f'Constituent {con_id}', row_type, series_id,
                        snap_type
                    ]
                )
            elif con_id in sediment_constituent_id_dict:
                used_sediment.append(string_id)
                sediment_assignments.append(
                    [
                        id_dict[string_id], sediment_constituent_id_dict[con_id], f'Constituent {con_id}', row_type,
                        series_id, snap_type
                    ]
                )
        if used_transport:
            uses_transport = self._get_use_transport_values(id_dict, used_transport)
            bc_data.transport_assignments = pd.DataFrame(
                assignments, columns=['TRAN_ID', 'CONSTITUENT_ID', 'NAME', 'TYPE', 'SERIES_ID', 'SNAPPING']
            )
            bc_data.uses_transport = pd.DataFrame(uses_transport, columns=['TRAN_ID', 'USES_TRANSPORT'])
        if used_sediment:
            uses_sediment = self._get_use_transport_values(id_dict, used_sediment)
            bc_data.sediment_assignments = pd.DataFrame(
                sediment_assignments, columns=['TRAN_ID', 'CONSTITUENT_ID', 'NAME', 'TYPE', 'SERIES_ID', 'SNAPPING']
            )
            bc_data.uses_sediment = pd.DataFrame(uses_sediment, columns=['TRAN_ID', 'USES_SEDIMENT'])

    @staticmethod
    def _get_use_transport_values(id_dict, used_transport):
        """Gets the lists of lists of transport values for creating a use transport dataframe.

        Args:
            id_dict (dict): A dictionary with string id in the file to the new string id that will be the component id.
            used_transport (list): A list of string ids in the file that were being used for transport.

        Returns:
            A list of lists with the inner list containing a new component id and a bool that is True if transport
            is used.
        """
        comp_id_to_used = {}
        for old_id, comp_id in id_dict.items():
            if comp_id in comp_id_to_used:
                comp_id_to_used[comp_id] = comp_id_to_used[comp_id] or old_id in used_transport
            else:
                comp_id_to_used[comp_id] = old_id in used_transport
        uses_transport = [[comp_id, is_used] for comp_id, is_used in comp_id_to_used.items()]
        return uses_transport

    @staticmethod
    def _get_component_and_attribute_ids(id_dict, string_id_to_att_id):
        """Gets the component and attribute ids from the string ids that were in the bc file.

        Args:
            id_dict (dict): A dictionary with string id from the file to the component id.
            string_id_to_att_id (dict): A dictionary of the string id from the file to the attribute id.

        Returns:
            Returns two parallel lists, a list of the attribute ids, and a list of the new component ids.
        """
        component_ids = []
        attribute_ids = []
        for string_id, attribute_id in string_id_to_att_id.items():
            if string_id in id_dict:
                if isinstance(attribute_id, list):
                    attribute_ids.extend(attribute_id)
                    comp_id = id_dict[string_id]
                    component_ids.extend([comp_id for _ in range(len(attribute_id))])
                else:
                    component_ids.append(id_dict[string_id])
                    attribute_ids.append(attribute_id)
        return attribute_ids, component_ids

    def _create_component_mainfile(self, mainfile):
        """Creates a directory for a mainfile for a component in a new UUID named folder in the temp area.

        Args:
            mainfile (str): The default filename for the mainfile.

        Returns:
            A tuple consisting of the new component UUID, the absolute path for the new mainfile, and the new
            component UUID directory.
        """
        comp_uuid = str(uuid4())
        comp_dir = os.path.join(self.comp_dir, comp_uuid)
        os.makedirs(comp_dir, exist_ok=True)
        new_mainfile = os.path.join(comp_dir, mainfile)
        self._file_to_fullpath[mainfile] = new_mainfile  # for testing
        return comp_uuid, new_mainfile, comp_dir

    def _get_material_coverage(self, cell_materials, projection, transport_uuid, constituent_old_to_new):
        """Gets the material coverage with its parts.

        Args:
           cell_materials (list): A list of materials parallel to the cells.
           projection (Projection): The projection of the project.
           transport_uuid (str): UUID of the transport constituents component.
           constituent_old_to_new (dict): Dictionary of constituent id in the file to the new constituent id.

        Returns:
            A xms.data_objects Coverage of the material geometry, a xms.data_objects Component, a list of used time
                series ids.
        """
        if not self.have_mesh:
            return None, None, None
        cov_name = self.adh_model.mesh.name + ' - Materials'
        cov_builder = GridCellToPolygonCoverageBuilder(self.cogrid, cell_materials, projection, cov_name)
        new_cov_geom = cov_builder.create_polygons_and_build_coverage()

        mat_comp_uuid, mat_mainfile, mat_comp_dir = self._create_component_mainfile('mat_comp.nc')
        mat_data = MaterialsIO(mat_mainfile)
        id_dict = {}
        for mat_id, mat in self.adh_model.boundary_conditions.material_properties.items():
            new_mat_id = mat_data.materials.add_material(mat)
            mat_data.materials.material_properties[new_mat_id] = mat
            display = mat_data.materials.material_display[new_mat_id]
            ColorList.get_next_color_and_texture(new_mat_id, display)
            old_transport = copy.deepcopy(mat.transport_properties)
            transport = {}
            for con_id, trans_prop in old_transport.items():
                if con_id in constituent_old_to_new:
                    new_con_id = constituent_old_to_new[con_id]
                    new_prop = MaterialTransportProperties()
                    new_prop.refinement_tolerance = trans_prop.refinement_tolerance
                    new_prop.turbulent_diffusion_rate = trans_prop.turbulent_diffusion_rate
                    transport[new_con_id] = new_prop
            mat_data.materials.material_properties[new_mat_id].transport_properties = transport
            id_dict[mat_id] = new_mat_id

        # add material friction
        filter_frame = self.adh_model.boundary_conditions.boundary_strings.CARD.eq('MTS')
        bc_strings = self.adh_model.boundary_conditions.boundary_strings[filter_frame].copy(deep=True)
        if not isinstance(bc_strings, pd.DataFrame):
            raise Exception(f"Wrong type! {type(bc_strings)}")
        bc_strings.sort_values('ID', inplace=True)
        id_list = list(bc_strings.ID.unique())
        filter_fric = self.adh_model.boundary_conditions.friction_controls.STRING_ID.isin(id_list)
        fric = self.adh_model.boundary_conditions.friction_controls[filter_fric].copy(deep=True)
        old_mat_strings = pd.Series(data=list(self.mat_strings.keys()))
        fric = fric[fric['STRING_ID'].isin(old_mat_strings)]
        fric.replace({'STRING_ID': id_dict}, inplace=True)
        mat_data.materials.friction = fric

        nb_filter = self.adh_model.boundary_conditions.solution_controls.CARD.eq('NB')
        vel_filter = self.adh_model.boundary_conditions.solution_controls.CARD_2.isin(['OVL', 'VEL'])
        id_filter = self.adh_model.boundary_conditions.solution_controls.STRING_ID.isin(id_list)
        other_filter = nb_filter & vel_filter & id_filter
        other_rows = self.adh_model.boundary_conditions.solution_controls[other_filter]
        old_mat_ids = list(other_rows.STRING_ID)
        mat_series = list(other_rows.XY_ID_0)
        for old_mat_id, series_id in zip(old_mat_ids, mat_series):
            new_mat_id = id_dict[old_mat_id]
            mat_data.materials.material_use_meteorological[new_mat_id] = True
            mat_data.materials.material_meteorological_curve[new_mat_id] = new_mat_id
            mat_data.materials.time_series[new_mat_id] = self.adh_model.boundary_conditions.time_series[series_id]

        nb_filter = self.adh_model.boundary_conditions.solution_controls.CARD.eq('FR')
        vel_filter = self.adh_model.boundary_conditions.solution_controls.CARD_2.isin(['SRF'])
        id_filter = self.adh_model.boundary_conditions.solution_controls.STRING_ID.isin(id_list)
        other_filter = nb_filter & vel_filter & id_filter
        other_rows = self.adh_model.boundary_conditions.solution_controls[other_filter]
        old_mat_ids = list(other_rows.STRING_ID)
        mat_series = list(other_rows.XY_ID_0)
        for old_mat_id, series_id in zip(old_mat_ids, mat_series):
            new_mat_id = id_dict[old_mat_id]
            mat_data.materials.friction_use_seasonal[new_mat_id] = True
            mat_data.materials.friction_seasonal_curve[new_mat_id] = new_mat_id
            mat_data.materials.time_series[new_mat_id] = self.adh_model.boundary_conditions.time_series[series_id]

        mat_data.info.attrs['cov_uuid'] = new_cov_geom.uuid
        mat_data.info.attrs['transport_uuid'] = transport_uuid
        mat_data.use_transport = 1 if transport_uuid else 0
        mat_data.commit()

        mat_poly_att_ids, mat_poly_comp_ids = self._get_component_and_attribute_ids(
            id_dict, cov_builder.dataset_polygon_ids
        )

        id_file = os.path.join(mat_comp_dir, xms.adh.components.display.MAT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, mat_poly_att_ids)
        id_file = os.path.join(mat_comp_dir, xms.adh.components.display.MAT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, mat_poly_comp_ids)

        mat_comp = Component(
            main_file=mat_mainfile, unique_name='MaterialConceptual', model_name='AdH', comp_uuid=mat_comp_uuid
        )
        return new_cov_geom, mat_comp, mat_series

    def _get_sediment_material_coverage(self, cell_materials, projection, transport_uuid, constituent_old_to_new):
        """Gets the sediment material coverage with its parts.

        Args:
           cell_materials (list): A list of materials parallel to the cells.
           projection (Projection): The projection of the project.
           transport_uuid (str): UUID of the transport constituents component.
           constituent_old_to_new (dict): Dictionary of constituent id in the file to the new constituent id.

        Returns:
            A xms.data_objects Coverage of the material geometry, a xms.data_objects Component. Can return None, None.
        """
        # If there are no sediment constituents, we assume that there is no sediment materials coverage.
        if self.adh_model.model_control.sediment_constituent_properties.sand.empty and \
                self.adh_model.model_control.sediment_constituent_properties.clay.empty:
            return None, None

        # If there is no mesh, we assume that there is no sediment materials coverage.
        if not self.have_mesh:
            return None, None

        # Find materials that have sediment specific properties.
        sed_props = self.adh_model.model_control.sediment_properties
        mat_list = sed_props.bed_layer_grain_fractions['MATERIAL_ID'].tolist()
        if len(mat_list) > 0:
            used_mats = set(cell_materials)  # MB SBA card for all materials
        else:
            used_mats = sed_props.material_consolidation['MATERIAL_ID'].tolist()
            used_mats.extend(sed_props.material_cohesive_bed['MATERIAL_ID'].tolist())
            used_mats.extend(sed_props.material_bed_layers['MATERIAL_ID'].tolist())
            used_mats.extend(sed_props.material_displacement_off['MATERIAL_ID'].tolist())
            used_mats.extend(sed_props.material_local_scour['MATERIAL_ID'].tolist())
            used_mats.extend(sed_props.material_diffusion['MATERIAL_ID'].tolist())
            used_mats = set(used_mats)
        sed_materials = [mat if mat in used_mats else 0 for mat in cell_materials]

        # Create the coverage.
        cov_name = self.adh_model.mesh.name + ' - Sediment Materials'
        # If there are no materials that use sediment, there should only be one big polygon per section of the domain.
        # A coverage with the global sediment properties should still be returned.
        cov_builder = GridCellToPolygonCoverageBuilder(self.cogrid, sed_materials, projection, cov_name)
        new_cov_geom = cov_builder.create_polygons_and_build_coverage()

        # Create the component.
        mat_comp_uuid, mat_mainfile, mat_comp_dir = self._create_component_mainfile('sed_mat_comp.nc')
        mat_data = SedimentMaterialsIO(mat_mainfile)
        id_dict = {}
        sorted_used_materials = list(used_mats)
        sorted_used_materials.sort()
        self._set_global_sediment_properties(sed_props, mat_data)
        self._add_global_sediment_material(sed_props, constituent_old_to_new, mat_data)
        self._add_sediment_materials(sed_props, sorted_used_materials, constituent_old_to_new, mat_data, id_dict)
        mat_data.info.attrs['cov_uuid'] = new_cov_geom.uuid
        mat_data.info.attrs['sediment_transport_uuid'] = transport_uuid
        mat_data.commit()

        # Get component ids to send to XMS.
        mat_poly_att_ids, mat_poly_comp_ids = self._get_component_and_attribute_ids(
            id_dict, cov_builder.dataset_polygon_ids
        )

        id_file = os.path.join(mat_comp_dir, xms.adh.components.display.SED_MAT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, mat_poly_att_ids)
        id_file = os.path.join(mat_comp_dir, xms.adh.components.display.SED_MAT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, mat_poly_comp_ids)

        # Get the component data object to send to XMS.
        mat_comp = Component(
            main_file=mat_mainfile, unique_name='SedimentMaterialConceptual', model_name='AdH', comp_uuid=mat_comp_uuid
        )
        return new_cov_geom, mat_comp

    @staticmethod
    def _set_global_sediment_properties(sed_props, mat_data):
        """Adds the properties for the global unassigned material.

        Args:
            sed_props (SedimentProperties): The adhparam sediment properties.
            mat_data (SedimentMaterialsIO): The properties for all sediment materials.
        """
        mat_data.info.attrs['cohesive_settling_velocity_method'] = \
            SedimentMaterialsIO.CSV_OPTIONS[sed_props.cohesive_settling]
        mat_data.info.attrs['wind_wave_shear_method'] = \
            SedimentMaterialsIO.WWS_OPTIONS[sed_props.wind_wave_stress]
        mat_data.info.attrs['noncohesive_suspended_method'] = \
            SedimentMaterialsIO.NSE_OPTIONS[sed_props.suspended_entrainment + 1]
        mat_data.info.attrs['noncohesive_bedload_method'] = \
            SedimentMaterialsIO.NBE_OPTIONS[sed_props.bedload_entrainment + 1]
        mat_data.info.attrs['noncohesive_hiding_method'] = \
            SedimentMaterialsIO.HID_OPTIONS[sed_props.hiding_factor]

        mat_data.info.attrs['a_csv'] = sed_props.cohesive_settling_a
        mat_data.info.attrs['b_csv'] = sed_props.cohesive_settling_b
        mat_data.info.attrs['m_csv'] = sed_props.cohesive_settling_m
        mat_data.info.attrs['n_csv'] = sed_props.cohesive_settling_n

        mat_data.info.attrs['critical_shear_sand'] = sed_props.critical_shear_sand
        mat_data.info.attrs['critical_shear_clay'] = sed_props.critical_shear_clay

        mat_data.info.attrs['hiding_factor'] = sed_props.hiding_factor_exponent

        mat_data.info.attrs['use_sediment_infiltration_factor'] = sed_props.use_infiltration_factor
        mat_data.info.attrs['sediment_infiltration_factor'] = sed_props.infiltration_factor

        mat_data.info.attrs['bed_layer_assignment_protocol'] = sed_props.bed_layer_thickness_protocol

    @staticmethod
    def _add_global_sediment_material(sed_props, constituent_old_to_new, mat_data):
        """Adds the properties for the global unassigned material.

        Args:
            sed_props (SedimentProperties): The adhparam sediment properties.
            constituent_old_to_new (dict): A dictionary of old integer constituent ids to new.
            mat_data (SedimentMaterialsIO): The properties for all sediment materials.
        """
        new_mat = mat_data.materials[SedimentMaterialsIO.UNASSIGNED_MAT]
        bed_layer_grain = \
            sed_props.bed_layer_grain_fractions.loc[
                sed_props.bed_layer_grain_fractions['MATERIAL_ID'] == SedimentMaterialsIO.UNASSIGNED_MAT]
        bed_layer_grain = bed_layer_grain.drop(columns=['MATERIAL_ID'])
        bed_layer_grain = bed_layer_grain.replace({'CONSTITUENT_ID': constituent_old_to_new})
        bed_layer_grain = bed_layer_grain.rename(
            columns={
                'BED_LAYER_ID': 'layer_id',
                'CONSTITUENT_ID': 'constituent_id',
                'FRACTION': 'fraction'
            }
        )
        new_mat.constituents = bed_layer_grain

        consolidation = sed_props.global_consolidation.drop(columns=['MP', 'CARD'])
        consolidation = consolidation.rename(
            columns={
                'TIME_ID': 'time_id',
                'ELAPSED_TIME': 'elapsed_time',
                'POROSITY': 'porosity',
                'CRITICAL_SHEAR': 'critical_shear',
                'EROSION_CONSTANT': 'erosion_constant',
                'EROSION_EXPONENT': 'erosion_exponent'
            }
        )
        new_mat.consolidation = consolidation

        cohesive_bed = sed_props.global_cohesive_bed.drop(columns=['MP', 'CARD'])
        cohesive_bed = cohesive_bed.rename(
            columns={
                'BED_LAYER_ID': 'layer_id',
                'POROSITY': 'porosity',
                'CRITICAL_SHEAR': 'critical_shear',
                'EROSION_CONSTANT': 'erosion_constant',
                'EROSION_EXPONENT': 'erosion_exponent'
            }
        )
        bed_layers = sed_props.global_bed_layers.drop(columns=['CARD'])
        bed_layers = bed_layers.rename(columns={'BED_LAYER_ID': 'layer_id', 'THICKNESS': 'thickness'})
        new_mat.bed_layers = pd.merge(bed_layers, cohesive_bed, on='layer_id', how='outer')
        new_mat.bed_layers.fillna(0.0, inplace=True)

        if not cohesive_bed.empty:
            mat_data.info.attrs['use_cohesive_bed_layers'] = 1
        if not consolidation.empty:
            mat_data.info.attrs['use_consolidation'] = 1

    def _add_sediment_materials(self, sed_props, sorted_used_materials, constituent_old_to_new, mat_data, id_dict):
        """Adds sediment materials with properties.

        Args:
            sed_props (SedimentProperties): The adhparam sediment properties.
            sorted_used_materials (list): List of material ids in the file with sediment attributes.
            constituent_old_to_new (dict): A dictionary of old integer constituent ids to new.
            mat_data (SedimentMaterialsIO): The properties for all sediment materials.
            id_dict (dict): A dictionary of file material ids to new sediment material ids.
        """
        for mat_id in sorted_used_materials:
            # Don't try to add the global options here.
            if mat_id == SedimentMaterialsIO.UNASSIGNED_MAT:
                continue
            new_mat_id = mat_data.add_material()
            new_mat = mat_data.materials[new_mat_id]
            ColorList.get_next_color_and_texture(new_mat_id, new_mat.display)
            consolidation = \
                sed_props.material_consolidation.loc[sed_props.material_consolidation['MATERIAL_ID'] == mat_id]
            cohesive_bed = sed_props.material_cohesive_bed.loc[sed_props.material_cohesive_bed['MATERIAL_ID'] == mat_id]
            bed_layers = sed_props.material_bed_layers.loc[sed_props.material_bed_layers['MATERIAL_ID'] == mat_id]
            displacement_off = \
                sed_props.material_displacement_off.loc[sed_props.material_displacement_off['MATERIAL_ID'] == mat_id]
            local_scour = sed_props.material_local_scour.loc[sed_props.material_local_scour['MATERIAL_ID'] == mat_id]
            diffusion = sed_props.material_diffusion.loc[sed_props.material_diffusion['MATERIAL_ID'] == mat_id]
            bed_layer_grain = \
                sed_props.bed_layer_grain_fractions.loc[sed_props.bed_layer_grain_fractions['MATERIAL_ID'] == mat_id]
            if not consolidation.empty:
                consolidation = consolidation.drop(columns=['MP', 'CARD', 'MATERIAL_ID'])
                consolidation = consolidation.rename(
                    columns={
                        'TIME_ID': 'time_id',
                        'ELAPSED_TIME': 'elapsed_time',
                        'POROSITY': 'porosity',
                        'CRITICAL_SHEAR': 'critical_shear',
                        'EROSION_CONSTANT': 'erosion_constant',
                        'EROSION_EXPONENT': 'erosion_exponent'
                    }
                )
                new_mat.consolidation = consolidation
                new_mat.consolidation_override = True

            self._set_bed_layers_for_material(
                sed_props, bed_layers, cohesive_bed, bed_layer_grain, constituent_old_to_new, new_mat
            )
            if not displacement_off.empty:
                new_mat.displacement_off = True
            if not local_scour.empty:
                new_mat.local_scour = False
            if not diffusion.empty:
                new_mat.use_bedload_diffusion = True
                new_mat.bedload_diffusion = diffusion['DIFFUSION'][0]
            id_dict[mat_id] = new_mat_id

        for bc_mat_id, mat in self.adh_model.boundary_conditions.material_properties.items():
            bc_mat_index = bc_mat_id
            old_transport = mat.sediment_transport_properties
            transport = {}
            for con_id, trans_prop in old_transport.items():
                if con_id in constituent_old_to_new:
                    new_con_id = constituent_old_to_new[con_id]
                    new_prop = MaterialTransportProperties()
                    new_prop.refinement_tolerance = trans_prop.refinement_tolerance
                    new_prop.turbulent_diffusion_rate = trans_prop.turbulent_diffusion_rate
                    transport[new_con_id] = new_prop
                mat_data.materials[bc_mat_index].sediment_material_properties = transport

    @staticmethod
    def _set_bed_layers_for_material(
            sed_props, bed_layers, cohesive_bed, bed_layer_grain, constituent_old_to_new, new_mat
    ):
        """Sets the bed layers for a material if any overrides are present.

        Args:
            sed_props (SedimentProperties): The adhparam sediment properties.
            bed_layers (DataFrame): Bed layer overrides for this material.
            cohesive_bed (DataFrame): Bed layer cohesive properties for this material.
            bed_layer_grain (DataFrame): Bed layer grain size distribution per bed layer for this material.
            constituent_old_to_new (dict): A dictionary of old integer constituent ids to new.
            new_mat (SedimentMaterial): The sediment material to add bed layers to.
        """
        is_material_cohesive = True
        if bed_layers.empty and not cohesive_bed.empty:
            # Copy the bed layer overrides from the global. While this can happen, I think this is a fringe case.
            is_material_cohesive = False
            bed_layers = sed_props.global_bed_layers
        if not bed_layers.empty:
            bed_layer_grain = bed_layer_grain.drop(columns=['MATERIAL_ID'])
            bed_layer_grain = bed_layer_grain.replace({'CONSTITUENT_ID': constituent_old_to_new})
            bed_layer_grain = bed_layer_grain.rename(
                columns={
                    'BED_LAYER_ID': 'layer_id',
                    'CONSTITUENT_ID': 'constituent_id',
                    'FRACTION': 'fraction'
                }
            )
            new_mat.constituents = bed_layer_grain

            if is_material_cohesive:
                bed_layers = bed_layers.drop(columns=['CARD', 'MATERIAL_ID'])
            else:
                bed_layers = bed_layers.drop(columns=['CARD'])
            bed_layers = bed_layers.rename(columns={'BED_LAYER_ID': 'layer_id', 'THICKNESS': 'thickness'})
            new_mat.bed_layer_override = True
            if not cohesive_bed.empty:
                cohesive_bed = cohesive_bed.drop(columns=['MP', 'CARD', 'MATERIAL_ID'])
                new_mat.bed_layer_cohesive_override = True
            else:
                # copy columns from the global
                cohesive_bed = sed_props.global_cohesive_bed.drop(columns=['MP', 'CARD'])

            if not cohesive_bed.empty:
                cohesive_bed = cohesive_bed.rename(
                    columns={
                        'BED_LAYER_ID': 'layer_id',
                        'POROSITY': 'porosity',
                        'CRITICAL_SHEAR': 'critical_shear',
                        'EROSION_CONSTANT': 'erosion_constant',
                        'EROSION_EXPONENT': 'erosion_exponent'
                    }
                )
                new_mat.bed_layers = pd.merge(bed_layers, cohesive_bed, on='layer_id')
            else:
                ids = bed_layers.layer_id.tolist()
                thickness = bed_layers.thickness.tolist()
                new_mat.bed_layers = pd.DataFrame(
                    data=[[layer, thick, 0.0, 0.0, 0.0, 0.0] for layer, thick in zip(ids, thickness)],
                    columns=[
                        'layer_id', 'thickness', 'porosity', 'critical_shear', 'erosion_constant', 'erosion_exponent'
                    ]
                )

    def _move_unsupported_to_advanced(self):
        """Move unsupported features in adhparam to the advanced cards section of the simulation."""
        unsupported_cards = []
        fake_file = io.StringIO()
        file_io.write_breach_control_cards(fake_file, self.adh_model.boundary_conditions)
        file_io.write_structure_cards(fake_file, self.adh_model.boundary_conditions)
        unsupported_cards.extend(filter(None, fake_file.getvalue().split('\n')))
        cards = pd.DataFrame(data=unsupported_cards, columns=["CARD_LINE"])
        self.sim_data.param_control.advanced_cards.advanced_cards = \
            concat_df(self.sim_data.param_control.advanced_cards.advanced_cards, cards)

    def _add_output_coverage(self):
        """Add the output coverage."""
        if self.output_cov:
            output_comp_uuid, output_main_file, bc_comp_dir = self._create_component_mainfile('output_comp.nc')
            output_data = OutputData(output_main_file)

            output_data.info.attrs['cov_uuid'] = self.output_cov_uuid
            output_data.commit()

            output_comp = Component(
                main_file=output_main_file, unique_name='OutputComponent', comp_uuid=output_comp_uuid, model_name='AdH'
            )
            output_comp.uuid = output_comp_uuid

            if self.query is not None:
                self.query.add_coverage(
                    self.output_cov, model_name='AdH', coverage_type='Output', components=[output_comp]
                )
                self.query.link_item(taker_uuid=self.sim_uuid, taken_uuid=self.output_cov_uuid)

    def _add_vessel_coverage(self):
        """Add the vessel coverage."""
        if not self.vessel_list or len(self.vessel_list.vessels) == 0:
            return
        vessel_coverages = []
        for vessel in self.vessel_list.vessels:
            builder = CoverageComponentBuilder(VesselComponent, 'Vessel', self.cogrid)

            # Draw vessel segments
            self._draw_vessel_segments(builder, vessel)

            coverage, component, keywords = builder.build()
            vessel_coverages.append(coverage.uuid)

            # Set vessel coverage attributes
            self._set_vessel_coverage_attributes(component, vessel)

            cov_comp = Component(
                main_file=component.main_file,
                unique_name='Vessel',
                comp_uuid=component.uuid,
                model_name='AdH'
            )
            self.query.add_coverage(
                coverage,
                model_name='Adh',
                coverage_type='Vessel',
                components=[cov_comp],
                component_keywords=keywords
            )
        if self.sim_data:
            self.sim_data.vessel_uuids = xr.Dataset({'uuids': vessel_coverages})
            self.sim_data.commit()

    @staticmethod
    def _draw_vessel_segments(builder: CoverageComponentBuilder, vessel: Vessel):
        """Draws vessel segments on the coverage and assigns segment attributes.

        Args:
            builder: The CoverageComponentBuilder.
            vessel: The Vessel object.
        """
        current_point = (vessel.initial_x_coordinate, vessel.initial_y_coordinate, 0)

        for _, segment in vessel.sailing_segments.iterrows():
            next_point = (segment.END_X_COORDINATE, segment.END_Y_COORDINATE, 0)
            point_string = [current_point, next_point]

            section = get_model().arc_parameters
            # Set the correct group active
            segment_number = 'first_segment' if segment.SEGMENT_NUMBER == 1 else 'segment'
            group = section.group(segment_number)
            group.is_active = True

            # Assign segment attributes
            group.parameter('segment_type').value = 'Line'
            group.parameter('time_of_arrival').value = segment.MOTION_VALUE

            if segment.SEGMENT_TYPE == 1:
                group.parameter('segment_type').value = 'Arc'
                group.parameter('x-center').value = segment.ARC_CENTER_X
                group.parameter('y-center').value = segment.ARC_CENTER_Y
                group.parameter('turn_direction').value = 'Left' if segment.TURN_DIRECTION == 1 else 'Right'

            values = section.extract_values()
            comp_id = builder.data.add_feature(TargetType.arc, values, segment_number)
            builder.add_point_string(point_string, comp_id)

            current_point = next_point

    @staticmethod
    def _set_vessel_coverage_attributes(component: VesselComponent, vessel: Vessel):
        """Sets the vessel and propeller coverage attributes from a Vessel object.

        Args:
            component: The component on which to attach the attributes.
            vessel: The vessel object containing the attributes.
        """
        section = get_model().model_parameters

        # Vessel Parameters
        section.group('vessel').is_active = True
        section.group('vessel').parameter('initial_speed').value = vessel.initial_speed
        section.group('vessel').parameter('vessel_draft').value = vessel.vessel_draft
        section.group('vessel').parameter('vessel_length').value = vessel.vessel_length
        section.group('vessel').parameter('vessel_width').value = vessel.vessel_width
        section.group('vessel').parameter('bow_length_ratio').value = vessel.bow_length_ratio
        section.group('vessel').parameter('stern_length_ratio').value = vessel.stern_length_ratio
        section.group('vessel').parameter('bow_draft_ratio').value = vessel.bow_draft_ratio
        section.group('vessel').parameter('stern_draft_ratio').value = vessel.stern_draft_ratio

        # Propeller Parameters
        if vessel.propeller is not None:
            section.group('propeller').is_active = True
            section.group('propeller').parameter('propeller_type').value = vessel.propeller.propeller_type
            section.group('propeller').parameter('propeller_diameter').value = vessel.propeller.propeller_diameter
            section.group('propeller').parameter('center_distance').value = vessel.propeller.propeller_center_distance
            section.group('propeller').parameter('tow_boat_length').value = vessel.propeller.tow_boat_length
            section.group('propeller').parameter('distance_to_stern').value = vessel.propeller.distance_to_stern

        values = section.extract_values()
        component.data.vessel_values = values
        component.data.commit()

    def _add_wind_boundaries(self):
        """Add the wind boundaries."""
        for _, ts in self.adh_model.boundary_conditions.time_series.items():
            if ts.series_type == 'SERIES WIND':
                self.bc_points.append(Point(ts.x_location, ts.y_location, 0.0, self.next_bc_node_id))
                self.bc_points_ts_id.append(ts.series_id)
                self.bc_points_ts_id_to_node_id[ts.series_id] = self.next_bc_node_id
                self.next_bc_node_id += 1

    def _wind_series_count(self) -> int:
        """Returns the number of wind time series."""
        wind_count = 0
        for _, ts in self.adh_model.boundary_conditions.time_series.items():
            if ts.series_type == 'SERIES WIND':
                wind_count += 1
        return wind_count
