"""A dialog for assigning distributions of sediment constituents per bed layer."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas
from matplotlib.figure import Figure
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.qx_table_view import QxTableView

# 4. Local modules
from xms.adh.gui.bed_layer_constituent_dialog_ui import Ui_BedLayerConstituentDialog
from xms.adh.gui.distribution_plot_model import DistributionPlotModel
from xms.adh.gui.distribution_table_model import DistributionTableModel
import xms.adh.gui.resources.adhqrc  # noqa F401


class BedLayerConstituentDialog(XmsDlg):
    """A dialog for setting global sediment options."""
    def __init__(self, win_cont, sediment_data):
        """Allows the user to edit global sediment options.

        This dialog assumes that sediment_data is a pandas DataFrame with columns: 'layer_id', 'constituent_id',
        'fraction', 'ID', 'NAME', 'GRAIN_DIAMETER'. The DataFrame should be sorted by 'GRAIN_DIAMETER'.

        Args:
            win_cont (QWidget): Parent window
            sediment_data (DataFrame): Data consisting of: constituent id, name, size, and distribution.
        """
        super().__init__(win_cont, 'xms.adh.gui.bed_layer_constituent_dialog')
        self.ui = Ui_BedLayerConstituentDialog()
        self.ui.setupUi(self)

        self.table = QxTableView()

        self.figure = None  # matplotlib.figure: the Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg: the FigureCanvas
        self.ax = None  # matplotlib: the Axes

        self.model = QxPandasTableModel(sediment_data)
        self.model.dataChanged.connect(self.on_data_changed)

        # Create a filter of the model for the plot
        self.plot_model = DistributionPlotModel()
        self.plot_model.setSourceModel(self.model)

        # Create a filter of the model for the table
        self.table_model = DistributionTableModel()
        self.table_model.setSourceModel(self.model)

        # Just use a fixed starting width of 300 for the table for now
        self.ui.splitter.setSizes([300, 600])
        self.ui.splitter.setChildrenCollapsible(False)
        self.ui.splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        self.ui.table_layout.insertWidget(0, self.table)
        self.plot_layout = QVBoxLayout()
        self.ui.plot_group.setLayout(self.plot_layout)
        self.setup_plot()
        self.table.setModel(self.table_model)
        self.table.verticalHeader().hide()
        self.dbl_validator = QxDoubleValidator()
        self.dbl_validator.setBottom(0.0)
        self.dbl_validator.setDecimals(10)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.table.setItemDelegateForColumn(2, self.edit_delegate)

        self.update_total_label()
        self.ui.normalize_button.pressed.connect(self.on_normalize)

        self.setWindowTitle('Bed Layer Constituents')

    def add_series(self):
        """Adds the XY line series to the plot."""
        if not self.ax:
            self.ax = self.figure.add_subplot(111)
        self.add_line_series('Grain size distribution', self.plot_model, True)
        self.canvas.draw()

    def setup_plot(self):
        """Sets up the plot."""
        self.figure = Figure()
        self.figure.set_tight_layout(True)  # Frames the plots
        self.canvas = FigureCanvas(self.figure)
        self.canvas.setMinimumWidth(100)  # So user can't resize it to nothing
        self.plot_layout.addWidget(self.canvas)
        self.add_series()

    def add_line_series(self, name, model, show_axis_titles):
        """Adds an XY line series to the plot.

        Args:
            name (str): The series name.
            model(QxPandasTableModel): The model.
            show_axis_titles (bool): If True, axis titles displayed

        """
        self.ax.clear()
        self.ax.set_title(name)
        self.ax.grid(True)
        self.ax.set_xscale('log', nonpositive='clip')

        if model.rowCount() == 0:
            return

        # Add data to plot
        x_column, y_column = model.get_plot_values()
        self.ax.plot(x_column, y_column)

        # Axis titles
        if show_axis_titles:
            self.ax.set_xlabel(model.headerData(1, Qt.Horizontal))
        if show_axis_titles:
            self.ax.set_ylabel(model.headerData(0, Qt.Horizontal))

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        del top_left_index, bottom_right_index  # Unused parameters
        if self.table.pasting:
            return

        self.add_line_series('Grain size distribution', self.plot_model, True)
        self.canvas.draw()
        self.update_total_label()

    def on_normalize(self):
        """Normalizes the distribution fractions."""
        self.table_model.normalize_distribution_fraction()
        self.update_total_label()

    def update_total_label(self):
        """Updates the text of the label telling the user what the sum of the distribution fractions is."""
        self.ui.total_label.setText(f'Total: {self.table_model.get_distribution_fraction_sum():.2f}')

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Saves data from the widgets to be stored for later."""
        return super().accept()

    def reject(self):
        """Exits the dialog without saving."""
        return super().reject()
