"""A dialog for sediment constituent values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import typing
import webbrowser

# 2. Third party modules
from adhparam.sediment_constituent_properties import CLAY_COLUMN_TYPES, SAND_COLUMN_TYPES
import pandas
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.gui.sediment_constituents_dialog_ui import Ui_SedimentConstituentsDialog
from xms.adh.gui.widgets.adh_table_widget import AdhTableWidget


class NamedConstituentsRenameModel(RenameModel):
    """A model to rename header titles."""
    def __init__(self, column_names, parent=None):
        """Initializes the filter model.

        Args:
            column_names (list): The column names.
            parent (Something derived from :obj:`QObject`): The parent object.

        """
        super().__init__(column_names, parent)
        self.id_to_name = {}

    def data(self, index, role=Qt.DisplayRole) -> typing.Any:
        """Gets the data for the model.

        Args:
            index (QModelIndex): The location index in the Qt model.
            role (int): The role the data represents.
        """
        source_data = super(RenameModel, self).data(index, role)
        if index.column() == 0 and role in [Qt.EditRole, Qt.DisplayRole]:
            data_id = int(source_data)
            if data_id not in self.id_to_name:
                self.id_to_name[data_id] = 'Constituent'
            return self.id_to_name[data_id]
        else:
            return source_data

    def setData(self, index, value, role=Qt.DisplayRole) -> bool:  # noqa: N802
        """Sets the data for the model.

        Args:
            index (QModelIndex): The location index in the Qt model.
            value (typing.Any): The value to set.
            role (int): The role the data represents.
        """
        if index.column() == 0 and role == Qt.EditRole:
            source_data = super(RenameModel, self).data(index, role)
            data_id = int(source_data)
            self.id_to_name[data_id] = value
            self.dataChanged.emit(index, index, [role])
            return True
        else:
            return super(RenameModel, self).setData(index, value, role)


class SedimentConstituentsTableWidget(AdhTableWidget):
    """A table of user defined sediment constituents."""
    next_id = 0

    def __init__(self, parent, data_frame, select_col, filter_model, column_delegates):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
            select_col (int): Column to select when adding/removing rows
            filter_model (QSortFilterProxyModel): A model for sorting, filtering, and changing how the data is viewed.
            column_delegates (dict): A dictionary with column index as the key and a QStyledItemDelegate as the value.

        """
        super().__init__(parent, data_frame, select_col, filter_model, column_delegates)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        super().on_btn_add_row()
        new_index = self.model.index(row_idx, 0)
        self.model.setData(new_index, SedimentConstituentsTableWidget.next_id)
        SedimentConstituentsTableWidget.next_id += 1


class SedimentConstituentsDialog(XmsDlg):
    """A dialog for assigning sediment transport constituents."""
    def __init__(self, win_cont, data: SedimentConstituentsIO):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (QWidget): Parent window
            data (SedimentConstituentsIO): The sediment transport constituents data.

        """
        super().__init__(win_cont, 'xms.adh.gui.sediment_constituents_dialog')
        self.param_data = data.param_control
        self.data = data
        self.sand = self.data.param_control.sand.drop('NAME', axis=1)
        self.clay = self.data.param_control.clay.drop('NAME', axis=1)

        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Sediment_Transport_and_Bed_Layers'

        self.ui = Ui_SedimentConstituentsDialog()
        self.ui.setupUi(self)

        clay_rename_model, sand_rename_model = self._setup_rename_models(data)
        self._setup_delegates()
        self._setup_tables(clay_rename_model, sand_rename_model)

        self.ui.button_box.helpRequested.connect(self.help_requested)

        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def _setup_tables(self, clay_rename_model, sand_rename_model):
        """Sets up the sand and clay tables.

        Args:
            clay_rename_model (NamedConstituentsRenameModel): The rename model the clay table.
            sand_rename_model (NamedConstituentsRenameModel): The rename model the sand table.
        """
        # Add the table for sand constituents.
        SedimentConstituentsTableWidget.next_id = self.data.info.attrs['next_constituent_id']
        self.sand_table = SedimentConstituentsTableWidget(
            self, self.sand, 0, sand_rename_model, {
                1: self.edit_delegate,
                2: self.edit_delegate,
                3: self.gravity_edit_delegate,
                4: self.edit_delegate
            }
        )
        self.sand_table.default_values = [[0, 0.0, 0.0, 1.0, 0.0]]
        self.sand_table.model.set_default_values(self.sand_table.default_values)
        sand_layout = QVBoxLayout(self)
        sand_layout.addWidget(self.sand_table)
        self.ui.sand_group.setLayout(sand_layout)

        # Add the table for clay constituents.
        self.clay_table = SedimentConstituentsTableWidget(
            self, self.clay, 0, clay_rename_model, {
                1: self.edit_delegate,
                2: self.edit_delegate,
                3: self.gravity_edit_delegate,
                4: self.edit_delegate,
                5: self.edit_delegate,
                6: self.edit_delegate,
                7: self.edit_delegate,
                8: self.edit_delegate
            }
        )
        self.clay_table.default_values = [[0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0]]
        self.clay_table.model.set_default_values(self.clay_table.default_values)
        clay_layout = QVBoxLayout(self)
        clay_layout.addWidget(self.clay_table)
        self.ui.clay_group.setLayout(clay_layout)

    def _setup_delegates(self):
        """Sets up the delegates used in the tables."""
        # Set up the delegates
        self.dbl_validator = QxDoubleValidator()
        self.dbl_validator.setBottom(0.0)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        self.gravity_dbl_validator = QxDoubleValidator()
        self.gravity_dbl_validator.setBottom(1.0)
        self.gravity_edit_delegate = EditFieldValidator(self.gravity_dbl_validator)

    def _setup_rename_models(self, data):
        """Sets up rename models for use in the sand and clay tables.

        Args:
            data (SedimentConstituentsIO): The sediment transport constituents data.

        Returns:
            A tuple of NamedConstituentsRenameModels, the first being clay and the second sand.
        """
        # Set up the rename model so our column headers make sense.
        sand_rename_model = NamedConstituentsRenameModel(
            ['Constituent name', 'Concentration (ppm)', 'Grain diameter (m)', 'Specific gravity', 'Porosity'], self
        )
        clay_rename_model = NamedConstituentsRenameModel(
            [
                'Constituent name', 'Concentration (ppm)', 'Grain diameter (m)', 'Specific gravity', 'Porosity',
                'Critical shear stress\nfor erosion (τc)', 'Erosion rate constant (M)',
                'Critical shear\nfor deposition', 'Free settling velocity (m/s)'
            ], self
        )

        # Get the constituent names from the sand and clay dataframes
        sand_df = data.param_control.sand[['ID', 'NAME']]
        clay_df = data.param_control.clay[['ID', 'NAME']]
        constituent_names = pandas.concat([sand_df, clay_df], ignore_index=True)

        sand_ids = data.param_control.sand['ID'].tolist()
        all_ids = constituent_names['ID'].tolist()
        all_names = constituent_names['NAME'].tolist()
        for con_id, con_name in zip(all_ids, all_names):
            if con_id in sand_ids:
                sand_rename_model.id_to_name[con_id] = con_name
            else:
                clay_rename_model.id_to_name[con_id] = con_name
        return clay_rename_model, sand_rename_model

    def _set_data_from_widgets(self):
        """Get values from the table."""
        row_list = [[con_id, con_name] for con_id, con_name in self.sand_table.filter_model.id_to_name.items()]
        row_list.extend([[con_id, con_name] for con_id, con_name in self.clay_table.filter_model.id_to_name.items()])
        constituent_names = pandas.DataFrame(row_list, columns=['ID', 'NAME'])

        # Get the new sand and clay DataFrames
        sand = self.sand_table.model.data_frame
        clay = self.clay_table.model.data_frame

        # Add the "NAME" column to each by merging on "ID"
        sand = sand.merge(constituent_names, on='ID', how='left')
        sand = sand[SAND_COLUMN_TYPES.keys()]
        clay = clay.merge(constituent_names, on='ID', how='left')
        clay = clay[CLAY_COLUMN_TYPES.keys()]

        # Add the updated sand and clay DataFrames back to the param_control attribute
        self.data.param_control.sand = sand
        self.data.param_control.clay = clay

        # Update additional attributes
        self.data.info.attrs['next_constituent_id'] = SedimentConstituentsTableWidget.next_id

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._set_data_from_widgets()
        super().accept()


# def main():
#     """Demonstrates a simple use of the dialog."""
#     import sys
#     from PySide2.QtWidgets import (QApplication)
#     from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
#
#     data = SedimentConstituentsIO('C:/Temp/AdH_Sediment_Constituents_file.nc')
#     app = QApplication(sys.argv)
#     dialog = SedimentConstituentsDialog(None, 'AdH Sediment Transport Constituents', data)
#     if dialog.exec():
#         data.param_control = dialog.data
#         data.commit()
#
#     sys.exit(app.exec_())
#
#
# if __name__ == "__main__":
#     main()
