"""Dialog for specifying an XY series with populate buttons."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex
from PySide2.QtWidgets import QHBoxLayout, QPushButton

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules
from xms.adh.gui.populate_flow_dialog import PopulateFlowDialog


class XySeriesPopulateDialog(XySeriesEditor):
    """An XY series editor dialog with buttons for populate methods."""
    def __init__(
        self,
        data_frame,
        series_name,
        dialog_title='XY Series Editor',
        icon=None,
        parent=None,
        readonly_x=False,
        can_add_rows=True,
        stair_step=False,
        populate_flow_data=None
    ):
        """Constructor of the dialog.

        Args:
            data_frame (DataFrame): The DataFrame.
            series_name (str): The name of the series.
            dialog_title (str): The dialog title.
            icon (QIcon): Icon to use for the dialog.
            parent (QWidget): The dialog parent.
            readonly_x (bool): True to make the X column read only.
            can_add_rows (bool): True if user can add rows.
            stair_step (bool): True to make curve stair-step like.
            populate_flow_data (PopulateFlowData): Data from the last time populate was used.
        """
        super().__init__(data_frame, series_name, dialog_title, icon, parent, readonly_x, can_add_rows, stair_step)
        self.populate_flow = populate_flow_data

    def setup_ui(self, df):
        """Sets up the UI.

        Args:
            df (DataFrame): Pandas DataFrame
        """
        super().setup_ui(df)
        populate_flow_layout = QHBoxLayout()
        populate_flow_button = QPushButton()
        populate_flow_button.setText('Populate...')
        populate_flow_button.pressed.connect(self._open_populate)
        populate_flow_layout.addWidget(populate_flow_button)
        self.ui.vlay_left.addLayout(populate_flow_layout)

    def _open_populate(self):
        """Opens the populate dialog."""
        dialog = PopulateFlowDialog(self.populate_flow, self)
        if dialog.exec_():
            times, q = self.populate_flow.get_time_series()
            num_rows = self.model.rowCount()
            self.ui.tbl_xy_data.pasting = True
            if num_rows > len(times):
                self.model.removeRows(0, num_rows - len(times))
            elif num_rows < len(times):
                self.model.insertRows(0, len(times) - num_rows)
            for idx, (flow_time, flow_q) in enumerate(zip(times, q)):
                self.model.setData(self.model.index(idx, 0), flow_time)
                self.model.setData(self.model.index(idx, 1), flow_q)
            self.ui.tbl_xy_data.pasting = False
            self.ui.tbl_xy_data.model().dataChanged.emit(QModelIndex(), QModelIndex())  # Update the table view
