"""Module for the XmsData class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from functools import cached_property
from typing import Optional

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.constraint import Grid
from xms.data_objects.parameters import Coverage, Projection
from xms.gmi.data.generic_model import Section
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.gmi.dmi.xms_data import XmsData as GmiXmsData

# 4. Local modules
from xms.aeolis.data.model import get_model
from xms.aeolis.data.sim_base_data import SimData


class XmsData(GmiXmsData):
    """Interprocess communication object."""
    model_name = 'AeoLiS'
    sim_component_unique_name = 'AeolisSimComponent'
    get_model = get_model

    # PyCharm doesn't know that cached properties can be set.
    # noinspection PyPropertyAccess
    def __init__(
        self,
        query: Optional[Query] = None,
        model_control: Optional[Section] = None,
        linked_grid: Optional[Grid] = None,
        linked_grid_projection: Optional[Projection] = None,
        linked_coverages_and_components: Optional[dict[str, tuple[Coverage, CoverageBaseData]]] = None,
        process_control: Optional[Section] = None
    ):
        """
        Initialize the class.

        The process_control argument overrides the value of self.process_control. All other parameters are passed
        through to the base class.
        """
        super().__init__(query, model_control, linked_grid, linked_grid_projection, linked_coverages_and_components)

        if process_control is not None:
            self.process_control = model_control

    @cached_property
    def linked_coverage(self) -> tuple[Coverage, CoverageBaseData]:
        """The coverage and component that are linked to the simulation."""
        return super()._linked_coverage_and_component('AeolisCoverageComponent')

    @cached_property
    def process_control(self) -> Section:
        """The main-file for the current simulation."""
        assert self.model_name and self.sim_component_unique_name and self.get_model
        sim_uuid = self._sim_node.uuid
        sim_comp = self._query.item_with_uuid(
            sim_uuid, model_name=self.model_name, unique_name=self.sim_component_unique_name
        )
        sim_data = SimData(sim_comp.main_file)
        section = type(self).get_model().model_parameters
        section.restore_values(sim_data.process_values)
        return section
