"""Query implementation used when playing previously recorded XMS communication for Python tests."""
# 1. Standard python modules
from collections.abc import Iterable
import datetime
from distutils.dir_util import copy_tree
import json
import os
import shutil
import tempfile

# 2. Third party modules

# 3. Aquaveo modules
import xms.data_objects.parameters as pydop
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.api.Agent import XmsAgent
from xms.api.dmi._QueryImpl import QueryImpl
from xms.api.dmi import XmsEnvironment as xmenv
from xms.api.tree import TreeNode


class QueryPlaybackImpl(QueryImpl):
    """Query implementation used when playing previously recorded XMS communication for Python tests."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.Query, optional): The C++ object to wrap. Mutually exclusive with 'agent'
                kwarg.
            agent (xms.api.XmsAgent, optional): The XMS interprocess communication manager, mutually exclusive with
                'instance' kwarg.
            timeout (int): The interprocess communication timeout interval in milliseconds. Defaults to 5 minutes,
                which might be a little excessive. Used to have lots of script timeouts when we serialized more
                data through the pipe.
            retries (int): Number of times to retry after a failed message request. Defaults to one. I have never
                set it to anything else, and doing sometimes causes duplicate build data being created in XMS
                from import scripts.
            progress_script (bool): True if created in a progress script.
            migrate_script (bool): True if created in a migrate script.
        """
        kwargs['agent'] = XmsAgent()  # Create a dummy XmsAgent before constructing base, no XMS to communicate with.
        super().__init__(**kwargs)
        self._already_flushed = False  # True as soon as we write the baseline files
        self._pe_tree = None
        self._requested_data = {}  # Data previously retrieved and recorded from XMS, see QueryRecordImpl for details.
        self._sent_data = None
        self._reset_sent_data()
        self._playback_folder = xmenv.xms_environ_playback_folder()
        self._load_recorded_data()

    def __del__(self):
        """Write the baseline files if they don't exist yet (script never called Query.send())."""
        self._write_output_files()

    def _reset_sent_data(self):
        """Clear out the maps of sent data."""
        self._sent_data = {
            'components': 0,
            'coverages': 0,
            'coverage_components': 0,
            'datasets': 0,
            'delete_items': 0,
            'generic_coverages': 0,
            'link_items': 0,
            'model_check_errors': 0,
            'rasters': 0,
            # 'shapefiles': 0,  Lazily add this so as not to break backwards compatibility
            'rename_items': 0,
            'simulations': 0,
            'simulation_components': 0,
            'ugrid_elevation_edits': 0,
            'ugrids': 0,
            'unlink_items': 0,
        }

    def _load_recorded_data(self):
        """Load data requested from XMS during a previous recording."""
        # Read the recorded JSON file.
        with open(os.path.join(self._playback_folder, xmenv.xms_environ_playback_record_file()), 'r') as f:
            self._requested_data = json.loads(f.read())

        # Convert relative paths in recorded data to absolute.
        pe_tree_dict = self._requested_data.get('project_tree')
        self._fix_component_paths()
        self._fix_component_paths_in_tree(pe_tree_dict)
        self._fix_component_id_file_paths(self._requested_data.get('component_id_files', {}))
        self._fix_component_id_file_paths(self._requested_data.get('selected_component_id_files', {}))
        self._fix_coverage_attribute_paths()
        self._fix_coverage_paths()
        self._fix_current_and_parent_item_paths()
        self._fix_dataset_paths()
        self._fix_generic_coverage_paths()
        self._fix_gis_paths()
        self._fix_ugrid_paths()

        # Create a TreeNode object from the recorded JSON now that all the component main file paths have been fixed.
        if pe_tree_dict:
            self._pe_tree = TreeNode()
            self._pe_tree.from_dict(pe_tree_dict)

    def _fix_component_paths(self):
        """Convert relative main file paths in the recorded component dumps to absolute."""
        comp_dumps = self._requested_data.get('components', {})
        for _, comp_dict in comp_dumps.items():
            comp_dict['main_file'] = os.path.join(self._playback_folder, comp_dict['main_file'])

    def _fix_component_paths_in_tree(self, tree_node_dict):
        """Convert relative main file paths in the recorded project tree to absolute.

        Args:
            tree_node_dict (dict): The recorded project tree JSON dict.
        """
        if not tree_node_dict:
            return

        if tree_node_dict['ctype'] == 'component':
            relative_path = tree_node_dict['main_file']
            tree_node_dict['main_file'] = os.path.join(self._playback_folder, relative_path)
        for child in tree_node_dict['children']:
            self._fix_component_paths_in_tree(child)

    def _fix_component_id_file_paths(self, id_files):
        """Convert relative component id file paths to absolute in the recorded data.

        Args:
            id_files (dict): The recorded component id file data
        """
        # id_files = {comp_uuid: {cov_uuid: {target_type: (att_id_file, comp_id_file)}}}
        for _, cov_dict in id_files.items():
            for _, entity_dict in cov_dict.items():
                for target_type, file_tuple in entity_dict.items():
                    entity_dict[target_type] = (
                        os.path.join(self._playback_folder, file_tuple[0]),
                        os.path.join(self._playback_folder, file_tuple[1]),
                    )

    def _fix_coverage_attribute_paths(self):
        """Convert relative coverage attribute file paths to absolute in the recorded data."""
        cov_atts = self._requested_data.get('coverage_attributes', {})
        # cov_atts = {cov_uuid: {str_target: [att_file]}}
        for _, entity_dict in cov_atts.items():
            for target_type, att_file in entity_dict.items():
                entity_dict[target_type] = os.path.join(self._playback_folder, att_file)

    def _fix_coverage_paths(self):
        """Convert relative coverage dump file paths to absolute in the recorded data."""
        cov_dumps = self._requested_data.get('coverages', {})
        for _, cov_dict in cov_dumps.items():
            cov_dict['filename'] = os.path.join(self._playback_folder, cov_dict['filename'])

    def _fix_current_and_parent_item_paths(self):
        """Fix paths in recorded current and parent item dumps."""
        current_item = self._requested_data.get('current_item')
        if current_item:
            if 'main_file' in current_item:  # Current item was a component
                current_item['main_file'] = os.path.join(self._playback_folder, current_item['main_file'])
            elif 'filename' in current_item:  # Current item was a coverage? Don't think this can happen.
                current_item['filename'] = os.path.join(self._playback_folder, current_item['filename'])

        parent_item = self._requested_data.get('parent_item')
        if parent_item:
            if 'main_file' in parent_item:  # Current item was a component
                parent_item['main_file'] = os.path.join(self._playback_folder, parent_item['main_file'])
            elif 'filename' in parent_item:  # Current item was a coverage? Don't think this can happen.
                parent_item['filename'] = os.path.join(self._playback_folder, parent_item['filename'])

    def _fix_dataset_paths(self):
        """Convert relative dataset dump file paths to absolute in the recorded data."""
        dset_dumps = self._requested_data.get('datasets', {})
        for _, dset_dict in dset_dumps.items():
            dset_dict['h5_filename'] = os.path.join(self._playback_folder, dset_dict['h5_filename'])

    def _fix_generic_coverage_paths(self):
        """Convert relative SMS generic coverage dump file paths to absolute in the recorded data."""
        cov_dumps = self._requested_data.get('generic_coverages', {})
        for _, cov_dict in cov_dumps.items():
            cov_dict['filename'] = os.path.join(self._playback_folder, cov_dict['filename'])

    def _fix_gis_paths(self):
        """Convert relative GIS file paths to absolute in the recorded data."""
        # When we initially wrote this we only exported rasters. Shapefiles got tacked on later but work exactly the
        # same way as far as the API is concerned. Stored shapefiles in "rasters" key of dict so as not to break
        # existing stuff.
        raster_files = self._requested_data.get('rasters', {})
        for _, raster_dict in raster_files.items():
            raster_dict['filename'] = os.path.join(self._playback_folder, raster_dict['filename'])

    def _fix_ugrid_paths(self):
        """Convert relative CoGrid file paths to absolute in the recorded data."""
        ugrid_dumps = self._requested_data.get('ugrids', {})
        for _, ugrid_dict in ugrid_dumps.items():
            ugrid_dict['cogrid_file'] = os.path.join(self._playback_folder, ugrid_dict['cogrid_file'])

    def _write_output_files(self):
        """Write the recording file for playback and the sent data baseline."""
        if self._already_flushed:
            return  # Already wrote the baselines
        self._already_flushed = True
        # Write the sent data baseline.
        with open(os.path.join(self._playback_folder, xmenv.xms_environ_sent_data_out_file()), 'w') as f:
            f.write(json.dumps(self._sent_data))

    def _xms_data_from_json(self, json_dict):
        """Reconstruct json data retrieved from XMS.

        Args:
            json_dict (dict): The object's json data
        """
        item_uuid = ''
        is_component = False
        if 'comp_uuid' in json_dict:
            item_uuid = json_dict['comp_uuid']
            is_component = True
        elif 'sim_uuid' in json_dict:
            item_uuid = json_dict['sim_uuid']
        elif 'uuid' in json_dict:
            item_uuid = json_dict['uuid']
        return self._xms_data_from_uuid(item_uuid, is_component)

    def _xms_data_from_uuid(self, item_uuid, is_component):
        """Reconstruct json data retrieved from XMS given its UUID.

        Args:
            item_uuid (str): UUID of the item to reconstruct
            is_component (bool): True if the item is a component in which case the UUID may be of the owning item
        """
        if is_component:  # If we know it is a component, don't look elsewhere. Might be accessed using the parent UUID.
            return pydop.Component(**self._requested_data['components'][item_uuid])

        if item_uuid in self._requested_data['coverages']:  # Is it a coverage?
            return self._do_coverage_from_json(item_uuid)
        if item_uuid in self._requested_data['datasets']:  # Is it a dataset?
            return DatasetReader(**self._requested_data['datasets'][item_uuid])
        if item_uuid in self._requested_data['generic_coverages']:  # Is it an SMS generic coverage dump?
            return self._generic_coverage_from_json(item_uuid)
        if item_uuid in self._requested_data['rasters']:  # Is it a GIS file?
            return self._requested_data['rasters'][item_uuid]['filename']
        if item_uuid in self._requested_data['simulations']:  # Is it a simulation?
            return pydop.Simulation(**self._requested_data['simulations'][item_uuid])
        if item_uuid in self._requested_data['ugrids']:  # Is it a UGrid?
            return self._do_ugrid_from_json(item_uuid)
        # Check the other items first if we didn't specify that this item was a component. May reference hidden
        # components using their parent's UUID, which could appear in other maps.
        if item_uuid in self._requested_data['components']:  # Is it a component?
            return pydop.Component(**self._requested_data['components'][item_uuid])
        return None

    def _do_coverage_from_json(self, cov_uuid):
        """Construct a data_objects Coverage from recorded json data.

        Args:
            cov_uuid (str): UUID of the coverage to convert

        Returns:
            pydop.Coverage: See description
        """
        json_dict = self._requested_data['coverages'][cov_uuid]
        display_projection = pydop.Projection(**json_dict['projection'])
        native_projection = pydop.Projection(**json_dict['native_projection'])
        do_cov = pydop.Coverage(
            json_dict['filename'],
            json_dict['group_path'],
            name=json_dict['name'],
            uuid=json_dict['uuid'],
            projection=display_projection
        )
        # Use the C++ bindings to set the native projection. No pure Python binding as it doesn't make sense.
        do_cov._instance.SetNativeProjection(native_projection._instance)
        return do_cov

    def _generic_coverage_from_json(self, cov_uuid):
        """Construct an SMS generic coverage dump from recorded json data.

        Args:
            cov_uuid (str): UUID of the coverage to convert

        Returns:
            pydop.Coverage: See description
        """
        json_dict = self._requested_data['generic_coverages'][cov_uuid]
        display_projection = pydop.Projection(**json_dict['projection'])
        native_projection = pydop.Projection(**json_dict['native_projection'])
        dump_type = json_dict['dump_type']

        # We have a circular dependency here between xmsapi and xmscoverage. I'm assuming that if you go into
        # any of the following if/elif branches, your package is already dependent on xmscoverage, so it should
        # be installed with your package. We won't make xmscoverage a requirement of the xmsapi package.
        if dump_type == 'xms.coverage.activity':
            from xms.coverage.activity import ActivityCoverage
            cov_dump = ActivityCoverage(json_dict['filename'])
        elif dump_type == 'xms.coverage.spatial':
            from xms.coverage.spatial import SpatialCoverage
            cov_dump = SpatialCoverage(json_dict['filename'])
        elif dump_type == 'xms.coverage.spectral':
            from xms.coverage.spectral import SpectralCoverage
            cov_dump = SpectralCoverage(json_dict['filename'])
        elif dump_type == 'xms.coverage.windCoverage':
            from xms.coverage.windCoverage import WindCoverage
            cov_dump = WindCoverage(json_dict['filename'])
        else:
            return None

        # Set members on the data_objects Coverage that are not written to the file.
        cov_dump.m_cov.name = json_dict['name']
        cov_dump.m_cov.uuid = json_dict['uuid']
        cov_dump.m_cov.projection = display_projection
        cov_dump.m_cov._instance.SetNativeProjection(native_projection._instance)  # No pure Python binding
        return cov_dump

    def _do_ugrid_from_json(self, ugrid_uuid):
        """Construct a data_objects UGrid from recorded json data.

        Args:
            ugrid_uuid (str): UUID of the UGrid to convert

        Returns:
            pydop.UGrid: See description
        """
        json_dict = self._requested_data['ugrids'][ugrid_uuid]
        display_projection = pydop.Projection(**json_dict['projection'])
        native_projection = pydop.Projection(**json_dict['native_projection'])
        do_ugrid = pydop.UGrid(
            json_dict['cogrid_file'], name=json_dict['name'], uuid=json_dict['uuid'], projection=display_projection
        )
        # Use the C++ bindings to set the native projection. No pure Python binding as it doesn't make sense.
        do_ugrid._instance.SetNativeProjection(native_projection._instance)
        return do_ugrid

    # Data commonly retrieved at the root simulation/component level.
    @property
    def project_tree(self):
        """Returns the XMS project explorer tree."""
        return self._pe_tree

    @property
    def global_time(self):
        """Returns the current XMS global time."""
        zero_time = self._requested_data.get('global_time')
        if zero_time:
            try:
                zero_time = datetime.datetime.strptime(zero_time, '%Y-%m-%d %H:%M:%S.%f')
            except ValueError:  # Try without fractional seconds
                zero_time = datetime.datetime.strptime(zero_time, '%Y-%m-%d %H:%M:%S')
        return zero_time

    @property
    def global_time_settings(self):
        """Returns the current XMS global time settings as a JSON string.

        See xms.guipy.time_format.XmsTimeFormatter for more details.
        """
        return self._requested_data.get('global_time_settings', '')

    @property
    def display_projection(self):
        """Returns the current XMS display projection."""
        projection = self._requested_data.get('display_projection')
        if projection:  # JSON keys match pydop.Projection constructor kwargs
            projection = pydop.Projection(**projection)
        return projection

    @property
    def cp(self):
        """Returns a dict of GUI module name text to copy protection enable/disable flag."""
        return self._requested_data.get('copy_protection', {})

    @property
    def read_file(self):
        """Returns the filesystem path to the file being imported by the process.

        Note only applicable if process is an import script.
        """
        filename = self._requested_data.get('read_file', '')
        if filename:
            filename = os.path.join(self._playback_folder, filename)
        return filename

    # Methods for retrieving data from XMS.
    def item_with_uuid(self, item_uuid, generic_coverage=False, model_name=None, unique_name=None):
        """Retrieves an item from XMS given its UUID.

        Note:
            To get a hidden component item_uuid should be the UUID of the hidden component's owner. model_name is the
            hidden component's XML-defined model name. unique_name is the hidden component's XML-defined unique name.

        Args:
            item_uuid (str): UUID of the item
            generic_coverage (bool): True if the item is one of the dumpable generic coverages
            model_name (str): The XML model name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify unique_name if model_name provided.
            unique_name (str): The XML unique name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify model_name if unique_name provided.

        Returns:
            Pure python interface to the requested item.
        """
        return self._xms_data_from_uuid(item_uuid, model_name is not None)

    def current_item(self):
        """Returns a KEYWORD_NONE Get request for the current position in the Query Context.

        Note assuming Context position is at something with a UUID. Don't try to use if you are down at the Coverage
        Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current Context position or None on failure.
        """
        return self._xms_data_from_json(self._requested_data.get('current_item', {}))

    def current_item_uuid(self):
        """Returns the UUID of the item at the current Context position, if it has one.

        Returns:
            str: The current item's UUID or empty string if it doesn't have one.
        """
        return self._requested_data.get('current_item_uuid', '')

    def parent_item(self):
        """Returns a KEYWORD_NONE Get request for the parent of the current position in the Query Context.

        Note assuming Context position's parent is something with a UUID. Don't try to use if you are down at the
        Coverage Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current context position's parent or None on failure.
        """
        return self._xms_data_from_json(self._requested_data.get('parent_item', {}))

    def parent_item_uuid(self):
        """Returns the UUID of the current Context position's parent, if it has one.

        Returns:
            str: The UUID of the current item's parent or empty string if it doesn't have one.
        """
        return self._requested_data.get('parent_item_uuid', '')

    def named_executable_path(self, exe_name):
        """Get the path to a model executable that is defined in the XML.

        Args:
            exe_name (str): The XML-defined executable name

        Returns:
            str: Path to the named executable or empty string if not found
        """
        # Not really sure what you would do with this. It is the path to the executable relative to the XMS Python
        # installation, assuming the executable was in the XMS Python installation at the time of recording.
        return self._requested_data['named_executable_path'].get(exe_name, '')

    def load_component_ids(self, component, points, arcs, arc_groups, polygons, only_selected, delete_files):
        """Query XMS for a dump of all the current component ids of the specified entity type.

        Args:
            component (xms.components.bases.coverage_component_base.CoverageComponentBase): The coverage component to
                load id files for
            points (bool): True if you want point component id maps
            arcs (bool): True if you want arc component id maps
            arc_groups (bool): True if you want arc group component id maps
            polygons (bool): True if you want polygon component id maps
            only_selected (bool): If True, only retrieves mappings for the currently selected features
            delete_files (bool): If True, id files will be deleted before return

        Returns:
            dict: The file dict.

            key = entity_type ('POINT', 'ARC', 'POLYGON'),

            value = (att_id_file, comp_id_file)

            Note that if delete_files=True, these files will not exist upon return. If you need to make a copy of
            the component id files, set delete_files=False and manually clean up the files after copying.
        """
        if only_selected:
            comp_dict = self._requested_data['selected_component_id_files'].get(component.uuid, {})
        else:
            comp_dict = self._requested_data['component_id_files'].get(component.uuid, {})
        files_dict = comp_dict.get(component.cov_uuid, {})
        component.load_coverage_component_id_map(files_dict)
        # Don't worry about deleting the files. The entire playback folder will get deleted.
        return files_dict

    def coverage_attributes(self, cov_uuid, points=False, arcs=False, polygons=False, arc_groups=False):
        """Retrieve coverage attribute table from XMS.

        Args:
            cov_uuid (str): UUID of the coverage
            points (bool): True to retrieve feature point attributes
            arcs (bool): True to retrieve feature arc attributes
            polygons (bool): True to retrieve feature polygon attributes
            arc_groups (bool): True to retrieve feature arc group attributes

        Returns:
            dict: The requested attribute tables. Keyed by 'points', 'arcs', 'polygons', and 'arc_groups'.
        """
        return self._requested_data['coverage_attributes'].get(cov_uuid, {})

    # Methods for adding data to be built in XMS.
    def add_simulation(self, do_sim, components=None, component_keywords=None):
        """Add a simulation to XMS.

        Args:
            do_sim (pydop.Simulation): The simulation to add
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
        """
        self._sent_data['simulations'] += 1
        if components:
            self._sent_data['simulation_components'] += len(components)

    def add_coverage(
        self,
        do_coverage,
        model_name=None,
        coverage_type=None,
        components=None,
        component_keywords=None,
        folder_path=None
    ):
        """Add a coverage to XMS.

        Args:
            do_coverage (Union[pydop.Coverage, xms.api._xmsapi.dmi.DataDumpIOBase]): The coverage to add. If type is an
                xmscoverage generic coverage dump, all other arguments are ignored.
            model_name (str): XML-defined name of the coverage's model. If specified, must provide coverage_type.
            coverage_type (str): XML-defined coverage type. If specified, must provide model_name.
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        if isinstance(do_coverage, pydop.Coverage):
            self._sent_data['coverages'] += 1
            if components:
                self._sent_data['coverage_components'] += len(components)
        else:
            self._sent_data['generic_coverages'] += 1

    def add_ugrid(self, do_ugrid):
        """Add a UGrid to be built in XMS.

        Args:
            do_ugrid (pydop.UGrid): The UGrid to add
        """
        self._sent_data['ugrids'] += 1

    def add_dataset(self, do_dataset, folder_path=None):
        """Add a Dataset to be built in XMS.

        Args:
            do_dataset (Union[pydop.Dataset, DatasetReader, xms.datasets.dataset_writer.DatasetWriter]): The dataset to
                add
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        self._sent_data['datasets'] += 1

    def add_raster(self, filename, item_uuid):
        """Add a GIS raster file to be loaded in XMS.

        Args:
            filename (str): Full path to the raster file
            item_uuid (Optional[str]): UUID to assign the raster in XMS
        """
        self._sent_data['rasters'] += 1

    def add_shapefile(self, filename, item_uuid):
        """Add a GIS shape file to be loaded in XMS.

        Args:
            filename (str): Full path to the shape file
            item_uuid (Optional[str]): UUID to assign the shapefile in XMS
        """
        # Lazily add this to the output since we added the feature for sending shapefiles after we implemented Query
        # recording and I don't want to have to update every existing recording test.
        if 'shapefiles' not in self._sent_data:
            self._sent_data['shapefiles'] = 0
        self._sent_data['shapefiles'] += 1

    def add_component(
        self,
        do_component,
        actions=None,
        messages=None,
        display_options=None,
        coverage_ids=None,
        shapefile_atts=None,
        folder_path=None
    ):
        """Add a non-hidden Component to be built in XMS.

        Args:
            do_component (pydop.Component): The component to add
            actions (list of pydop.ActionRequest): ActionRequests to send back with the component. See
                xms.components.bases.component_base.ComponentBase for more details.
            messages (list of tuple of str): Messages to send back with component. Should be formatted as follows:
                [('MSG_LEVEL', 'message text')...]  See xms.components.bases.component_base.ComponentBase for more
                details.
            display_options (list): Display options to send back with the component. Data should be in format returned
                by xms.components.bases.component_base.ComponentBase.get_display_options().
            coverage_ids (list): Coverage component id assignments to send back with the component. Data should be in
                format returned by
                xms.components.bases.coverage_component_base.CoverageComponentBase.get_component_coverage_ids().
            shapefile_atts (dict): Shapefile attributes to send back with the component. Data should be in
                format returned by xms.components.bases.component_base.ComponentBase.get_shapefile_att_files().
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        self._sent_data['components'] += 1

    def add_model_check_errors(self, errors):
        """Adds model check errors to be sent back to XMS.

        Errors will all be set at the Context level we had at construction. It is assumed that nothing else will be
        added during the script.

        Args:
            errors (Union[pydop.ModelCheckError, Sequence[pydop.ModelCheckError]): The model check errors to add. Can
                pass an iterable or a single error
        """
        if not isinstance(errors, Iterable):  # Allow argument to be a single error
            self._sent_data['model_check_errors'] += 1
        else:
            self._sent_data['model_check_errors'] += len(errors)

    def link_item(self, taker_uuid, taken_uuid):
        """Link one item under another.

        Args:
            taker_uuid (str): UUID of the item to link the other item under. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to link under the other item
        """
        self._sent_data['link_items'] += 1

    def unlink_item(self, taker_uuid, taken_uuid):
        """Unlink one item from under another.

        Args:
            taker_uuid (str): UUID of the item to unlink the other item from. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to unlink from the other item
        """
        self._sent_data['unlink_items'] += 1

    def delete_item(self, item_uuid):
        """Delete an item in XMS.

        Args:
            item_uuid (str): UUID of the item to delete
        """
        self._sent_data['delete_items'] += 1

    def rename_item(self, item_uuid, new_name):
        """Rename a tree item in XMS.

        Args:
            item_uuid (str): UUID of the item to rename
            new_name (str): New name to assign the item. Not guaranteed to be the name XMS assigns the item. Will be
                made unique if another item with the same name already exists.
        """
        self._sent_data['rename_items'] += 1

    def edit_ugrid_top_bottom_elevations(self, json_file):
        """Edit the top and bottom elevations of a UGrid in XMS.

        Args:
            json_file (str): Path to the JSON file containing the new elevations. See xms.components.dataset_metadata
                for more information of the expected JSON structure.
        """
        self._sent_data['ugrid_elevation_edits'] += 1

    def get_ugrid_selection(self, ugrid_uuid):
        """Get the currently selected UGrid cells and points.

        Returns:
            dict: Possible keys are 'POINT' and 'CELL'. Values are lists of UGrid point/cell ids. If key does not
            exist, no UGrid entities of that type are selected.
        """
        return self._requested_data['ugrid_selections'].get(ugrid_uuid, {})

    # Send any built data to XMS.
    def send(self, component_event=False):
        """Send data queued from add() and set() calls to XMS.

        Args:
            component_event (bool): True if send is being sent from a component event script. They do all sorts of their
                own special messing with the build context, so don't try to manage it.

        Returns:
            xms.api.dmi.SetDataResult: Result object reply from the send request
        """
        self._write_output_files()

    def clear(self):
        """Call this to clear any data added to the Context to prevent it from being sent to XMS."""
        self._reset_sent_data()
        super().clear()

    @property
    def playback_folder(self):
        """Returns the location of the playback folder if a QueryPlaybackImpl, else empty string."""
        return self._playback_folder
