# 1. Standard python modules
import fnmatch
import os
from setuptools import Extension
from setuptools.command.build_py import build_py

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


class PydBuilder(build_py):
    """Implementation of build_py to allow mix of pure Python and Cython pyd extensions."""

    cython_excludes = []

    def find_package_modules(self, package, package_dir):
        """Include any pure Python modules that have been specified in cython_excludes.

        Args:
            package (str): Name of the package
            package_dir (str): Location of the pockage source

        Returns:
            list: List of the pure Python modules that should be included in the package
        """
        modules = super().find_package_modules(package, package_dir)
        return filter(self._not_cythonized, modules)

    def _not_cythonized(self, tup):
        """Determine if a pure Python module should be included in the package.

        Args:
            tup (tuple): package name, module name, filepath

        Returns:
            bool: True if the module should be included.
        """
        (package, module, filepath) = tup
        return any(fnmatch.fnmatch(filepath, pat=pattern) for pattern in self.cython_excludes)


def generate_cython_extensions(cythonize_path, cython_excludes):
    """Generate cython extensions by walking the path and excluding items

        Args:
            cythonize_path (str): full path to directory to walk to find py files
            cython_excludes (list(str)): list of files to exclude

        Returns:
            list (Extensions): cython extensions
    """
    extensions = []
    ext_dict = _generate_cython_extensions_dict(cythonize_path, cython_excludes)
    for key, val in ext_dict.items():
        extensions.append(Extension(key, [val]))
    return extensions


def _generate_cython_extensions_dict(cythonize_path, cython_excludes):
    """Generate cython extensions by walking the path and excluding items

        Args:
            cythonize_path (str): full path to directory to walk to find py files
            cython_excludes (list(str)): list of files to exclude

        Returns:
            dict (str: str): dict of strings
    """
    # print(f'cythonize_path: {cythonize_path}')
    ret_dict = {}
    excludes = [os.path.basename(x) for x in cython_excludes]
    path_to_walk = cythonize_path
    # print(f'path_to_walk: {path_to_walk}')
    root = os.path.dirname(path_to_walk) + os.sep
    # print(f'root: {root}')
    for path, subdirs, files in os.walk(path_to_walk):
        for name in files:
            if name.endswith('.py'):
                if name in excludes:
                    continue
                full_path = os.path.join(path, name)
                name1 = full_path.replace(root, '')
                key = os.path.splitext(name1.replace(os.sep, '.'))[0]
                ret_dict[key] = name1
    # print(f'{ret_dict}')
    return ret_dict
