# 1. Standard python modules
import argparse
import os
import sys
from importlib import metadata

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


def find_xms_model_definitions():
    """Package finder for XMS DMI components and model interfaces

    This script is called by XMS on startup. It will be called with the Python installed with XMS. To register a package
    for loading on XMS startup, install to the XMS Python as follows:
        1. Define an 'xms.dmi.interfaces' entry point. The text of the key is irrelevant, but should be unique among all
        registered packages. The value should be the package name.

        2. Define an 'XMS DMI Definition' classifier containing the import path to the component/model interface
        definition file. Only one definition per package is supported. Must specify the format of the definition file
        ('DIN' or 'XML'). Can optionally define 'XMS DMI Migration' classifiers using the same format to load migration
        definitions.
    Example setuptools.setup call in setup.py:
    ::
        setup(
            name='xmsexample',
            version='0.0.0',
            packages=find_packages(),
            include_package_data=True,

            # Register an entry point so XMS can find the package on startup.
            entry_points={
                'xms.dmi.interfaces': 'UNIQUE_KEY = xmsexample'
            },

            # Define a classifier pointing to the definition file. Must be relative from package import root.
            classifiers=[
                'XMS DMI Definition :: XML :: xmsexample/example_module/xmsexample_definition.xml',
                'XMS DMI Migration :: DIN :: xmsexample/example_module/xmsexample_migration.din',
            ]
        )

    """

    arguments = argparse.ArgumentParser(description="Package finder for XMS DMI model interfaces and components.")
    arguments.add_argument(dest='command', type=str, help='script to run')
    arguments.add_argument(
        dest='xml_file', type=str, help='Path to file where *.xml file locations and package versions will be written'
    )
    arguments.add_argument(
        dest='din_file', type=str, help='Path to file where *.din file locations and package versions will be written'
    )
    arguments.add_argument(
        dest='migration_xml_file',
        type=str,
        help='Path to file where migration *.xml file locations and package versions will be written'
    )
    arguments.add_argument(
        dest='migration_din_file',
        type=str,
        help='Path to file where migration *.din file locations and package versions will be written'
    )
    parsed_args = arguments.parse_args()
    try:
        py_model_defs = _XmsModelDefinitions(parsed_args)
        py_model_defs.find_packages()
    except Exception:
        sys.exit(-1)
    sys.exit(0)


class _XmsModelDefinitions:
    """Finds the xms dmi model definitions from installed package metadata."""

    def __init__(self, parsed_args):
        """Initialize the class."""
        self.xml_out_file = open(parsed_args.xml_file, 'w')
        self.din_out_file = open(parsed_args.din_file, 'w')
        self.migration_xml_out_file = open(parsed_args.migration_xml_file, 'w')
        self.migration_din_out_file = open(parsed_args.migration_din_file, 'w')
        self._package_data = []

    def find_packages(self):
        """Finds the xms dmi packages."""
        self._get_xms_dmi_distributions()
        self._write_files()

    def _write_files(self):
        """Write the files."""
        for model_def in self._package_data:
            if model_def['dmi_def']:
                if model_def['is_din']:
                    self.din_out_file.write(model_def["file_line"])
                else:
                    self.xml_out_file.write(model_def["file_line"])
            else:
                if model_def['is_din']:
                    self.migration_din_out_file.write(model_def["file_line"])
                else:
                    self.migration_xml_out_file.write(model_def["file_line"])

    def _get_xms_dmi_distributions(self):
        """Build a list of xms dmi distributions."""
        # make static list because metadata.distributions() is not stable if packages are loaded while looping
        dist_list = list(metadata.distributions())
        for dist in dist_list:
            entry_pt = [x for x in dist.entry_points if x.group == 'xms.dmi.interfaces']
            if len(entry_pt) == 0:
                continue
            if 'classifier' not in dist.metadata.json:
                continue
            self._store_paths_from_dist(dist, entry_pt[0])

    def _store_paths_from_dist(self, dist, entry_pt):
        """Read the path to the xml (or din) from the distribution and store it.

        Args:
            dist (:obj:`Distribution`): The distribution to get the model definitions from.
            entry_pt (:obj:`EntryPoint`): The entry point for the distribution.
        """
        for line in dist.metadata.json['classifier']:
            model_def = {
                'dmi_def': line.startswith('XMS DMI Definition'),
                'dmi_migration': line.startswith('XMS DMI Migration'),
            }
            if not model_def['dmi_def'] and not model_def['dmi_migration']:
                continue
            cards = line.split(':')
            if len(cards) < 4:
                continue
            model_def['is_din'] = cards[2].strip().upper() == 'DIN'
            file_path = self._check_package_file_location(cards[4].strip(), dist, entry_pt)
            if not file_path:
                continue
            model_def['file_path'] = file_path
            model_def['file_line'] = f'"{file_path}" "{dist.version}" "{dist.name}"\n'
            self._package_data.append(model_def)

    def _check_package_file_location(self, xml_din_file, dist, entry_pt):
        """Locate the package file."""
        # prefer call dist.locate_file(xml_file) but it doesn't work with pyproject develop installs
        file_path = str(dist.locate_file(xml_din_file))
        if not os.path.isfile(file_path):
            dist_path = entry_pt.load().__path__[0]
            dist_path = os.path.normpath(os.path.join(dist_path, '..', '..'))
            file_path = os.path.normpath(os.path.join(dist_path, xml_din_file))
        if not os.path.isfile(file_path):
            return ''
        return file_path
