"""Python wrapper for xmsapi.dmi.ProjectExplorerItem class."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Projection as DoProj
import xms.data_objects.parameters.spatial.spatial_vector.Dataset as DoDset

# 4. Local modules
import xms.api._xmsapi.dmi as xmd

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

DATA_OBJECTS_TYPE_STRINGS = {
    xmd.SimulationItem: 'simulation',
    xmd.CoverageItem: 'coverage',
    xmd.ComponentItem: 'component',
    xmd.UGridItem: 'ugrid',
    xmd.DatasetItem: 'dataset',
    xmd.ProjectExplorerItem: 'other',
}


class TreeNode:
    """Class used to represent an item in the Project Explorer."""
    def __init__(self, data_object=None, other=None):
        """Initializes the class.

        Args:
            data_object (xmsapi.dmi.ProjectExplorerItem): The data_object representing a project explorer
                tree item
            other (TreeNode): TreeNode to copy construct from if creating from another TreeNode instance. Note that
                this this is a shallow copy. Use ProjectExplorerTreeCreator.copy() to make deep copies.
        """
        self._data_object = data_object
        self._parent = None
        self._children = []
        if other is not None:
            self._parent = other.parent
            self._children = other.children
            self._data_object = other.data

    @property
    def data(self):
        """Get the underlying data_object this class wraps."""
        return self._data_object

    # Universal tree item attributes
    @property
    def name(self):
        """Tree item name in XMS."""
        return self._data_object.GetName()

    @name.setter
    def name(self, item_name):
        """Set tree item name.

        Args:
            item_name (str): Name of the TreeNode
        """
        self._data_object.SetName(item_name)

    @property
    def uuid(self):
        """UUID of the tree node data in XMS, if it has one."""
        return self._data_object.GetUuid()

    @uuid.setter
    def uuid(self, item_uuid):
        """Set the UUID of the TreeNode.
        Args:
            item_uuid (str): UUID of the TreeNode
        """
        self._data_object.SetUuid(item_uuid)

    @property
    def item_typename(self):
        """Get the typename text of the item's XMS tree item object.

        Note:
            This attribute is a string defined for project explorer tree item types in XMS. Use with caution. String
            values may change with versions. Only use if type is not directly supported by the
            xmsapi.dmi.ProjectExplorerItemType enum.

        """
        try:
            # XMS ETreeItem typename was added with data_objects>=1.2.0. Only useful for determining the type
            # of items when the item type is not explicitly supported by the API.
            return self._data_object.GetItemTypename()
        except AttributeError:
            return ''

    @item_typename.setter
    def item_typename(self, typename):
        """Set the typename text of the item's XMS tree item object.

        Args:
            typename (str): The XMS defined tree item enum string
        """
        self._data_object.SetItemTypename(typename)

    @property
    def is_ptr_item(self):
        """True if node is linked pointer to another tree item."""
        return self._data_object.GetIsPointerItem()

    @is_ptr_item.setter
    def is_ptr_item(self, is_ptr):
        """Set the pointer item flag.

        Args:
            is_ptr (bool): True if node is linked pointer to another tree item.
        """
        self._data_object.SetIsPointerItem(is_ptr)

    @property
    def is_active_item(self):
        """True if node is the active item in its activity group."""
        return self._data_object.GetIsActiveItem()

    @is_active_item.setter
    def is_active_item(self, is_active):
        """Set the active item flag.

        Args:
            is_active (bool): True if node is the active item in its activity group.
        """
        self._data_object.SetIsActiveItem(is_active)

    @property
    def has_check_box(self):
        """True if the tree item has a checkbox."""
        return self._data_object.HasCheckBox()

    @property
    def check_state(self):
        """The tree item's check state.

        Returns:
            int: -1 = no check box, 0 = unchecked, 1 = checked, 2 = partial checked
        """
        return self._data_object.GetCheckState()

    @check_state.setter
    def check_state(self, item_state):
        """Set the tree item's check state.

        Args:
            item_state (int): -1 = no check box, 0 = unchecked, 1 = checked, 2 = partial checked
        """
        self._data_object.SetCheckState(item_state)

    @property
    def children(self):
        """Children of the tree node in a list. If terminal tree node, empty list."""
        return self._children

    @children.setter
    def children(self, new_children):
        """Reset the children of the tree item. Useful when filtering a project explorer tree.

        Note that this only sets the Python pointers to child TreeNodes, not the underlying data_object.
        """
        self._children = new_children

    @property
    def parent(self):
        """Parent of the tree node. None if root."""
        return self._parent

    @parent.setter
    def parent(self, new_parent):
        """Set parent of a tree_node.

        Note that this only sets the Python pointer to the parent TreeNode, not the underlying data_object.
        """
        self._parent = new_parent

    # Model name is valid attribute for simulation, coverage, and component tree items
    @property
    def model_name(self):
        """Model name of associated with tree item, if it exists."""
        if type(self._data_object) not in [xmd.DatasetItem, xmd.UGridItem, xmd.ProjectExplorerItem]:
            return self._data_object.GetModelName()
        return ''

    @model_name.setter
    def model_name(self, model):
        """Set the XML-defined model name of associated with tree item.

        Args:
            model (str): The XML-defined model name of the tree item
        """
        if type(self._data_object) not in [xmd.DatasetItem, xmd.UGridItem, xmd.ProjectExplorerItem]:
            self._data_object.SetModelName(model)

    # Type name is only valid coverage tree items
    @property
    def coverage_type(self):
        """Coverage type of associated with tree item, if it exists."""
        if type(self._data_object) == xmd.CoverageItem:
            return self._data_object.GetCoverageType()
        return ''

    @coverage_type.setter
    def coverage_type(self, cov_type):
        """Set coverage type of associated with tree item, if item is a coverage.

        Args:
            cov_type (str): The coverage's type
        """
        if type(self._data_object) == xmd.CoverageItem:
            self._data_object.SetCoverageType(cov_type)

    # Component-specific tree item attributes
    @property
    def unique_name(self):
        """XML defined unique name of a component tree item."""
        if type(self._data_object) in [xmd.ComponentItem, xmd.CoverageItem, xmd.SimulationItem]:
            return self._data_object.GetUniqueName()
        return ''

    @unique_name.setter
    def unique_name(self, xml_unique_name):
        """Set the XML defined unique name of a component tree item.

        Args:
            xml_unique_name (str): The XML defined unique name of the component tree item
        """
        if type(self._data_object) in [xmd.ComponentItem, xmd.CoverageItem, xmd.SimulationItem]:
            self._data_object.SetUniqueName(xml_unique_name)

    @property
    def main_file(self):
        """Main file of a component tree item."""
        if type(self._data_object) in [xmd.ComponentItem, xmd.CoverageItem, xmd.SimulationItem]:
            return self._data_object.GetMainFile()
        return ''

    @main_file.setter
    def main_file(self, filename):
        """Set the main file of a component tree item.

        Args:
            filename (str): Path to the component's main file
        """
        if type(self._data_object) in [xmd.ComponentItem, xmd.CoverageItem, xmd.SimulationItem]:
            self._data_object.SetMainFile(filename)

    @property
    def component_uuid(self):
        """UUID of the coverage or simulation's hidden component if it exists."""
        if type(self._data_object) in [xmd.CoverageItem, xmd.SimulationItem]:
            return self._data_object.GetComponentUuid()
        return ''

    @component_uuid.setter
    def component_uuid(self, comp_uuid):
        """Set the UUID of the coverage or simulation's hidden component.

        Args:
            comp_uuid (str): The hidden component's UUID
        """
        if type(self._data_object) in [xmd.CoverageItem, xmd.SimulationItem]:
            self._data_object.SetComponentUuid(comp_uuid)

    @property
    def native_projection(self):
        """Native projection of a Coverage or UGrid tree item."""
        if type(self._data_object) in [xmd.CoverageItem, xmd.UGridItem]:
            return DoProj(instance=self._data_object.GetNativeProjection())
        return None

    @native_projection.setter
    def native_projection(self, projection):
        """Set native projection of a Coverage or UGrid tree item.

        Args:
            projection (data_objects.parameters.Projection): The object's native projection
        """
        if type(self._data_object) in [xmd.CoverageItem, xmd.UGridItem]:
            self._data_object.SetNativeProjection(projection._instance)

    # UGrid-specific tree item attributes
    @property
    def num_cells(self):
        """Number of cells in a UGrid tree item."""
        if type(self._data_object) == xmd.UGridItem:
            return self._data_object.GetNumberOfCells()
        return 0

    @num_cells.setter
    def num_cells(self, cell_count):
        """Set number of cells in a UGrid tree item.

        Args:
            cell_count (int): The number of cells in the geometry
        """
        if type(self._data_object) == xmd.UGridItem:
            self._data_object.SetNumberOfCells(cell_count)

    @property
    def num_points(self):
        """Number of points in a UGrid tree item."""
        if type(self._data_object) == xmd.UGridItem:
            return self._data_object.GetNumberOfNodes()
        return 0

    @num_points.setter
    def num_points(self, point_count):
        """Set number of points in a UGrid tree item.

        Args:
            point_count (int): The number of points in the geometry
        """
        if type(self._data_object) == xmd.UGridItem:
            self._data_object.SetNumberOfNodes(point_count)

    # Dataset-specific tree item attributes
    @property
    def num_times(self):
        """Number of time steps in a Dataset tree item."""
        if type(self._data_object) == xmd.DatasetItem:
            return self._data_object.GetNumberOfTimesteps()
        return 0

    @num_times.setter
    def num_times(self, time_count):
        """Set number of time steps in a Dataset tree item.

        Args:
            time_count (int): Number of timesteps in the dataset
        """
        if type(self._data_object) == xmd.DatasetItem:
            self._data_object.SetNumberOfTimesteps(time_count)

    @property
    def active_timestep(self):
        """Active timestep of a Dataset tree item."""
        if type(self._data_object) == xmd.DatasetItem:
            return self._data_object.GetActiveTimestep()
        return 0

    @active_timestep.setter
    def active_timestep(self, active_timestep):
        """Set active timestep of a Dataset tree item.

        Args:
            active_timestep (int): Active dataset timestep.
        """
        if type(self._data_object) == xmd.DatasetItem:
            self._data_object.SetActiveTimestep(active_timestep)

    @property
    def num_components(self):
        """Number of components (scalar=1, vector=2) in a Dataset tree item."""
        if type(self._data_object) == xmd.DatasetItem:
            return self._data_object.GetNumberOfComponents()
        return 0

    @num_components.setter
    def num_components(self, component_count):
        """Set number of components (scalar=1, vector=2) in a Dataset tree item.

        Args:
            component_count (int): Number of components in the dataset
        """
        if type(self._data_object) == xmd.DatasetItem:
            self._data_object.SetNumberOfComponents(component_count)

    @property
    def num_vals(self):
        """Number of values in a Dataset tree item."""
        if type(self._data_object) == xmd.DatasetItem:
            return self._data_object.GetNumberOfDataLocations()
        return 0

    @num_vals.setter
    def num_vals(self, value_count):
        """Set number of values in a Dataset tree item.

        Args:
            value_count (int): Number of values in the dataset
        """
        if type(self._data_object) == xmd.DatasetItem:
            self._data_object.SetNumberOfDataLocations(value_count)

    @property
    def data_location(self):
        """Get the location basis of a Dataset tree item (e.g. nodes vs cells)."""
        if type(self._data_object) == xmd.DatasetItem:
            return DoDset.dataset_enum_to_text(DoDset.DatasetDataBasisStrings, self._data_object.GetDataMappingType())
        return ''

    @data_location.setter
    def data_location(self, data_loc):
        """Set location basis (e.g. nodes vs cells) of a Dataset tree item.

        Args:
            data_loc (str): The location basis of the dataset
        """
        if type(self._data_object) == xmd.DatasetItem:
            enum_location = DoDset.dataset_text_to_enum(DoDset.DatasetDataBasisStrings, data_loc)
            self._data_object.SetDataMappingType(enum_location)

    def add_child(self, node):
        """Adds the node as a child.

        Note that this only sets the Python pointers to child TreeNodes, not the underlying data_object.

        Args:
            node (TreeNode): A node
        """
        if not node:
            return
        self._children.append(node)
        node.parent = self

    def add_children(self, nodes):
        """Adds the nodes as children.

        Args:
            nodes (list of TreeNode): nodes
        """
        self._children.extend(list(nodes))
        for node in nodes:
            node.parent = self

    def remove_child(self, node):
        """Removes the node from children.

        Args:
            node (TreeNode): A child node
        """
        if not node:
            return
        self._children.remove(node)
        node.parent = None

    def _type_string(self):
        """Return a stringified version of the data_object type."""
        if type(self._data_object) == xmd.DatasetItem:
            return ''

    def __iter__(self):
        """Used for conversion to dict."""
        yield 'ctype', DATA_OBJECTS_TYPE_STRINGS[type(self._data_object)]
        yield 'name', self.name
        yield 'uuid', self.uuid
        yield 'item_typename', self.item_typename
        yield 'is_ptr_item', self.is_ptr_item
        yield 'is_active_item', self.is_active_item
        yield 'check_state', self.check_state
        yield 'model_name', self.model_name
        yield 'coverage_type', self.coverage_type
        yield 'unique_name', self.unique_name
        yield 'main_file', self.main_file
        yield 'component_uuid', self.component_uuid
        yield 'native_projection', projection_to_json(self.native_projection)
        yield 'num_cells', self.num_cells
        yield 'num_points', self.num_points
        yield 'num_times', self.num_times
        yield 'num_components', self.num_components
        yield 'num_vals', self.num_vals
        yield 'data_location', self.data_location
        yield 'children', [dict(child) for child in self._children]

    def from_dict(self, json_dict):
        """Construct this TreeNode from a JSON dict.

        Args:
            json_dict (dict): The serialized member data as returned by self.__iter__()
        """
        # Construct the proper data_object first
        ctype = json_dict.get('ctype', 'other')
        if ctype == 'simulation':
            self._data_object = xmd.SimulationItem()
        elif ctype == 'coverage':
            self._data_object = xmd.CoverageItem()
        elif ctype == 'component':
            self._data_object = xmd.ComponentItem()
        elif ctype == 'ugrid':
            self._data_object = xmd.UGridItem()
        elif ctype == 'dataset':
            self._data_object = xmd.DatasetItem()
        else:  # 'other'
            self._data_object = xmd.ProjectExplorerItem()

        for attr_name, attr_value in json_dict.items():
            if attr_name == 'children':
                for child_attrs in attr_value:
                    # Construct the proper data_object
                    node = TreeNode()
                    node.from_dict(child_attrs)
                    node._data_object.SetParent(self._data_object)
                    self._data_object.AddChild(node._data_object)
                    self.add_child(node)
            elif attr_name == 'native_projection':
                if ctype in ['coverage', 'ugrid']:
                    self.native_projection = DoProj(**attr_value)
            elif attr_name != 'ctype':
                setattr(self, attr_name, attr_value)


def projection_to_json(projection):
    """Returns a JSON dict for a data_objects Projection.

    Notes:
        This really should be in the Python wrapper for data_objects.parameters.Projection, but this avoids updating
        data_objects.

    Args:
        projection (pydop.Projection): The projection to serialize

    Returns:
        dict: See description
    """
    if not projection:
        return {}

    return {
        'wkt': projection.well_known_text,
        'system': projection.coordinate_system,
        'zone': projection.coordinate_zone,
        'horizontal_units': projection.horizontal_units,
        'vertical_units': projection.vertical_units,
        'vertical_datum': projection.vertical_datum,
        'projection_name': projection.projection_name,
    }
