"""The Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import sys
import webbrowser

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QProcess, QSize, Qt
from PySide2.QtWidgets import (
    QComboBox, QDialog, QDialogButtonBox, QHBoxLayout, QLabel, QLineEdit, QMessageBox, QScrollArea, QTabWidget,
    QVBoxLayout, QWidget
)
from shapely.geometry import LineString

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xmf
from xms.data_objects.parameters import UGrid as DoUGrid
from xms.gdal.rasters import RasterInput
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.tool_gui.tool_dialog import run_tool_with_feedback

# 4. Local modules
from xms.bridge import structure_util as su
from xms.bridge.calc.footprint_calc import FootprintCalculator
import xms.bridge.gui.struct_dlg.arc_props as ap
from xms.bridge.gui.struct_dlg.bridge_tab import StructureBridgeTab
import xms.bridge.gui.struct_dlg.culvert_arc_props as culvert_ap
from xms.bridge.gui.struct_dlg.culvert_tab import StructureCulvertTab
from xms.bridge.gui.struct_dlg.map_viewer_dialog import map_viewer_file_fullpath
from xms.bridge.gui.struct_dlg.overtop_tab import StructureOvertopTab
from xms.bridge.gui.struct_dlg.preview_html import PreviewHtml
from xms.bridge.gui.struct_dlg.profile_tab import StructureProfileTab


class StructureDialog(XmsDlg):
    """A dialog for editing structure properties."""
    def __init__(self, parent, dlg_data, arc_data, help_url=None):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            dlg_data (:obj:`StructureData`): data for dialog
            arc_data (:obj:`dict`): arc data from the coverage
            help_url (:obj:`str`): URL for the help page
        """
        super().__init__(parent, 'xms.bridge.structure_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:3D_Structure' if not help_url else help_url
        self.data = dlg_data
        self.arc_data = arc_data
        self.output_ugrid = None
        self.output_footprint_ugrid = None
        self.output_coverage = None
        self.title = 'Structure Properties'

        self.description = 'Edit properties for the 3D structure.'
        self._help_pane = None
        self._profile_tab = None
        self._bridge_tab = None
        self._culvert_tab = None
        self.arc_props = None
        self.culvert_arc_props = None
        self._overtop_tab = None
        self._widgets = {}
        self._map_viewer_proc = None
        self.elev_uuid = self.data.data_dict['elev_raster']
        self.elev_file = ''
        self.footprint_calc = FootprintCalculator(
            struct_comp=self.arc_data['component'],
            coverage=self.arc_data['coverage'],
            data_folder=self.arc_data['tmp_dir'],
            cl_arc_pts=self.arc_data['arc_pts'],
            wkt=self.arc_data['wkt'],
            vertical_units=self.arc_data.get('vertical_units', '')
        )
        self._setup_ui()
        self._update_raster_file_name(show_msg=False)
        self.setMinimumSize(QSize(400, 550))

    def _setup_ui(self):
        """Sets up the widgets for the UI."""
        self.setWindowTitle(self.title)
        self._widgets['top_layout'] = QVBoxLayout()
        self.setLayout(self._widgets['top_layout'])
        self._widgets['top_hlayout'] = QHBoxLayout()
        self._widgets['top_layout'].addLayout(self._widgets['top_hlayout'])
        self._widgets['controls_widget'] = QWidget()
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['controls_widget'].setLayout(self._widgets['main_vert_layout'])

        self._add_widgets_to_main_layout()
        self._widgets['top_hlayout'].addWidget(self._widgets['controls_widget'])
        self._add_button_box()

    def _add_widgets_to_main_layout(self):
        """Adds widgets to the main layout."""
        layout = self._widgets['main_vert_layout']

        # structure type
        self._widgets['txt_structure_type'] = QLabel("Structure type:")
        layout.addWidget(self._widgets['txt_structure_type'])
        struct_types = ['Bridge', 'Culvert']
        self._widgets['cbx_struct_type'] = QComboBox()
        self._widgets['cbx_struct_type'].addItems(struct_types)
        self._widgets['cbx_struct_type'].setCurrentIndex(0)
        if self.data.data_dict['structure_type'] in struct_types:
            self._widgets['cbx_struct_type'].setCurrentText(self.data.data_dict['structure_type'])
        layout.addWidget(self._widgets['cbx_struct_type'])
        self._widgets['cbx_struct_type'].currentIndexChanged.connect(self._on_change_structure_type)

        self._widgets['tab0'] = QWidget()
        self._widgets['tab1'] = QWidget()
        self._widgets['tab2'] = QWidget()
        self._widgets['tab3'] = QWidget()
        self._widgets['tabs'] = QTabWidget()
        self._widgets['tabs'].currentChanged.connect(self._on_tab_changed)

        self.arc_props = ap.StructureArcProps(self, self._widgets)
        self.culvert_arc_props = culvert_ap.CulvertArcProps(self, self._widgets)
        layout.addWidget(self._widgets['tabs'])
        self._make_tab_layout(0)
        self._profile_tab = StructureProfileTab(self, self._widgets)
        self._make_tab_layout(1)
        self._bridge_tab = StructureBridgeTab(self, self._widgets)
        self._make_tab_layout(2)
        self._overtop_tab = StructureOvertopTab(self, self._widgets)
        self._make_tab_layout(3)
        self._culvert_tab = StructureCulvertTab(self, self._widgets)
        self._on_change_structure_type()

    def _on_tab_changed(self, idx):
        """Called when the tab is changed."""
        if self._help_pane is not None:
            self._help_pane.restore_scroll_position = True

    def _make_tab_layout(self, idx):
        """Create layout for a tab control.

        Args:
            idx (:obj:`int`): index of the tab
        """
        self._widgets[f'tab{idx}_vert_layout'] = QVBoxLayout()
        self._widgets[f'tab{idx}'].setLayout(self._widgets[f'tab{idx}_vert_layout'])
        self._widgets[f'tab{idx}_scroll_area'] = QScrollArea(self._widgets[f'tab{idx}'])
        self._widgets[f'tab{idx}_vert_layout'].addWidget(self._widgets[f'tab{idx}_scroll_area'])
        self._widgets[f'tab{idx}_scroll_area'].setWidgetResizable(True)
        self._widgets[f'tab{idx}_scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self._widgets[f'tab{idx}_scroll_area_widget'] = QWidget()
        self._widgets[f'tab{idx}_scroll_area'].setWidget(self._widgets[f'tab{idx}_scroll_area_widget'])
        self._widgets[f'tab{idx}_scroll_area_vert_layout'] = QVBoxLayout()
        self._widgets[f'tab{idx}_scroll_area_widget'].setLayout(self._widgets[f'tab{idx}_scroll_area_vert_layout'])

    def add_floating_point_widget(self, base_name, label, data, layout, set_positive_validator=True):
        """Create floating point widget.

        Args:
            base_name (:obj:`str`): widget name
            label (:obj:`str`): label for the widget
            data (:obj:`float`): the value for the widget
            layout (:obj:`QVBoxLayout`): QT layout
            set_positive_validator (:obj:`bool`): flag to set the positive validator
        """
        self._widgets[f'txt_{base_name}'] = QLabel(label)
        layout.addWidget(self._widgets[f'txt_{base_name}'])
        self._widgets[f'edt_{base_name}'] = QLineEdit(f'{data}')

        pos_validator = QxDoubleValidator(parent=self)
        if set_positive_validator:
            pos_validator.setBottom(0.0)
        self._widgets[f'edt_{base_name}'].setValidator(pos_validator)

        layout.addWidget(self._widgets[f'edt_{base_name}'])

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['top_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def _on_change_structure_type(self):
        """Change to the structure type combobox."""
        self._widgets['tabs'].clear()
        self._widgets['tabs'].addTab(self._widgets['tab0'], 'Profiles')
        type_index = self._widgets['cbx_struct_type'].currentIndex()
        is_bridge = type_index in [0, 2]
        # hide widgets that are not used by culvert
        self._widgets['grp_up'].setVisible(is_bridge)
        self._widgets['tog_downstream'].setVisible(is_bridge)
        if is_bridge and self._widgets['tog_downstream'].isChecked():
            self._widgets['grp_down'].setVisible(True)
        else:
            self._widgets['grp_down'].setVisible(False)

        if is_bridge:
            self._widgets['tabs'].addTab(self._widgets['tab1'], 'Bridge Properties')
            self._widgets['tabs'].addTab(self._widgets['tab2'], 'Overtopping')
            self._bridge_tab.on_assign_arcs()
        else:
            self._widgets['tabs'].addTab(self._widgets['tab3'], 'Culvert Properties')
            self._widgets['tabs'].addTab(self._widgets['tab2'], 'Overtopping')

    def _run_bridge_footprint_tool(self):
        """Runs the bridge footprint tool."""
        self.footprint_calc.setup_tool(self.data)
        if self.footprint_calc.culvert_calc.err_msg:
            return

        run_tool_with_feedback(
            win_cont=self,
            tool=self.footprint_calc.tool,
            tool_arguments=self.footprint_calc.args,
            auto_str='Auto load mesh.',
            modal=True
        )
        self.footprint_calc.load_tool_results(self.elev_file, self.arc_data)

    def on_btn_preview_mesh(self):
        """Button click to preview the bridge pier mesh."""
        self._on_btn_preview()

    def on_btn_preview_mesh_browser(self):
        """Button click to preview the bridge pier mesh."""
        self._on_btn_preview(browser=True)

    def _on_btn_preview(self, browser=False):
        """Button click to preview the bridge pier mesh."""
        self.close_map_viewer_proc()
        self.get_dlg_data()
        self._run_bridge_footprint_tool()
        if self.footprint_calc.bridge_mesh:
            tool = self.footprint_calc.tool
            tool_data = tool.arc_data if hasattr(tool, 'arc_data') else None
            if self.footprint_calc.culvert_calc.arc_data:
                tool_data = self.footprint_calc.culvert_calc.arc_data
            if tool_data is not None:
                tool_data = {aa['id']: aa['arc_type'] for aa in tool_data}
            dlg_data = {
                'ugrid': self.footprint_calc.bridge_mesh,
                'projection': self.arc_data['projection'],
                'url': os.path.join(self.arc_data['tmp_dir'], 'map.html'),
                'arc_data': self.arc_data,
                'tool_arc_data': tool_data,
                'struct_type': self.data.data_dict['structure_type']
            }
            PreviewHtml(dlg_data)
            if browser:
                webbrowser.open(dlg_data['url'])
            else:
                self.run_map_viewer_proc(dlg_data['url'])
        elif self.footprint_calc.culvert_calc.err_msg:
            QMessageBox.warning(self, '3D Structure', self.footprint_calc.culvert_calc.err_msg, QMessageBox.Ok)

    def _viewer_proc(self):
        """Set the QProcess for the map viewer."""
        self._map_viewer_proc = QProcess(self)

    def run_map_viewer_proc(self, url):
        """Run the map viewer process.

        Args:
            url (:obj:`str`): html file
        """
        self._viewer_proc()
        self._map_viewer_proc.setProgram(sys.executable)
        py_file_path = map_viewer_file_fullpath()
        parent_id = str(self.parent().winId())
        self._map_viewer_proc.setArguments([py_file_path, '--parent_id', parent_id, '--url', url])
        self._map_viewer_proc.start()

    def close_map_viewer_proc(self):
        """Close the map viewer process if it is running."""
        if self._map_viewer_proc is not None:
            self._map_viewer_proc.kill()
            self._map_viewer_proc = None

    def _update_raster_file_name(self, show_msg):
        """Update the name of the raster file.

        Args:
            show_msg (:obj:`bool`): show the message if True
        """
        elev_file = ''
        if self.elev_uuid:
            elev_file = self.arc_data['xms_data'].raster_file_from_uuid(self.elev_uuid)
        self.elev_file = elev_file if elev_file is not None else ''
        self._widgets['txt_elev_raster'].setText('')
        self._widgets['txt_culvert_elev_raster'].setText('')
        if self.elev_uuid:
            try:
                raster = RasterInput(self.elev_file)
                if raster:
                    if not _is_float_raster(raster):
                        raise ValueError
                    tree_item = self.arc_data['xms_data'].tree_item_from_uuid(self.elev_uuid)
                    raster_name = tree_item.name if tree_item else os.path.basename(self.elev_file)
                    self._widgets['txt_elev_raster'].setText(raster_name)
                    self._widgets['txt_culvert_elev_raster'].setText(raster_name)
            except ValueError:
                self.elev_uuid = ''
                self.elev_file = ''
                msg = 'Invalid elevation raster selected. Select a different raster.'
                if show_msg:
                    QMessageBox.warning(self, '3D Structure', msg, QMessageBox.Ok)

    def on_btn_elev_raster(self):
        """Select the elevation raster."""
        dlg = TreeItemSelectorDlg(
            title='Select Elevation Raster',
            target_type=type(None),
            pe_tree=self.arc_data['xms_data'].project_tree,
            previous_selection=self.elev_uuid,
            show_root=False,
            parent=self,
            selectable_xms_types=['TI_IMAGE']
        )
        if dlg.exec() == QDialog.Accepted:
            item_uuid = dlg.get_selected_item_uuid()
            self.elev_uuid = item_uuid if item_uuid is not None else ''
            self._update_raster_file_name(show_msg=True)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def get_dlg_data(self):
        """Update the structure data from values in the dlg controls."""
        if self._widgets['tog_const_crest'].isChecked():
            elev = float(self._widgets['edt_const_crest'].text())
            self.data.curves['top_profile'] = pd.DataFrame({'Distance': [0.0, 1.0], 'Elevation': [elev, elev]})
        if self._widgets['tog_const_upstream'].isChecked():
            elev = float(self._widgets['edt_const_upstream'].text())
            self.data.curves['upstream_profile'] = pd.DataFrame({'Distance': [0.0, 1.0], 'Elevation': [elev, elev]})

        self.data.data_dict['structure_type'] = self._widgets['cbx_struct_type'].currentText()
        val = 1 if self._widgets['tog_assign_arcs'].isChecked() else 0
        self.data.data_dict['specify_arc_properties'] = val
        if val:
            self.data.data_dict['arc_properties'] = self.arc_props.get_arc_data_csv()
        self.data.data_dict['bridge_width'] = float(self._widgets['edt_width'].text())
        self.data.data_dict['bridge_ceiling_roughness'] = float(self._widgets['edt_manning'].text())
        self.data.data_dict['bridge_pier_base_elev'] = float(self._widgets['edt_pier_base_elev'].text())
        val = 1 if self._widgets['tog_downstream'].isChecked() else 0
        self.data.data_dict['bridge_specify_downstream_profile'] = val
        if val == 0:  # copy the upstream profile to the downstream profile
            self.data.curves['downstream_profile'] = self.data.curves['upstream_profile'].copy()
        # bridge/pier properties
        val = 1 if self._widgets['tog_bridge_cl_segments'].isChecked() else 0
        self.data.data_dict['specify_bridge_cl_num_segments'] = val
        self.data.data_dict['number_bridge_cl_segments'] = int(self._widgets['edt_bridge_cl_num_segments'].text())
        self.data.data_dict['bridge_wrapping_width'] = float(self._widgets['edt_wrap_width'].text())
        val = 1 if self._widgets['tog_wrap_upstream'].isChecked() else 0
        self.data.data_dict['bridge_wrap_upstream'] = val
        val = 1 if self._widgets['tog_wrap_downstream'].isChecked() else 0
        self.data.data_dict['bridge_wrap_downstream'] = val
        self.data.data_dict['bridge_num_side_elem'] = int(self._widgets['edt_bridge_num_side_elem'].text())
        val = 1 if self._widgets['tog_specify_segments'].isChecked() else 0
        self.data.data_dict['specify_number_segments'] = val
        self.data.data_dict['number_segments'] = int(self._widgets['edt_num_segments'].text())
        val = 1 if self._widgets['tog_abutments'].isChecked() else 0
        self.data.data_dict['has_abutments'] = val
        self.data.data_dict['pier_type'] = self._widgets['cbx_pier_type'].currentText()
        self.data.data_dict['pier_wrapping_width'] = float(self._widgets['edt_elem_wrap_width'].text())
        self.data.data_dict['pier_group_diameter'] = float(self._widgets['edt_pier_diameter'].text())
        self.data.data_dict['pier_group_num_in_group'] = int(self._widgets['edt_num_piers'].text())
        self.data.data_dict['pier_group_spacing'] = float(self._widgets['edt_pier_spacing'].text())
        self.data.data_dict['pier_wall_width'] = float(self._widgets['edt_wall_width'].text())
        self.data.data_dict['pier_wall_length'] = float(self._widgets['edt_wall_length'].text())
        self.data.data_dict['pier_wall_num_side_elem'] = int(self._widgets['edt_wall_num_side'].text())
        self.data.data_dict['pier_wall_end_type'] = self._widgets['cbx_pier_end_type'].currentText()
        # srh overtopping
        val = 1 if self._widgets['tog_srh_overtop'].isChecked() else 0
        self.data.data_dict['srh_overtopping'] = val
        self.data.data_dict['srh_weir_length'] = float(self._widgets['edt_srh_weir_length'].text())
        self.data.data_dict['srh_weir_type'] = self._widgets['cbx_weir_type'].currentText()
        self.data.data_dict['srh_weir_cw'] = float(self._widgets['edt_srh_weir_cw'].text())
        self.data.data_dict['srh_weir_a'] = float(self._widgets['edt_srh_weir_a'].text())
        self.data.data_dict['srh_weir_b'] = float(self._widgets['edt_srh_weir_b'].text())
        # culvert
        val = 1 if self._widgets['tog_culvert_assign_arcs'].isChecked() else 0
        self.data.data_dict['specify_culvert_arc_properties'] = val
        if val:
            self.data.data_dict['culvert_arc_properties'] = self.culvert_arc_props.get_arc_data_csv()
        self.data.data_dict['culvert_type'] = self._widgets['cbx_culvert_type'].currentText()
        self.data.data_dict['culvert_diameter'] = float(self._widgets['edt_culvert_diameter'].text())
        self.data.data_dict['culvert_rise'] = float(self._widgets['edt_culvert_rise'].text())
        self.data.data_dict['culvert_span'] = float(self._widgets['edt_culvert_span'].text())
        self.data.data_dict['culvert_up_invert'] = float(self._widgets['edt_culvert_up_invert'].text())
        self.data.data_dict['culvert_dn_invert'] = float(self._widgets['edt_culvert_dn_invert'].text())
        self.data.data_dict['culvert_embed_depth'] = float(self._widgets['edt_culvert_embed_depth'].text())
        self.data.data_dict['culvert_num_seg_barrel'] = int(self._widgets['edt_culvert_num_elem_barrel'].text())
        self.data.data_dict['culvert_num_barrels'] = int(self._widgets['edt_culvert_num_barrels'].text())
        self.data.data_dict['culvert_wall_width'] = float(self._widgets['edt_culvert_wall_width'].text())
        if self.data.data_dict['structure_type'] == 'Culvert':
            self.data.data_dict['bridge_width'] = float(self._widgets['edt_culvert_width'].text())
            self.data.data_dict['bridge_ceiling_roughness'] = float(self._widgets['edt_culvert_manning'].text())
            self.data.data_dict['bridge_wrapping_width'] = float(self._widgets['edt_culvert_wrap_width'].text())
            val = 1 if self._widgets['tog_culvert_abutments'].isChecked() else 0
            self.data.data_dict['has_abutments'] = val
            self.data.data_dict['pier_wall_num_side_elem'] = int(self._widgets['edt_culvert_num_side'].text())
        self.data.data_dict['elev_raster'] = self.elev_uuid

    def accept(self):
        """Save data to persistent storage on OK."""
        try:
            self.get_dlg_data()
            add_ugrid_toggle = self._widgets['tog_add_ugrid']
            if self.data.data_dict['structure_type'] == 'Culvert':
                add_ugrid_toggle = self._widgets['tog_culvert_add_ugrid']

            self.arc_data['elev_file'] = self.elev_file
            ug_success, msg = su.generate_ugrids_from_structure(self.data, self.arc_data, self.footprint_calc)
            if not ug_success and add_ugrid_toggle.isChecked():
                QMessageBox.warning(self, '3D Structure', msg, QMessageBox.Ok)
                raise RuntimeError
            elif add_ugrid_toggle.isChecked():
                self.output_ugrid = self.arc_data['output_ugrid']
            self._add_twod_mesh_to_xms()
            self._add_coverage_to_xms()

            self.close_map_viewer_proc()
            super().accept()
        except RuntimeError:
            return

    def _add_coverage_to_xms(self):
        """Add the coverage to XMS."""
        add_coverage_toggle = self._widgets['tog_add_coverage']
        if self.data.data_dict['structure_type'] == 'Culvert':
            add_coverage_toggle = self._widgets['tog_add_culvert_coverage']

        if add_coverage_toggle.isChecked():
            if self.footprint_calc.bridge_mesh_coverage is not None:
                self.output_coverage = self.footprint_calc.bridge_mesh_coverage
            else:
                msg = 'Unable to generate the coverage for the 3D Structure. Preview the mesh to ensure ' \
                      'that all parameters are correct.'
                QMessageBox.warning(self, '3D Structure', msg, QMessageBox.Ok)
                raise RuntimeError

    def _add_twod_mesh_to_xms(self):
        """Add the 2D mesh to XMS."""
        add_2dmesh_toggle = self._widgets['tog_add_footprint_ugrid']
        if self.data.data_dict['structure_type'] == 'Culvert':
            add_2dmesh_toggle = self._widgets['tog_add_culvert_footprint_ugrid']

        if add_2dmesh_toggle.isChecked():
            if self.footprint_calc.bridge_mesh is not None:
                tmp_grd = xmf.temp_filename()
                self.footprint_calc.bridge_mesh.write_to_file(tmp_grd)
                name = f'{self.arc_data["coverage"].name}_footprint_grid'
                self.output_footprint_ugrid = DoUGrid(
                    cogrid_file=tmp_grd, name=name, projection=self.arc_data['projection']
                )
            else:
                msg = 'Unable to generate the 2D mesh for the 3D Structure. ' \
                      'Preview the mesh to ensure that all parameters are correct.'
                QMessageBox.warning(self, '3D Structure', msg, QMessageBox.Ok)
                raise RuntimeError

    def reject(self):
        """Cancel command."""
        self.close_map_viewer_proc()
        super().reject()


def run_structure_dlg(parent, xms_data):
    """Runs the structure properties dialog.

    Args:
        parent: parent window
        xms_data (:obj:`XmsData`): Object for retrieving data from XMS
    """
    structure_comp = xms_data.structure_component
    arc_data = su.setup_structure_generation(xms_data)
    if structure_comp.data.data_dict['srh_weir_length'] <= 0.0:
        if 'arc_pts' in arc_data:
            ls = LineString(arc_data['arc_pts'])
            structure_comp.data.data_dict['srh_weir_length'] = round(ls.length, 1)

    dlg = StructureDialog(parent, structure_comp.data, arc_data)
    if dlg.exec():
        dlg.data.commit()
        if dlg.output_ugrid is not None:
            xms_data.add_ugrid(dlg.output_ugrid)
        if dlg.output_coverage is not None:
            xms_data.add_coverage(dlg.output_coverage)
        if dlg.output_footprint_ugrid is not None:
            xms_data.add_ugrid(dlg.output_footprint_ugrid)
        xms_data.update_component_ids()

    shutil.rmtree(arc_data['tmp_dir'], ignore_errors=True)


def _is_float_raster(raster):
    """Determines if the raster is a float raster.

    Args:
        raster (:obj:`RasterInput`): the raster

    Returns:
        (:obj:`bool`): True if the raster is a float raster
    """
    if raster.gdal_raster.RasterCount > 1:
        return False
    if raster.data_type not in [raster.GDT_Float32, raster.GDT_Float64]:
        return False
    return True
