"""The Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QSize, Qt
from PySide2.QtWidgets import (QDialogButtonBox, QHBoxLayout, QScrollArea, QTabWidget, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy.data.plot_and_table_data_srh import PlotsAndTableDataSrh
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.bridge import structure_util as su
from xms.bridge.gui.summary_dlg.plot_sim_data import PlotSimData
from xms.bridge.gui.summary_dlg.profile_tab import ProfileTab
from xms.bridge.gui.summary_dlg.srh_plot_tab import StructureSrhPlotTab
from xms.bridge.gui.summary_dlg.summary_tab import StructureSummaryTab


class SummaryDialog(XmsDlg):
    """A dialog for editing structure properties."""
    def __init__(self, parent, plot_data=None, xms_data=None, cl_arc_pts=None, help_url=None):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (:obj:`QWidget`): Parent window
            plot_data (:obj:`PlotsAndTableDataSrh`): Plot data
            xms_data (:obj:`XmsData`): Object for retrieving data from XMS
            cl_arc_pts (:obj:`list`): List of center line arc points
            help_url (:obj:`str`): URL for the help page
        """
        super().__init__(parent, 'xms.bridge.summary_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:3D_Structure_Summary' if not help_url else help_url
        self.title = '3D Structure Summary'
        self.description = 'Summary for the 3D structure.'
        self._plot_data = plot_data
        self._xms_data = xms_data
        self._cl_arc_pts = cl_arc_pts if cl_arc_pts else [0., 0.]
        self._summary_tab = None
        self._struct_plot_tab = None
        self._profile_tab = None
        self._cross_section_tab = None
        self._widgets = {}
        self._setup_ui()
        self.setMinimumSize(QSize(400, 400))

    def _setup_ui(self):
        """Sets up the widgets for the UI."""
        self.setWindowTitle(self.title)
        self._widgets['top_layout'] = QVBoxLayout()
        self.setLayout(self._widgets['top_layout'])
        self._widgets['controls_widget'] = QWidget()
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['controls_widget'].setLayout(self._widgets['main_vert_layout'])
        self._widgets['top_layout'].addWidget(self._widgets['controls_widget'])

        self._add_widgets_to_main_layout()
        self._add_button_box()

    def _add_widgets_to_main_layout(self):
        """Adds widgets to the main layout."""
        layout = self._widgets['main_vert_layout']

        self._widgets['tab0'] = QWidget()
        self._widgets['tab1'] = QWidget()
        self._widgets['tab2'] = QWidget()
        self._widgets['tab3'] = QWidget()
        self._widgets['tabs'] = QTabWidget()

        layout.addWidget(self._widgets['tabs'])
        self._make_tab_layout(0)
        self._struct_plot_tab = StructureSrhPlotTab(self, self._widgets, self._plot_data)
        self._make_tab_layout(1)
        self._summary_tab = StructureSummaryTab(self, self._widgets, self._plot_data)
        self._make_tab_layout(2)
        plot_sim_data = PlotSimData(self._xms_data)
        self._profile_tab = ProfileTab(self, self._widgets, self._xms_data, [], plot_sim_data)
        self._make_tab_layout(3)
        self._cross_section_tab = ProfileTab(self, self._widgets, self._xms_data, self._cl_arc_pts, plot_sim_data)

        self._widgets['tabs'].addTab(self._widgets['tab0'], 'Structure Plots')
        self._widgets['tabs'].addTab(self._widgets['tab1'], 'Summary')
        self._widgets['tabs'].addTab(self._widgets['tab2'], 'Profile Plot')
        self._widgets['tabs'].addTab(self._widgets['tab3'], 'Cross Section Plot')

    def _make_tab_layout(self, idx):
        """Create layout for a tab control.

        Args:
            idx (:obj:`int`): index of the tab
        """
        self._widgets[f'tab{idx}_vert_layout'] = QVBoxLayout()
        self._widgets[f'tab{idx}'].setLayout(self._widgets[f'tab{idx}_vert_layout'])
        self._widgets[f'tab{idx}_scroll_area'] = QScrollArea(self._widgets[f'tab{idx}'])
        self._widgets[f'tab{idx}_vert_layout'].addWidget(self._widgets[f'tab{idx}_scroll_area'])
        self._widgets[f'tab{idx}_scroll_area'].setWidgetResizable(True)
        self._widgets[f'tab{idx}_scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self._widgets[f'tab{idx}_scroll_area_widget'] = QWidget()
        self._widgets[f'tab{idx}_scroll_area'].setWidget(self._widgets[f'tab{idx}_scroll_area_widget'])
        self._widgets[f'tab{idx}_scroll_area_vert_layout'] = QVBoxLayout()
        self._widgets[f'tab{idx}_scroll_area_widget'].setLayout(self._widgets[f'tab{idx}_scroll_area_vert_layout'])

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['top_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)


def run_summary_dlg(parent, xms_data):
    """Runs the structure properties dialog.

    Args:
        parent: parent window
        xms_data (:obj:`XmsData`): Object for retrieving data from XMS
    """
    coverage = xms_data.coverage
    comp = xms_data.structure_component
    arc, arc_pts = su.bridge_centerline_from_coverage(coverage, comp)
    arc_id = -1
    if arc is not None:
        arc_id = arc.id
    dlg_data = PlotsAndTableDataSrh(
        pe_tree=xms_data.project_tree,
        feature_id=arc_id,
        feature_type='Arc',
        cov_uuid=coverage.uuid,
        model_name='SRH-2D'
    )

    dlg = SummaryDialog(parent=parent, plot_data=dlg_data, xms_data=xms_data, cl_arc_pts=arc_pts)
    if dlg.exec():
        comp = xms_data.structure_component
        comp.data.commit()
