"""This module is for the boundary conditions hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.cmsflow.components.cmsflow_component import CmsflowComponent
from xms.cmsflow.components.id_files import BC_INITIAL_ATT_ID_FILE, BC_INITIAL_COMP_ID_FILE
from xms.cmsflow.data.bc_data import BCData
from xms.cmsflow.dmi.xms_data import XmsData
from xms.cmsflow.file_io import io_util
from xms.cmsflow.gui.boundary_condition_dlg import BoundaryConditionDlg


class BCComponent(CmsflowComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Flow model simulation."""
    def __init__(self, main_file: str | Path):
        """
        Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Display Options...', 'open_display_options')
        ]
        self.arc_commands = [('Assign Boundary Conditions...', 'open_arc_attributes')]
        self.data = BCData(self.main_file)
        self.comp_id_files = {
            'Unassigned': 'unassigned_arcs.display_ids',
            'Flow rate-forcing': 'flow_arcs.display_ids',
            'WSE-forcing': 'wse_arcs.display_ids'
        }
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), 'bc_display_options.json')
        if os.path.exists(self.main_file):
            self.cov_uuid = self.data.info.attrs['cov_uuid']
            if not os.path.exists(self.disp_opts_file):
                # Read the default arc display options, and save ourselves a copy with a randomized UUID.
                arc_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                arc_default_file = os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                    'bc_default_display_options.json'
                )
                arc_json_dict = read_display_options_from_json(arc_default_file)
                arc_categories.from_dict(arc_json_dict)
                arc_categories.comp_uuid = self.uuid
                write_display_options_to_json(self.disp_opts_file, arc_categories)
                # Save our display list UUID to the main file
                self.data.info.attrs['arc_display_uuid'] = arc_categories.uuid
                self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            # Change the display UUID to be something different.
            new_comp = BCComponent(new_main_file)
            new_uuid = self._update_display_uuids_for_duplicate(new_comp.uuid, new_comp.disp_opts_file)
            new_comp.data.info.attrs['arc_display_uuid'] = new_uuid
            new_comp.data.commit()
            action_requests.append(self.get_display_refresh_action())

        # Update external file references if saving new project
        if save_type == 'SAVE':  # Update paths in existing main file
            error = self.data.update_file_paths()
            if error:
                messages.append(('ERROR', error))
        elif save_type in ['SAVE_AS', 'PACKAGE']:  # Update filepaths in the new main file.
            is_package = save_type == 'PACKAGE'
            if is_package:  # If doing a Save As Package copy all referenced external files.
                error = self.copy_external_files(new_main_file)
            else:
                error = self.update_proj_dir(new_main_file, True)
            if error:
                messages.append(('ERROR', error))
        elif save_type == 'UNLOCK':
            # Store off the project directory in case this is the first time a package project has been opened.
            # If no project save location, try to resolve any paths by going up to the project directory. This happens
            # after a package save. All file paths will be in the project directory. Otherwise, all paths should be
            # absolute until the first save.
            error = self.update_proj_dir(new_main_file, False)
            if error:
                messages.append(('ERROR', error))
        return new_main_file, messages, action_requests

    def update_id_files(self):
        """Writes the display id files."""
        for display_name, filename in self.comp_id_files.items():
            vals = self.data.arcs.where(self.data.arcs.bc_type == display_name, drop=True)
            if len(vals.comp_id) > 0:
                id_file = os.path.join(os.path.dirname(self.main_file), f'{filename}')
                write_display_option_ids(id_file, vals.comp_id.data)
            else:  # If we don't have any of this type, just delete the id file.
                io_util.remove(filename)

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_refresh_action()]
        return messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        if query is not None:
            self.cov_uuid = query.parent_item_uuid()
            if not self.cov_uuid:
                return [('ERROR', 'Could not get CMS-Flow boundary conditions coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.main_file), BC_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), BC_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            if query is not None:
                # Wrapped in if statement fot testing
                io_util.remove(initial_att_file)
                io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.arc, att_id, comp_id)
            self.update_id_files()

        self.data.info.attrs['cov_uuid'] = self.cov_uuid
        self.data.commit()
        # # Send the default display list to XMS.
        self.display_option_list.append(
            XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        return [], []

    def open_arc_attributes(self, query, params, win_cont):
        """Opens the save point dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        comp_id = self._unpack_xms_data(params[0], TargetType.arc)
        incremented_id = False
        if comp_id not in self.data.arcs.comp_id:  # Create default attributes if the arc(s) have never been assigned.
            comp_id = self.data.add_bc_arc_atts()
            incremented_id = True  # We just got a new component id, so don't duplicate on OK
        single_arc = self.data.arcs.loc[dict(comp_id=[comp_id])]
        xms_data = XmsData(query)
        dlg = BoundaryConditionDlg(
            single_arc, self.data, comp_id, query.project_tree, xms_data, self.dlg_message, win_cont
        )
        if XmEnv.xms_environ_running_tests() == 'TRUE' or dlg.exec_():
            if incremented_id:  # Was previously unassigned, so we already got a new component id. Just update it.
                self.data.update_bc_arc(comp_id, dlg.bc_data)
            else:  # Always generate a new component id, old one might be shared if assigned during a multi-select.
                comp_id = self.data.add_bc_arc_atts(dlg.bc_data)
            self._append_comp_id_to_file(self.comp_id_files[dlg.bc_data['bc_type'].item()], comp_id)
            self.data.commit()
            for pt_id in self.selected_att_ids:
                self.update_component_id(TargetType.arc, pt_id, comp_id)
            # Send back updated display lists to XMS after ActionRequest
            self.display_option_list.append(
                # Update the arc display list only.
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid)
            )
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for boundary condition arcs.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Read the default arc display options, and save ourselves a copy with a randomized UUID.
        arc_categories = CategoryDisplayOptionList()
        arc_json_dict = read_display_options_from_json(self.disp_opts_file)
        arc_categories.from_dict(arc_json_dict)

        dlg = CategoryDisplayOptionsDialog([arc_categories], win_cont)
        if dlg.exec_():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    # def _remove_unused_component_ids(self):
    #     """Remove unused component ids from the data and display option id files.
    #
    #     Notes:
    #         Coverage component ids must be loaded from the Query before calling this.
    #     """
    #     # Drop unused component ids from the datasets
    #     comp_ids = list(self.comp_to_xms.get(self.cov_uuid, {}).get(TargetType.arc, {}).keys())
    #     mask = self.data.arcs.comp_id.isin(comp_ids)
    #     self.data.arcs = self.data.arcs.where(mask, drop=True)
    #     # Rewrite the display option id files now to clear out non-existent component ids.
    #     self.update_id_files()
    #     self.data.vacuum()  # Reclaim disk space in the NetCDF file

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (str): The location of the new main file.
            convert_filepaths (bool): False if only the project directory should be updated.

        Returns:
            (str): Message on failure, empty string on success
        """
        new_data = BCData(new_main_file)
        if not convert_filepaths:
            # This case is to handle opening a package project for the first time.
            comp_folder = os.path.dirname(self.main_file)
            package_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            new_data.info.attrs['proj_dir'] = package_proj_dir
            new_data.commit()  # Save the updated project directory
            return ''
        err_msg = new_data.update_proj_dir()
        # Copy the newly saved file to temp.
        io_util.copyfile(new_main_file, self.main_file)
        return err_msg

    def copy_external_files(self, new_main_file):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_main_file (str): The location of the new component main file in the package.

        Returns:
            (str): Message on failure, empty string on success
        """
        new_data = BCData(new_main_file)
        new_data.copy_external_files()
        new_data.commit()
        return ''
