"""This module is a group of utility functions for dealing with files."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.adcirc.file_io.fort14_reader import Fort14Reader
from xms.adcirc.file_io.fort15_reader import DataForThe14
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree.tree_util import find_tree_node_by_uuid
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.cmsflow.feedback.xmlog import XmLog


class ParentAdcircReader:
    """Reads Adcric solutions in h5 format."""
    def __init__(self, pe_tree, file_map):
        """Constructor."""
        self._pe_tree = pe_tree
        self._file_map = file_map
        self._ugrids_to_build = {}
        self._elvation_uuid = ''  # UUID of the current elevation dataset
        self._velocity_uuid = ''  # UUID of the current velocity dataset
        self._testing = XmEnv.xms_environ_running_tests() == 'TRUE'  # Don't log absolute paths

        self.ugrids = []  # [data_objects.parameters.UGrid]
        self.datasets = {}  # {Dataset uuid: DatasetReader}
        self.bc_dset_uuids = {}  # {bc_id: (geom_uuid, elev_dset_uuid, velocity_dset_uuid)}

    def read(self):
        """Reads the datasets and ugrids."""
        try:
            self._read_solutions_datasets()
            self._read_ugrids()
        except Exception:
            XmLog().instance.exception('Error reading adcirc solutions.')

    def _read_solutions_datasets(self):
        """Read the solution datasets."""
        for bc_id, adcirc_file in self._file_map.items():
            geom_uuid = ''
            elevation_uuid = ''
            velocity_uuid = ''
            if adcirc_file['wse_solution'] and os.path.isfile(adcirc_file['wse_solution'][0]):
                filename = adcirc_file["wse_solution"][0] if not self._testing else \
                    os.path.basename(adcirc_file["wse_solution"][0])
                XmLog().instance.info(f'Reading parent adcirc file {filename}')
                elevation_uuid, geom_uuid = self._get_datasets_and_ugrids(
                    adcirc_file['wse_solution'][0], adcirc_file['wse_solution'][1], adcirc_file['fort14']
                )
            if adcirc_file['velocity_solution'] and os.path.isfile(adcirc_file['velocity_solution'][0]):
                filename = adcirc_file["velocity_solution"][0] if not self._testing else \
                    os.path.basename(adcirc_file["velocity_solution"][0])
                XmLog().instance.info(f'Reading parent adcirc file {filename}')
                velocity_uuid, geom_uuid = self._get_datasets_and_ugrids(
                    adcirc_file['velocity_solution'][0], adcirc_file['velocity_solution'][1], adcirc_file['fort14']
                )
            self.bc_dset_uuids[bc_id] = (geom_uuid, elevation_uuid, velocity_uuid)

    def _get_datasets_and_ugrids(self, dataset_file, group_path, fort14_file):
        """Get the datasets and ugrids to read based on the adcirc files passed in.

        Args:
            dataset_file (str):  Path to the dataset .h5 file.
            group_path (str):  Name of the group path in the dataset file.
            fort14_file (str):  Path to the adcirc fort14 file.

        Returns:
            tuple[str, str]: The UUID of the dataset, UUID of the dataset's geometry
        """
        dset_reader = DatasetReader(h5_filename=dataset_file, group_path=group_path)
        if dset_reader.uuid not in self.datasets:
            dset_item = find_tree_node_by_uuid(tree_node=self._pe_tree, uuid=dset_reader.uuid)
            if not dset_item:
                # This doesn't exist, so add it
                self.datasets[dset_reader.uuid] = dset_reader
                if dset_reader.geom_uuid not in self._ugrids_to_build:
                    ugrid_item = find_tree_node_by_uuid(tree_node=self._pe_tree, uuid=dset_reader.geom_uuid)
                    if not ugrid_item:
                        self._ugrids_to_build[dset_reader.geom_uuid] = fort14_file
        return dset_reader.uuid, dset_reader.geom_uuid

    def _read_ugrids(self):
        """Reads the ugrids to build."""
        # Create a fake Components folder that will get cleaned up when the process ends. The ADCIRC fort.14 reader
        # will create a mapped BC component that we don't want.
        comp_dir = os.path.join(XmEnv.xms_environ_process_temp_directory(), 'Components')
        os.makedirs(comp_dir, exist_ok=True)
        data_from_15 = DataForThe14()
        data_from_15.comp_dir = comp_dir
        for ugrid_uuid, fort14_file in self._ugrids_to_build.items():
            XmLog().instance.info(
                f'Reading parent adcirc file {fort14_file if not self._testing else os.path.basename(fort14_file)}'
            )
            data_from_15.mesh_uuid = ugrid_uuid
            reader = Fort14Reader(filename=fort14_file, query=True, datafrom15=data_from_15)
            reader.read()
            ugrid = reader.built_data['domain_mesh']
            ugrid.uuid = ugrid_uuid
            self.ugrids.append(ugrid)
