"""This is a table widget for specifying advanced cards."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules
from xms.cmsflow.gui.cmsflow_table_widget import CmsflowTableWidget


class AdvancedCardTableWidget(CmsflowTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
        """
        self.card_method_delegate = None
        super().__init__(parent, data_frame, 0, {'block_name': '', 'card': '', 'value': ''})
        self.model.show_nan_as_blank = True
        self.table_view.setColumnHidden(0, True)

    def set_user_blocks(self, user_blocks):
        """Set the available default and user blocks in the user defined cards table.

        Args:
            user_blocks: The user blocks
        """
        blocks = [
            '-- No Block --',
            'BED_LAYER',
            'BOUNDARY',
            'DREDGE',
            'DREDGE_OPERATION',
            'FLUX',
            'HARMONIC',
            'HORIZONTAL_PROJECTION',
            'MET_STATION',
            'PARENT',
            'PLACEMENT',
            'RUBBLE_MOUND',
            'SEDIMENT_SIZE_CLASS',
            'TIDAL',
            'VERTICAL_PROJECTION',
            'WSE',
        ]
        # change everything into a string, prevents a problem with NaN
        user_blocks = [str(user_block) for user_block in user_blocks]
        blocks.extend(user_blocks)
        self.card_method_delegate.set_strings(blocks)

        # remove rows from card table that were using the deleted blocks
        bad_rows = []
        data_frame = self.model.data_frame
        for row, block in enumerate(data_frame['Block']):
            if type(block) is float and np.isnan(block):  # Mantis issue 14891
                block = '-- No Block --'
            if block not in blocks:
                bad_rows.append(row)
        bad_rows.reverse()
        for row in bad_rows:
            self.model.removeRows(row, 1)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        row_idx = self.model.rowCount() - 1

        # Empty tables default to float, even if they should be str. That gives us NaNs, so force to string.
        for column in range(self.model.columnCount()):
            model_index = self.model.index(row_idx, column)
            self.model.setData(model_index, '')

    def setup_ui(self):
        """Sets up the table."""
        self.card_method_delegate = QxCbxDelegate(self)
        delegates = {0: self.card_method_delegate}
        super()._setup_ui(delegates, False, False)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
