"""This is a table widget for specifying meteorological stations."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QHeaderView

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.cmsflow.gui.cmsflow_table_widget import CmsflowTableWidget
from xms.cmsflow.gui.meteorological_curve_button_delegate import MeteorologicalCurveButtonDelegate


class MeteorologicalStationsTableWidget(CmsflowTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame, sim_data):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
            sim_data (SimulationData): The simulation data to view
        """
        self.rename_model = None
        self.met_curve_delegate = None
        self.edit_delegate = None
        self.dbl_validator = None
        self.sim_data = sim_data
        super().__init__(parent, data_frame, 0, {'name': '', 'x': 0.0, 'y': 0.0, 'height': 0.0, 'direction': -1})

    def setup_ui(self):
        """Sets up the delegates, validators, and filter model for this table."""
        self.rename_model = RenameModel(['Name', 'X (m)', 'Y (m)', 'Height (m)', 'Wind Curves'], self)
        self.met_curve_delegate = MeteorologicalCurveButtonDelegate(
            self._get_met_dataframe, self._set_met_dataframe, self.parent()
        )
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        delegates = {1: self.edit_delegate, 2: self.edit_delegate, 3: self.edit_delegate, 4: self.met_curve_delegate}
        super()._setup_ui(delegates, False, False, self.rename_model)
        self.table_view.verticalHeader().setDefaultSectionSize(75)
        self.table_view.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)

    def _get_met_dataframe(self, curve_id):
        """Returns a dataframe.

        Args:
            curve_id (int): The wind direction curve id.

        Returns:
            A pandas dataframe for the curve and the new curve id.
            If the given curve id was valid, then it will be the same id.
        """
        if curve_id < 0:
            curve_id = int(self.sim_data.meteorological_stations_table.attrs['NEXT_CURVE_ID'])
            self.sim_data.meteorological_stations_table.attrs['NEXT_CURVE_ID'] = curve_id + 1
        return self.sim_data.direction_curve_from_meteorological_station(curve_id).to_dataframe(), curve_id

    def _set_met_dataframe(self, curve_id, dataframe):
        """Sets the meteorological station curve.

        Args:
            curve_id (int): The wind direction curve id.
            dataframe (pandas.Dataframe): The wind direction curve.
        """
        self.sim_data.set_direction_curve_from_meteorological_station(curve_id, dataframe.to_xarray())
