"""Map Rubble Mound Jetties coverage locations and attributes to the CMS-Flow domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    write_display_option_polygon_locations, write_display_options_to_json
)
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType
from xms.snap.snap_polygon import SnapPolygon

# 4. Local modules
from xms.cmsflow.components.mapped_rm_component import MappedRmComponent


class RmStructuresMapper:
    """Class for mapping rubble mound jetties coverage to a quadtree for CMS-Flow."""
    def __init__(self, coverage_mapper, generate_snap):
        """Constructor."""
        self._snap_poly = SnapPolygon()
        self._snap_poly.set_grid(grid=coverage_mapper._quadtree, target_cells=False)
        self._snap_poly.add_polygons(coverage_mapper._rm_cov.polygons)
        self._polygon_to_grid_cells = {}
        self._logger = coverage_mapper._logger
        self._wkt = coverage_mapper._wkt
        self._generate_snap = generate_snap
        self._co_grid = coverage_mapper._quadtree
        self._poly_to_cells = {}
        self._rm_coverage = coverage_mapper._rm_cov
        self._new_comp_unique_name = 'Mapped_Rm_Component'
        self.rm_mapped_comp_uuid = None
        self.rm_mapped_comp_display_uuid = None
        self._comp_uuid = ''
        self._rm_component = coverage_mapper._rm_comp
        self._rm_component_file = coverage_mapper._rm_comp.main_file
        self._poly_files = []

    def do_map(self):
        """Creates the mapped rubble mound component."""
        self._get_polygon_cells()
        if self._generate_snap:
            self._create_component_folder()
            self._create_drawing()
            self._create_display_options()
            # Create the data_objects component
            do_comp = Component(
                main_file=self._comp_main_file,
                name=f'Snapped {self._rm_coverage.name} display',
                model_name='CMS-Flow',
                unique_name=self._new_comp_unique_name,
                comp_uuid=os.path.basename(os.path.dirname(self._comp_main_file))
            )
            comp = MappedRmComponent(self._comp_main_file)
            return do_comp, comp
        return None, None  # pragma: no cover

    @property
    def snapper(self):
        """Gets the snapper used for rubble mound jetty polygons.

        Returns:
            The polygon snapper used for rubble mound jetty polygons.
        """
        return self._snap_poly

    def _create_component_folder(self):
        """Creates the folder for the mapped rubble mound jetties component."""
        if self.rm_mapped_comp_uuid is None:
            self._comp_uuid = str(uuid.uuid4())  # pragma: no cover
        else:
            self._comp_uuid = self.rm_mapped_comp_uuid
        self._logger.info('Creating component folder')
        rm_comp_path = os.path.dirname(self._rm_component_file)
        self._comp_path = os.path.join(os.path.dirname(rm_comp_path), self._comp_uuid)

        if os.path.exists(self._comp_path):
            shutil.rmtree(self._comp_path)  # pragma: no cover
        os.mkdir(self._comp_path)

    def _create_display_options(self):
        """Creates the display options file for the mapped rubble mound jetties component."""
        comp_display_file = os.path.join(self._comp_path, 'rm_display_options.json')

        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        if self.rm_mapped_comp_display_uuid is None:
            categories.uuid = str(uuid.uuid4())  # pragma: no cover
        else:
            categories.uuid = self.rm_mapped_comp_display_uuid
        categories.comp_uuid = self._comp_uuid
        categories.is_ids = False
        categories.target_type = TargetType.polygon
        for names in self._poly_files:
            cat = CategoryDisplayOption()
            cat.file = names[0]
            cat.description = names[1]
            cat.is_unassigned_category = False
            cat.label_on = False
            cat.options = PolygonOptions()
            categories.categories.append(cat)
        categories.projection = {'wkt': self._wkt}

        write_display_options_to_json(comp_display_file, categories)
        self._comp_main_file = comp_display_file

    def _create_drawing(self):
        """Uses cell ids to get cell point coords to draw polygons for rubble mound jetties mapped to cells."""
        ugrid = self._co_grid.ugrid
        for comp_id, cell_ids in self._poly_to_cells.items():
            poly_list = []
            for cid in cell_ids:
                cell_locs = ugrid.get_cell_locations(cid)
                locs_list = [item for sublist in cell_locs for item in sublist]
                if len(locs_list) < 9:
                    continue  # pragma: no cover
                # repeat the first point
                locs_list.append(locs_list[0])
                locs_list.append(locs_list[1])
                locs_list.append(locs_list[2])
                outer_dict = {'outer': locs_list}
                poly_list.append(outer_dict)
            filename = os.path.join(self._comp_path, f'rm_{comp_id}.jetties')
            names = self._rm_component.data.polygons.where(self._rm_component.data.polygons.comp_id == comp_id)['name']
            name = names.values.tolist()[0]
            self._poly_files.append((filename, name))
            write_display_option_polygon_locations(filename, poly_list)

    def _get_polygon_cells(self):
        """Uses xmssnap to get the cells for each polygon."""
        self._logger.info('Mapping rubble mound jetties to quadtree.')
        polys = self._rm_coverage.polygons
        for poly in polys:
            pid = poly.id
            cells = self._snap_poly.get_cells_in_polygon(pid)
            comp_id = self._rm_component.get_comp_id(TargetType.polygon, pid)
            if comp_id is None:
                comp_id = 0  # pragma: no cover
            if comp_id not in self._poly_to_cells:
                self._poly_to_cells[comp_id] = []
            self._poly_to_cells[comp_id].extend(cells)
