"""Map Save Points coverage locations and attributes to the CMS-Flow domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_options_from_json, write_display_options_to_json)
from xms.components.display.display_options_io import write_display_option_point_locations
from xms.data_objects.parameters import Component, FilterLocation
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.snap.snap_point import SnapPoint

# 4. Local modules
from xms.cmsflow.components.mapped_save_points_component import MappedSavePointsComponent


class SavePointsMapper:
    """Class for mapping Save Points coverage to a quadtree for CMS-Flow."""
    def __init__(self, coverage_mapper, generate_snap):
        """Constructor."""
        self._snapper = SnapPoint()
        if coverage_mapper._small_quadtree:
            self._snapper.set_grid(grid=coverage_mapper._small_quadtree, target_cells=True)
        else:
            self._snapper.set_grid(grid=coverage_mapper._quadtree, target_cells=True)
        self._snapped_grid_points = None
        self._logger = coverage_mapper._logger
        self._wkt = coverage_mapper._wkt
        self._generate_snap = generate_snap
        self._save_pts_coverage = coverage_mapper._save_pts_cov
        self._save_pts_component = coverage_mapper._save_pts_comp
        self.mapped_comp_uuid = None
        self.mapped_comp_display_uuid = None
        self._new_comp_unique_name = 'Mapped_SavePoints_Component'
        self._new_comp_main_file = ''
        self._new_comp_path = ''
        self._save_pts_component_file = self._save_pts_component.main_file

    @property
    def snapper(self):
        """Gets the snapper used for boundary condition arcs.

        Returns:
            The arc snapper used for boundary condition arcs.
        """
        return self._snapper

    def do_map(self):
        """Creates the mapped boundary conditions component."""
        self._get_grid_points_from_points()
        if self._generate_snap:
            self._create_component_folder_and_copy_display_options()
            self._create_drawing()
            # Create the data_objects component
            do_comp = Component(
                main_file=self._new_comp_main_file,
                name=f'Snapped {self._save_pts_coverage.name} display',
                model_name='CMS-Flow',
                unique_name=self._new_comp_unique_name,
                comp_uuid=os.path.basename(os.path.dirname(self._new_comp_main_file))
            )
            comp = MappedSavePointsComponent(self._new_comp_main_file)
            return do_comp, comp
        return None, None  # pragma: no cover

    def _create_component_folder_and_copy_display_options(self):
        """Creates the folder for the mapped bc component and copies the display options from the bc coverage."""
        if self.mapped_comp_uuid is None:
            comp_uuid = str(uuid.uuid4())  # pragma: no cover
        else:
            comp_uuid = self.mapped_comp_uuid
        self._logger.info('Creating component folder')
        old_comp_path = os.path.dirname(self._save_pts_component_file)
        self._new_comp_path = os.path.join(os.path.dirname(old_comp_path), comp_uuid)

        if os.path.exists(self._new_comp_path):
            shutil.rmtree(self._new_comp_path)  # pragma: no cover
        os.mkdir(self._new_comp_path)

        bc_comp_display_file = os.path.join(old_comp_path, 'save_point_display_options.json')
        comp_display_file = os.path.join(self._new_comp_path, 'save_point_display_options.json')
        if os.path.isfile(bc_comp_display_file):
            shutil.copyfile(bc_comp_display_file, comp_display_file)
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            json_dict = read_display_options_from_json(comp_display_file)
            if self.mapped_comp_display_uuid is None:
                json_dict['uuid'] = str(uuid.uuid4())  # pragma: no cover
            else:
                json_dict['uuid'] = self.mapped_comp_display_uuid
            json_dict['comp_uuid'] = comp_uuid
            json_dict['is_ids'] = 0
            categories.from_dict(json_dict)
            categories.projection = {'wkt': self._wkt}

            # TODO: Do we want to change the display of snapped points in any way?

            write_display_options_to_json(comp_display_file, categories)
            self._new_comp_main_file = comp_display_file
        else:
            self._logger.info('Could not find save_point_display_options.json file')  # pragma: no cover

    def _create_drawing(self):
        """Uses cell ids to get cell point coords to draw points for mapped cells."""
        if self._snapped_grid_points is not None:
            filename = os.path.join(self._new_comp_path, 'save_point.display_ids')
            write_display_option_point_locations(filename, self._snapped_grid_points)

    def _get_grid_points_from_points(self):
        """Uses xmssnap to get the points from points."""
        self._logger.info('Snapping Save Points coverage to quadtree.')
        points = self._save_pts_coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        snap_output = self._snapper.get_snapped_points(points)
        if snap_output:
            self._snapped_grid_points = snap_output['location'].flatten()
