"""This handles the hidden component side of merging multiple rubble mound coverages together."""

# 1. Standard Python modules
import os
import traceback

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import read_display_option_ids, write_display_option_ids

# 4. Local modules
from xms.cmsflow.components.id_files import RM_INITIAL_ATT_ID_FILE, RM_INITIAL_COMP_ID_FILE
from xms.cmsflow.data.rm_structures_data import RMStructuresData


class RubbleMoundMerger:
    """Handles merging of Rubble Mound Jetties coverages."""
    ATT_FILE_IDX = 0
    COMP_FILE_IDX = 1
    MAIN_FILE_IDX = 0
    ID_FILE_IDX = 1

    def __init__(self, comp):
        """Construct the merger.

        Args:
            comp (SavePointsComponent): The Save Points coverage component to merge data into

        """
        self._comp = comp
        self._poly_att_ids = []
        self._poly_comp_ids = []

    def _merge_poly_data(self, data, att_id_file, comp_id_file):
        """Copy over the polygon attributes of a coverage in the merge list.

        Args:
            data (RMStructuresData): The data to merge
            att_id_file (str): File path to the source XMS feature id file
            comp_id_file (str): File path to the source component id file

        """
        old_to_new_station_point_ids = self._comp.data.concat_rm_atts(data)
        att_ids = read_display_option_ids(att_id_file)
        comp_ids = read_display_option_ids(comp_id_file)
        self._poly_att_ids.extend(att_ids)
        for comp_id in comp_ids:
            self._poly_comp_ids.append(old_to_new_station_point_ids[comp_id])

    def _update_id_files(self):
        """Store save point att ids and comp ids in a temp file so they can be processed later.

        Also writes the persistent display id referenced in the display options json file for the component.

        """
        point_att_file = os.path.join(os.path.dirname(self._comp.main_file), RM_INITIAL_ATT_ID_FILE)
        write_display_option_ids(point_att_file, self._poly_att_ids)
        point_comp_file = os.path.join(os.path.dirname(self._comp.main_file), RM_INITIAL_COMP_ID_FILE)
        write_display_option_ids(point_comp_file, self._poly_comp_ids)

    def merge(self, merge_list):
        """Merge Save Points coverage data into destination component.

        Args:
            merge_list (:obj:`list` of :obj:`tuple`): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        """
        messages = []
        for cov in merge_list:
            try:
                # TODO: Merge display options of the first coverage in the list when implemented.

                # Concatenate the polygon data.
                rubble_mound_data = RMStructuresData(cov[self.MAIN_FILE_IDX])
                id_files = cov[self.ID_FILE_IDX].get('POLYGON')
                if not id_files:
                    continue  # Empty coverage

                self._merge_poly_data(
                    data=rubble_mound_data,
                    att_id_file=id_files[self.ATT_FILE_IDX],
                    comp_id_file=id_files[self.COMP_FILE_IDX]
                )
            except Exception as ex:
                trace = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                messages.append(('ERROR', f'Could not merge Rubble Mound Jetties coverage attributes: {trace}'))

        self._update_id_files()

        # Wipe the old coverage's UUID. Will Query for a the new coverage's UUID in get_initial_display_options.
        self._comp.data.info.attrs['cov_uuid'] = ''
        self._comp.data.commit()

        # Send back ActionRequest to Query for the new coverage's UUID and update id files.
        return messages, [self._comp.get_display_refresh_action()]
