"""Migrates SMS projects from older versions to the current interface."""

# 1. Standard Python modules
import datetime
import math
import os
import traceback
import uuid

# 2. Third party modules
from harmonica.tidal_constituents import Constituents
import numpy as np
import xarray as xr

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.components.bases.migrate_base import MigrateBase
from xms.components.display.display_options_io import write_display_option_ids
from xms.components.runners import migrate_runner as mgrun
from xms.data_objects.parameters import Component, Coverage, FilterLocation, Simulation
from xms.tides.components.tidal_component import TidalComponent
from xms.tides.data.tidal_data import ADCIRC_INDEX, LEPROVOST_INDEX

# 4. Local modules
from xms.cmsflow.components.bc_component import BCComponent
from xms.cmsflow.components.id_files import (
    BC_INITIAL_ATT_ID_FILE, BC_INITIAL_COMP_ID_FILE, SAVE_INITIAL_ATT_ID_FILE, SAVE_INITIAL_COMP_ID_FILE
)
from xms.cmsflow.components.save_points_component import SavePointsComponent
from xms.cmsflow.components.sim_component import SimComponent


class MigrateCms(MigrateBase):
    """Convert old SMS projects to component-based interface."""
    _1_VALUE = 0
    _KEYWORD = 0
    _ROW = 1
    _VALUE = 2

    def __init__(self):
        """Constructs a class for handling the migration of older SMS projects."""
        super().__init__()
        self._errors = []
        self._xms_data = None  # Old data to migrate. Get from Query or pass in for testing.
        self._old_takes = None  # passed in to first pass of xmscomponents migrate process
        self._old_items = None  # passed in to first pass of xmscomponents migrate process
        self._widgets = None  # passed in to first pass of xmscomponents migrate process
        self._default_datetime = datetime.datetime.now().strftime('%Y-%m-%dT%H:%M:%S')

        # Stuff we want to send back to SMS
        self._delete_uuids = []  # Things we want to delete: [uuid]
        self._new_sims = []  # [(do_simulation, do_component)]
        self._new_covs = {}  # Old cov UUID: (cov data_object, covtype, comp data_object)
        self._take_uuids = []  # [(taken_uuid, taker_uuid)]
        self._dset_paths = {}  # Geometry UUID: {dataset path: dataset UUID}
        self._new_tidal_comps = []  # [(tidal simulation data_object, tidal_component data_object)]
        self._cov_to_tidal_uuid = {}  # Old cov UUID: new tidal UUID

    def map_to_typename(
        self, uuid_map, widget_map, take_map, main_file_map, hidden_component_map, material_atts, material_polys, query
    ):
        """This is the first pass of a two-pass system.

        This function performs all the data object migrations necessary for use between versions of model interfaces.

        Args:
            uuid_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's to object attributes.
                This is a collection of things you want to find the conversions for.
            widget_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a larger dictionary mapping uuid's to
                their conversions.
            take_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's of taking
                objects to a list of uuids of objects taken by the taker.
            main_file_map (:obj:`dict` of :obj:`uuid` to str): This is a map from uuid's to main
                files of components.
            hidden_component_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's of owning
                objects to a list of uuids of hidden components owned by the object.
            material_atts (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is tuple of display options (name, r, g, b, alpha, texture)
            material_polys (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is a set of polygon ids with that material assignment.
            query (:obj:`data_objects.parameters.Query`): This is used to communicate with XMS.
                Do not call the 'send' method on this instance as 'send' will be called later.

        Returns:
            (:obj:`dict` of :obj:`uuid` to :obj:): This is the map of delete and replacement requests.
        """
        if not query:  # Testing. Initialize XMS data.
            self._xms_data = mgrun.xms_migration_data

        self._old_items = uuid_map
        self._old_takes = take_map
        self._widgets = widget_map
        if query:
            self._get_xms_data(query)

        try:
            # Migrate all CMS-Flow coverages
            replace_map = self._migrate_coverages()

            # Migrate all CMS-Flow simulations
            replace_map.update(self._migrate_sims())
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not migrate CMS-Flow project:\n"{traceback_msg}'))
            return {}

        return replace_map

    def send_replace_map(self, replace_map, query):
        """This is the second pass of a two-pass system.

        Args:
            replace_map (:obj:`dict` of :obj:`uuid` to :obj:): This is the map of delete and replace requests.
                This can be generated by map_to_typename().
            query (:obj:`data_objects.parameters.Query`): This is used to communicate with SMS.

        """
        if not query:
            return  # testing

        # Add delete requests to the Query
        for delete_uuid in self._delete_uuids:
            query.delete_item(delete_uuid)

        # Add new coverages to the Query
        for _, new_cov in self._new_covs.items():
            # new_cov = (data_objects.parameters.Coverage, covtype, dat_objects.parameters.Component)
            cov_type = new_cov[1]
            cov_dump = new_cov[0]
            components = [new_cov[2]] if new_cov[2] else None
            query.add_coverage(cov_dump, model_name='CMS-Flow', coverage_type=cov_type, components=components)

        # Add new tidal constituents components to the Query
        for new_comp in self._new_tidal_comps:
            query.add_simulation(new_comp[0], components=[new_comp[1]])

        # Add new simulations to the Query
        for new_sim in self._new_sims:
            query.add_simulation(new_sim[0], components=[new_sim[1]])

        # Link up takes to the new simulations
        for take_pair in self._take_uuids:
            query.link_item(taker_uuid=take_pair[1], taken_uuid=take_pair[0])

    def get_messages_and_actions(self):
        """Called at the end of migration.

        This is for when a message needs to be given to the user about some change due to migration.
        Also, an ActionRequest can be given if there is ambiguity in the migration.

        Returns:
            (:obj:`tuple` of :obj:`list` of :obj:`tuple` of :obj:`str`, :obj:`list` of :obj:`xmsapi.dmi.ActionRequest`):
                messages, action_requests - Where messages is a list of tuples with the first element of the tuple being
                the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                action_requests is a list of actions for XMS to perform.

        """
        if self._errors:
            messages = [('ERROR', 'Error(s) encountered during migration of CMS-Flow project.')]
            messages.extend(self._errors)
        else:
            messages = [('INFO', 'Successfully migrated CMS-Flow project.')]

        return messages, []

    def _get_xms_data(self, query):
        """Get the XMS temp component directory for creating new components.

        self._xms_data = {
            'comp_dir': ''  # XMS temp component directory
            'cov_dumps': []  # Old coverages to migrate [(dump object, coveragetype)]
            'projection': data_objects.parameters.Projection  # Projection geometric items exported in (should be same)
            'proj_dir': str  # Path to the project's folder
        }

        Args:
            query (xms.dmi.Query): Object for communicating with SMS.

        """
        if self._xms_data:  # Initial XMS data provided (probably testing)
            return

        self._xms_data = {
            'comp_dir': '',
            'cov_dumps': [],
            'projection': None,
            'proj_dir': '',
        }
        try:
            self._xms_data['proj_dir'] = os.path.dirname(mgrun.project_db_file)
            # Get the SMS temp components directory
            self._xms_data['comp_dir'] = os.path.join(query.xms_temp_directory, 'Components')

            self._get_coverages_to_migrate(query)
            self._get_dataset_paths(query)
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not retrieve SMS data required for migration:\n"{traceback_msg}'))

    def _get_coverages_to_migrate(self, query):
        """Get dumps of all the coverages we need to migrate.

        Args:
            query (xms.dmi.Query): Object for communicating with SMS.

        """
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'CMS-Flow' and info[2] == 'Coverage':
                cov_dump = query.item_with_uuid(item_uuid)
                if not cov_dump:
                    continue
                self._xms_data['cov_dumps'].append((cov_dump, info[1]))
                self._xms_data['projection'] = cov_dump.projection

    def _get_dataset_paths(self, query):
        """Gets the paths of the datasets for taken geometries.

        Args:
            query (xms.dmi.Query): Object for communicating with SMS.
        """
        uuids = []
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == '' and info[2] == 'Coverage':
                uuids.append(item_uuid)
        self._get_dataset_path_recursive(uuids, '', query.project_tree, '')

    def _get_dataset_path_recursive(self, uuids, build_uuid, item, path):
        """Recursive function for getting the dataset paths and uuid. Stores the path and uuid in self._dset_paths.

        Args:
            uuids (list): UUID strings of geometries taken by any simulation.
            build_uuid (str): UUID of the current geometry.
            item (xmsguipy.tree.TreeNode): Current item in the project explorer.
            path (str): Path from the geometry in the project explorer.
        """
        geom_uuid = ''
        if build_uuid:
            if not path:
                path = item.name
            else:
                path = path + '\\' + item.name
            if type(item.data) is xmd.DatasetItem:
                if build_uuid not in self._dset_paths:
                    self._dset_paths[build_uuid] = {}
                self._dset_paths[build_uuid][path] = item.uuid
            geom_uuid = build_uuid
        elif item.uuid in uuids:
            geom_uuid = item.uuid
        for child in item.children:
            if child.is_ptr_item:
                continue
            self._get_dataset_path_recursive(uuids, geom_uuid, child, path)

    def _migrate_coverages(self):
        """Migrate all CMS-Flow coverages.

        Returns:
            dict: Mapping of old simulation to list containing the UUID of the new simulation that replaces it.
                Needed for second pass of migration.

        """
        replace_map = {}
        for old_cov in self._xms_data['cov_dumps']:
            try:
                cov_dump = old_cov[0]
                type_name = old_cov[1]
                new_uuids = []
                old_uuid = cov_dump.uuid
                if old_uuid in self._widgets:
                    if type_name == 'Boundary Conditions':
                        arc_widgets = self._widgets[old_uuid]['Arc']
                        new_comp = self._migrate_bc_coverage(arc_widgets, old_uuid)
                        new_cov = self._get_do_coverage(cov_dump.name, arcs=cov_dump.arcs)
                        new_uuid = new_cov.uuid
                        self._new_covs[old_uuid] = (new_cov, 'Boundary Conditions', new_comp)
                        new_uuids.append(new_uuid)
                    elif type_name == 'Save Points':
                        cov_widgets = self._widgets[old_uuid]['Coverage'][-1]
                        point_widgets = self._widgets[old_uuid]['Point']
                        new_comp = self._migrate_save_points_coverage(cov_widgets, point_widgets)
                        new_cov = self._get_do_coverage(
                            cov_dump.name, pts=cov_dump.get_points(FilterLocation.PT_LOC_ALL)
                        )
                        new_uuid = new_cov.uuid
                        self._new_covs[old_uuid] = (new_cov, 'Save Points', new_comp)
                        new_uuids.append(new_uuid)
                self._delete_uuids.append(old_uuid)
                replace_map[old_uuid] = new_uuids
            except Exception as ex:
                traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                self._errors.append(('ERROR', f'Could not migrate CMS-Flow coverage:\n"{traceback_msg}'))
        return replace_map

    def _migrate_sims(self):
        """Migrate simulations to current component based version.

        Returns:
            dict: Mapping of old simulation to list containing the UUID of the new simulation that replaces it.
                Needed for second pass of migration.

        """
        replace_map = {}
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'CMS-Flow' and info[2] == 'Simulation':
                try:
                    # Create a new simulation and its hidden component
                    new_sim_uuid = str(uuid.uuid4())
                    new_sim = Simulation(model='CMS-Flow', sim_uuid=new_sim_uuid, name=info[1])
                    replace_map[item_uuid] = [new_sim_uuid]
                    sim_comp, sim_data, main_file = self._migrate_model_control(item_uuid)

                    # Add the new simulation and its component to the Query.
                    self._new_sims.append((new_sim, sim_comp))

                    # Search for taken items under the old simulation.
                    if item_uuid in self._old_takes:
                        for take_uuid in self._old_takes[item_uuid]:
                            if take_uuid not in self._old_items:
                                continue

                            type_name = self._old_items[take_uuid][1]
                            if type_name == 'QUADTREE':
                                # No migration needed for the quadtree, just relink.
                                sim_data.info.attrs['domain_uuid'] = take_uuid
                                new_take_uuid = take_uuid
                            elif type_name == 'MAP':
                                # This is probably an activity coverage.
                                new_take_uuid = take_uuid
                            else:
                                if take_uuid not in self._new_covs:
                                    continue

                                if type_name == 'Save Points':  # 0 or 1 per simulation
                                    pass  # TODO: figure out what goes here
                                elif type_name == 'Boundary Conditions':  # 0 or 1 per simulation
                                    if take_uuid in self._cov_to_tidal_uuid:
                                        new_tidal_take_uuid = self._cov_to_tidal_uuid[take_uuid]
                                        self._take_uuids.append((new_tidal_take_uuid, new_sim_uuid))

                                # Replace old take coverage with migrated one.
                                new_cov = self._new_covs[take_uuid][0]
                                new_take_uuid = new_cov.uuid

                            self._take_uuids.append((new_take_uuid, new_sim_uuid))
                    sim_data.commit()
                except Exception as ex:
                    traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                    self._errors.append(('ERROR', f'Could not migrate CMS-Flow simulation:\n"{traceback_msg}'))

        return replace_map

    def _migrate_model_control(self, old_sim_uuid):
        """Migrate the simulation Model Control widgets and Dredge widgets.

        Args:
            old_sim_uuid (str): UUID of the old simulation

        Returns:
            data_objects.parameters.Component, ModelControl, str: The new simulation's hidden component, its data,
                and path to the main file where data should be written.
        """
        # Create a folder and UUID for the new simulation component.
        comp_uuid = self._get_next_sim_comp_uuid()
        sim_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(sim_comp_dir, exist_ok=True)

        # Create the component data_object to send back to SMS.
        sim_main_file = os.path.join(str(sim_comp_dir), 'simulation_comp.nc')
        sim_comp = Component(
            comp_uuid=comp_uuid, model_name='CMS-Flow', unique_name='Simulation_Component', main_file=sim_main_file
        )

        # Fill component data from widget values in the old database.
        sim_py_comp = SimComponent(sim_main_file)  # Initialize some default data
        data = sim_py_comp.data

        try:
            if old_sim_uuid in self._widgets:
                sim_widgets = self._widgets[old_sim_uuid]['Simulation'][-1]
                # Add newer options if needed
                sed_used = sim_widgets.get('togSedTransport') == 1
                if sed_used and sim_widgets.get('dsetMultiD90') is None:
                    sim_widgets['dsetMultiD90'] = sim_widgets['dsetMultiD50']
                    sim_widgets['dsetMultiD35'] = sim_widgets['dsetMultiD50']
                # New option needs to have a value for later.
                if sim_widgets.get('togFractionBedload') is None:
                    sim_widgets['togFractionBedload'] = sim_widgets['togFractionSuspended']

                # General options
                self._set_general_values(data, sim_widgets)
                self._set_flow_values(data, sim_widgets)
                self._set_sediment_values(data, sim_widgets)
                self._set_salinity_values(data, sim_widgets)
                self._set_wave_values(data, sim_widgets)
                self._set_wind_values(data, sim_widgets)
                self._set_output_values(data, sim_widgets)
                self._set_dredge_values(data, sim_widgets)
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not migrate CMS-Flow model control:\n"{traceback_msg}'))

        return sim_comp, data, sim_main_file

    def _get_next_sim_comp_uuid(self):
        """Get a randomly generated UUID for a new sim component or hard-coded one for testing."""
        if 'new_sim_comp_uuids' in self._xms_data and self._xms_data['new_sim_comp_uuids']:
            return self._xms_data['new_sim_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _get_next_cov_comp_uuid(self):
        """Get a randomly generated UUID for a new coverage component or hard-coded one for testing."""
        if 'new_cov_comp_uuids' in self._xms_data and self._xms_data['new_cov_comp_uuids']:
            return self._xms_data['new_cov_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _get_next_tidal_comp_uuid(self):
        """Get a randomly generated UUID for a new coverage component or hard-coded one for testing."""
        if 'new_tidal_comp_uuids' in self._xms_data and self._xms_data['new_tidal_comp_uuids']:
            return self._xms_data['new_tidal_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _set_value_and_unit(self, old_widget_values, new_values, new_value_str, new_unit_str):
        """Sets the value and units from a widget. This assumes the widget was not in a table.

        Args:
            old_widget_values (list of tuple): The old widget values from the project DB for a value and units widget.
            new_values (xr.Dataset): The dataset where the value will be stored.
            new_value_str (str): The string for the new value location.
            new_unit_str (str): The string for the new units location.
        """
        value_idx = 0
        unit_idx = 1
        if old_widget_values[value_idx][self._KEYWORD] == 'units':
            value_idx = 1
            unit_idx = 0
        new_values.attrs[new_value_str] = old_widget_values[value_idx][self._VALUE]
        old_unit = old_widget_values[unit_idx][self._VALUE]
        if old_unit == 'meters':
            old_unit = 'm'
        elif old_unit == 'sec':
            old_unit = 'seconds'
        elif old_unit == 'min':
            old_unit = 'minutes'
        elif old_unit == 'kg/m³':
            old_unit = 'kg/m^3'
        elif old_unit == 'gr/cm³':
            old_unit = 'gr/cm^3'
        elif old_unit == 'lb/ft³':
            old_unit = 'lb/ft^3'
        new_values.attrs[new_unit_str] = old_unit

    def _set_table_value_and_unit(self, old_widget_values):
        """Sets the value and units from a widget in a table.

        Args:
            old_widget_values (list of tuple): The old widget values from the project DB for a value and units widget.

        Returns:
            Returns a tuple of lists of values. The first list in the tuple is the value, second is units.
            Lists are in row id order.
        """
        values = []
        units = []
        old_widget_values.sort(key=lambda x: x[self._ROW])  # sort by row
        for widget_value in old_widget_values:
            if widget_value[self._KEYWORD] == 'units':
                units.append(widget_value[self._VALUE])
            else:
                values.append(widget_value[self._VALUE])
        if len(values) != len(units):
            return [], []
        return values, units

    def _dataset_widget_to_uuid(self, old_widget_values):
        """Converts the path based dataset widget storage to a UUID.

        Args:
            old_widget_values (list of tuple): The old widget values from the project DB for a dataset selector widget.

        Returns:
            A UUID string for the dataset.
        """
        # The widget stores a UUID for the geometry and a path to the dataset relative to the geometry.
        geom_uuid_idx = 0
        path_idx = 1
        if old_widget_values[geom_uuid_idx][self._KEYWORD] == 'sms_path':
            geom_uuid_idx = 1
            path_idx = 0
        geom_uuid = old_widget_values[geom_uuid_idx][self._VALUE]
        path = old_widget_values[path_idx][self._VALUE]
        if geom_uuid not in self._dset_paths or path not in self._dset_paths[geom_uuid]:
            return ''
        return self._dset_paths[geom_uuid][path]

    def _table_dataset_widget_to_uuids(self, old_widget_values):
        """Converts the path based dataset widget storage to UUIDs.

        Args:
            old_widget_values (list of tuple): The old widget values from the project DB for a dataset selector widget.

        Returns:
            A list of UUID strings for the datasets.
        """
        # The widget stores a UUID for the geometry and a path to the dataset relative to the geometry.
        geom_uuids = {}
        paths = {}
        num_rows = 0
        for widget_value in old_widget_values:
            row_id = widget_value[self._ROW]
            num_rows = max(row_id + 1, num_rows)
            if widget_value[self._KEYWORD] == 'sms_path':
                paths[row_id] = widget_value[self._VALUE]
            else:
                geom_uuids[row_id] = widget_value[self._VALUE]

        dsets = []
        for row in range(num_rows):
            geom_uuid = geom_uuids.get(row, '')
            path = paths.get(row, '')
            dset_uuid = self._dset_paths.get(geom_uuid, {}).get(path, '')
            dsets.append(dset_uuid)
        return dsets

    def _set_general_values(self, data, sim_widgets):
        """Sets the values in the general tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        widget_time_value = sim_widgets['dateStart'][self._1_VALUE][self._VALUE]
        data.general.attrs['DATE_START'] = self._create_date_time_string(widget_time_value)
        self._set_value_and_unit(sim_widgets['ccSimDuration'], data.general, 'SIM_DURATION_VALUE', 'SIM_DURATION_UNITS')
        self._set_value_and_unit(
            sim_widgets['ccRampDuration'], data.general, 'RAMP_DURATION_VALUE', 'RAMP_DURATION_UNITS'
        )
        data.general.attrs['SKEW_CORRECT'] = sim_widgets['togSkewCorrect'][self._1_VALUE][self._VALUE]
        data.general.attrs['USE_INIT_CONDITIONS_FILE'] =\
            sim_widgets['togInitialConditionsFile'][self._1_VALUE][self._VALUE]
        data.general.attrs['INIT_CONDITIONS_FILE'] = sim_widgets['fileConditions'][self._1_VALUE][self._VALUE]
        data.general.attrs['USE_HOT_START_OUTPUT_FILE'] =\
            sim_widgets['togHotStartOutputFile'][self._1_VALUE][self._VALUE]
        if 'ccTimeToWriteOut' in sim_widgets:
            self._set_value_and_unit(
                sim_widgets['ccTimeToWriteOut'], data.general, 'HOT_WRITE_OUT_DURATION_VALUE',
                'HOT_WRITE_OUT_DURATION_UNITS'
            )
        elif 'edtTimeToWriteOut' in sim_widgets:
            # from before 13.0, units is hours
            data.general.attrs['HOT_WRITE_OUT_DURATION_VALUE'] =\
                sim_widgets['edtTimeToWriteOut'][self._1_VALUE][self._VALUE]
            data.general.attrs['HOT_WRITE_OUT_DURATION_UNITS'] = 'hours'
        data.general.attrs['RECURRING_HOT_START_FILE'] =\
            sim_widgets['togAutoRecurringHotStartFile'][self._1_VALUE][self._VALUE]
        if 'ccAutoHotStartInterval' in sim_widgets:
            self._set_value_and_unit(
                sim_widgets['ccAutoHotStartInterval'], data.general, 'AUTO_HOT_DURATION_VALUE',
                'AUTO_HOT_DURATION_UNITS'
            )
        elif 'edtInterval' in sim_widgets:
            # from before 13.0, units is hours
            data.general.attrs['AUTO_HOT_DURATION_VALUE'] = \
                sim_widgets['edtInterval'][self._1_VALUE][self._VALUE]
            data.general.attrs['AUTO_HOT_DURATION_UNITS'] = 'hours'
        data.general.attrs['SOLUTION_SCHEME'] = sim_widgets['cbxSolutionScheme'][self._1_VALUE][self._VALUE]
        data.general.attrs['MATRIX_SOLVER'] = sim_widgets['cbxMatrixSolver'][self._1_VALUE][self._VALUE]
        data.general.attrs['NUM_THREADS'] = sim_widgets['edtThreads'][self._1_VALUE][self._VALUE]

    def _set_flow_values(self, data, sim_widgets):
        """Sets the values in the flow tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        if 'ccHydroTs' in sim_widgets:
            self._set_value_and_unit(
                sim_widgets['ccHydroTs'], data.flow, 'HYDRO_TIME_STEP_VALUE', 'HYDRO_TIME_STEP_UNITS'
            )
        elif sim_widgets['cbxSolutionScheme'][self._1_VALUE][self._VALUE] == 'Implicit':
            self._set_value_and_unit(
                sim_widgets['ccHydroTsImplicit'], data.flow, 'HYDRO_TIME_STEP_VALUE', 'HYDRO_TIME_STEP_UNITS'
            )
        else:
            self._set_value_and_unit(
                sim_widgets['ccHydroTsExplicit'], data.flow, 'HYDRO_TIME_STEP_VALUE', 'HYDRO_TIME_STEP_UNITS'
            )
        data.flow.attrs['WETTING_DEPTH'] = sim_widgets['edtWettingDepth'][self._1_VALUE][self._VALUE]
        data.flow.attrs['WAVE_FLUXES'] = sim_widgets['togWaveFluxes'][self._1_VALUE][self._VALUE]
        data.flow.attrs['ROLLER_FLUXES'] = sim_widgets['togRollerFluxes'][self._1_VALUE][self._VALUE]
        data.flow.attrs['LATITUDE_CORIOLIS'] = sim_widgets['cbxLatitudeCoriolis'][self._1_VALUE][self._VALUE]
        data.flow.attrs['DEGREES'] = sim_widgets['edtDegrees'][self._1_VALUE][self._VALUE]
        data.flow.attrs['TURBULENCE_MODEL'] = sim_widgets['cbxTurbModel'][self._1_VALUE][self._VALUE]
        if 'togTurbParams' in sim_widgets:
            data.flow.attrs['TURBULENCE_PARAMETERS'] = sim_widgets['togTurbParams'][self._1_VALUE][self._VALUE]
        data.flow.attrs['BASE_VALUE'] = sim_widgets['edtViscConst'][self._1_VALUE][self._VALUE]
        data.flow.attrs['CURRENT_BOTTOM_COEFFICIENT'] = sim_widgets['edtViscBottom'][self._1_VALUE][self._VALUE]
        data.flow.attrs['CURRENT_HORIZONTAL_COEFFICIENT'] = sim_widgets['edtViscHoriz'][self._1_VALUE][self._VALUE]
        data.flow.attrs['WAVE_BOTTOM_COEFFICIENT'] = sim_widgets['edtViscWave'][self._1_VALUE][self._VALUE]
        data.flow.attrs['WAVE_BREAKING_COEFFICIENT'] = sim_widgets['edtViscBreaking'][self._1_VALUE][self._VALUE]
        data.flow.attrs['WAVE_CURRENT_BOTTOM_FRIC_COEFFICIENT'] =\
            sim_widgets['togWaveBottomFric'][self._1_VALUE][self._VALUE]
        data.flow.attrs['QUAD_WAVE_BOTTOM_COEFFICIENT'] = sim_widgets['edtWaveBottomCoef'][self._1_VALUE][self._VALUE]
        data.flow.attrs['BED_SLOPE_FRIC_COEFFICIENT'] = sim_widgets['togBedSlopeFricCoef'][self._1_VALUE][self._VALUE]
        data.flow.attrs['WALL_FRICTION'] = sim_widgets['togWallFriction'][self._1_VALUE][self._VALUE]
        bottom_rough_source = sim_widgets['cbxBottomRoughDataset'][self._1_VALUE][self._VALUE]
        data.flow.attrs['BOTTOM_ROUGHNESS'] = bottom_rough_source
        if bottom_rough_source == 'Mannings N':
            data.flow.attrs['BOTTOM_ROUGHNESS_DSET'] = self._dataset_widget_to_uuid(sim_widgets['dsetManningsN'])
        elif bottom_rough_source == 'Bottom friction coefficient':
            data.flow.attrs['BOTTOM_ROUGHNESS_DSET'] = self._dataset_widget_to_uuid(sim_widgets['dsetBottomFricCoef'])
        elif bottom_rough_source == 'Roughness height (m)':
            data.flow.attrs['BOTTOM_ROUGHNESS_DSET'] = self._dataset_widget_to_uuid(sim_widgets['dsetRoughnessHeight'])

    def _set_sediment_values(self, data, sim_widgets):
        """Sets the values in the sediment tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        data.sediment.attrs['CALCULATE_SEDIMENT'] = sim_widgets['togSedTransport'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccTransportTS'], data.sediment, 'TRANSPORT_TIME_VALUE', 'TRANSPORT_TIME_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccMorphologicTS'], data.sediment, 'MORPHOLOGIC_TIME_VALUE', 'MORPHOLOGIC_TIME_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccMortphologyST'], data.sediment, 'MORPHOLOGY_TIME_VALUE', 'MORPHOLOGY_TIME_UNITS'
        )
        formulation_units = sim_widgets['cbxFormaultionUnits'][self._1_VALUE][self._VALUE].capitalize()
        if formulation_units == 'Equilibrium bed load plus nonequilibrium susp load':
            formulation_units = 'Equilibrium bed load plus nonequilibrium susp. load'
        data.sediment.attrs['FORMULATION_UNITS'] = formulation_units
        data.sediment.attrs['TRANSPORT_FORMULA'] = sim_widgets['cbxTransportFormula'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['CONCENTRATION_PROFILE'] = sim_widgets['cbxConcentrationPro'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['WATANABE_RATE'] = sim_widgets['edtWatanabeCoeff'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccDensity'], data.sediment, 'SEDIMENT_DENSITY_VALUE', 'SEDIMENT_DENSITY_UNITS'
        )
        data.sediment.attrs['SEDIMENT_POROSITY'] = sim_widgets['edtSedPorosity'][self._1_VALUE][self._VALUE]
        if 'ccConstantGrainSize' in sim_widgets:
            self._set_value_and_unit(
                sim_widgets['ccConstantGrainSize'], data.sediment, 'GRAIN_SIZE_VALUE', 'GRAIN_SIZE_UNITS'
            )
        data.sediment.attrs['BED_LOAD_SCALING'] = sim_widgets['edtBLSCF'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['SUSPENDED_LOAD_SCALING'] = sim_widgets['edtSLSF'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['MORPHOLOGIC_ACCELERATION'] = sim_widgets['edtMAF'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['BED_SLOPE_DIFFUSION'] = sim_widgets['edtBC'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['HIDING_AND_EXPOSURE'] = sim_widgets['edtHEC'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['TOTAL_ADAPTATION_METHOD'] = sim_widgets['cbxAdaptationMethod'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccAdaptationLenTotal'], data.sediment, 'TOTAL_ADAPTATION_LENGTH_VALUE',
            'TOTAL_ADAPTATION_LENGTH_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccAdaptationTimeTotal'], data.sediment, 'TOTAL_ADAPTATION_TIME_VALUE',
            'TOTAL_ADAPTATION_TIME_UNITS'
        )
        data.sediment.attrs['BED_ADAPTATION_METHOD'] = sim_widgets['cbxBLAM'][self._1_VALUE][self._VALUE].capitalize()
        self._set_value_and_unit(
            sim_widgets['ccAdapationLengthBed'], data.sediment, 'BED_ADAPTATION_LENGTH_VALUE',
            'BED_ADAPTATION_LENGTH_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccAdaptationTimeBed'], data.sediment, 'BED_ADAPTATION_TIME_VALUE', 'BED_ADAPTATION_TIME_UNITS'
        )
        data.sediment.attrs['BED_ADAPTATION_DEPTH'] = sim_widgets['edtADFB'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['SUSPENDED_ADAPTATION_METHOD'] = sim_widgets['cbxSLAM'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccAdaptationLenSusp'], data.sediment, 'SUSPENDED_ADAPTATION_LENGTH_VALUE',
            'SUSPENDED_ADAPTATION_LENGTH_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccAdaptationTimeSusp'], data.sediment, 'SUSPENDED_ADAPTATION_TIME_VALUE',
            'SUSPENDED_ADAPTATION_TIME_UNITS'
        )
        data.sediment.attrs['SUSPENDED_ADAPTATION_COEFFICIENT'] = sim_widgets['edtACS'][self._1_VALUE][self._VALUE]
        if 'togAdvSizeClass' in sim_widgets:
            data.sediment.attrs['USE_ADVANCED_SIZE_CLASSES'] =\
                sim_widgets['togAdvSizeClass'][self._1_VALUE][self._VALUE]
        if 'togSimpleMultiGrainSize' in sim_widgets:
            data.sediment.attrs['ENABLE_SIMPLIFIED_MULTI_GRAIN_SIZE'] =\
                sim_widgets['togSimpleMultiGrainSize'][self._1_VALUE][self._VALUE]
            data.sediment.attrs['MULTIPLE_GRAIN_SIZES'] =\
                sim_widgets['cbxSimpleMultiGrainSizeType'][self._1_VALUE][self._VALUE]
            data.sediment.attrs['SIMPLE_MULTI_SIZE'] = sim_widgets['edtSimpleMultiSizeOnly'][self._1_VALUE][self._VALUE]
            data.sediment.attrs['SEDIMENT_STANDARD_DEVIATION'] =\
                sim_widgets['edtSimpleStdDev'][self._1_VALUE][self._VALUE]
            data.sediment.attrs['BED_COMPOSITION_INPUT'] =\
                sim_widgets['cbxSimpleBedCompInput'][self._1_VALUE][self._VALUE]

            if 'dsetMultiD35' not in sim_widgets:
                # This default_val is the same as the default value for a similar object.
                default_val = [('geom_guid', -1, 'cccccccc-cccc-cccc-cccc-cccccccccccc'), ('sms_path', -1, '')]
                sim_widgets['dsetMultiD35'] = default_val
                sim_widgets['dsetMultiD90'] = default_val
            data.sediment.attrs['MULTI_D16'] = self._dataset_widget_to_uuid(sim_widgets['dsetMultiD16'])
            data.sediment.attrs['MULTI_D50'] = self._dataset_widget_to_uuid(sim_widgets['dsetMultiD50'])
            data.sediment.attrs['MULTI_D84'] = self._dataset_widget_to_uuid(sim_widgets['dsetMultiD84'])
            data.sediment.attrs['MULTI_D35'] = self._dataset_widget_to_uuid(sim_widgets['dsetMultiD35'])
            data.sediment.attrs['MULTI_D90'] = self._dataset_widget_to_uuid(sim_widgets['dsetMultiD90'])

            data.sediment.attrs['NUMBER_BED_LAYERS'] =\
                int(sim_widgets['edtSimpleBedLayerNumber'][self._1_VALUE][self._VALUE])
            data.sediment.attrs['THICKNESS_FOR_MIXING'] =\
                sim_widgets['edtSimpleMixConstantThickness'][self._1_VALUE][self._VALUE]
            data.sediment.attrs['THICKNESS_FOR_BED'] =\
                sim_widgets['edtSimpleBedLayerConstantThickness'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['MAX_NUMBER_BED_LAYERS'] = int(sim_widgets['edtBLMN'][self._1_VALUE][self._VALUE])
        self._set_value_and_unit(
            sim_widgets['ccMNBL'], data.sediment, 'MIN_BED_LAYER_THICKNESS_VALUE', 'MIN_BED_LAYER_THICKNESS_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets['ccMXBL'], data.sediment, 'MAX_BED_LAYER_THICKNESS_VALUE', 'MAX_BED_LAYER_THICKNESS_UNITS'
        )
        data.sediment.attrs['MIXING_LAYER_THICKNESS'] = sim_widgets['cbxMixLayForm'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccMXLCT'], data.sediment, 'CONSTANT_MIXING_LAYER_THICKNESS_VALUE',
            'CONSTANT_MIXING_LAYER_THICKNESS_UNITS'
        )
        data.sediment.attrs['CALCULATE_AVALANCHING'] = sim_widgets['togAvalanching'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['CRITICAL_BED_SLOPE'] = sim_widgets['edtACBS'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['MAX_NUMBER_ITERATIONS'] = sim_widgets['edtAvMxIt'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['USE_HARD_BOTTOM'] = sim_widgets['togHardbottom'][self._1_VALUE][self._VALUE]
        data.sediment.attrs['HARD_BOTTOM'] = self._dataset_widget_to_uuid(sim_widgets['Hard_Bottom'])

        if 'edtSimpleGrainSize' in sim_widgets and sim_widgets['edtSimpleGrainSize']:
            grain_sizes = [row[self._VALUE] for row in sim_widgets['edtSimpleGrainSize']]
            grain_sizes_table = {'grain_size': xr.DataArray(data=np.array(grain_sizes, dtype=float))}
            data.simple_grain_sizes_table = xr.Dataset(data_vars=grain_sizes_table)
        self._set_size_class_values(data, sim_widgets)
        self._set_bed_layer_values(data, sim_widgets)

    def _set_size_class_values(self, data, sim_widgets):
        """Sets the values in the sediment size classes of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        have_simple = 'ccColumnDiameter' in sim_widgets and sim_widgets['ccColumnDiameter']
        have_advanced = 'tblColCbxFallVelocity' in sim_widgets and sim_widgets['tblColCbxFallVelocity']
        if 'togAdvSizeClass' in sim_widgets:
            use_advanced = sim_widgets['togAdvSizeClass'][self._1_VALUE][self._VALUE] != 0
            advanced_col_diameter = 'ccColumnDiameterFull'
        else:
            use_advanced = True
            advanced_col_diameter = 'ccColumnDiameter'
        sizes_used = True
        diameter_value = []
        diameter_units = []
        fall_velocity_method = []
        fall_velocity_value = []
        fall_velocity_units = []
        corey_shape_factor = []
        critical_shear_method = []
        critical_shear_stress = []
        if use_advanced and have_advanced:
            diameter_value, diameter_units = self._set_table_value_and_unit(sim_widgets[advanced_col_diameter])
            fall_velocity_method = [row[self._VALUE] for row in sim_widgets['tblColCbxFallVelocity']]
            fall_velocity_value, fall_velocity_units =\
                self._set_table_value_and_unit(sim_widgets['ccColumnFallVelocity'])
            if 'edtCSF' in sim_widgets:
                corey_shape_factor = [row[self._VALUE] for row in sim_widgets['edtCSF']]
            else:
                corey_shape_factor = [0.7 for row in sim_widgets['tblColCbxFallVelocity']]
            critical_shear_method = [row[self._VALUE] for row in sim_widgets['tblColCbxCriticalShear']]
            critical_shear_stress = [row[self._VALUE] for row in sim_widgets['edtCriticalShear']]
        elif not use_advanced and have_simple:
            diameter_value, diameter_units = self._set_table_value_and_unit(sim_widgets['ccColumnDiameter'])
            num_values = len(diameter_value)
            fall_velocity_method = ['Soulsby (1997)' for _ in range(num_values)]
            fall_velocity_value = [0.0 for _ in range(num_values)]
            fall_velocity_units = ['m/s' for _ in range(num_values)]
            corey_shape_factor = [0.7 for _ in range(num_values)]
            critical_shear_method = ['Soulsby (1997)' for _ in range(num_values)]
            critical_shear_stress = [0.2 for _ in range(num_values)]
        else:
            sizes_used = False

        if sizes_used:
            advanced_sediment_diameters_table = {
                'diameter_value': xr.DataArray(data=np.array(diameter_value, dtype=float)),
                'diameter_units': xr.DataArray(data=np.array(diameter_units, dtype=object)),
                'fall_velocity_method': xr.DataArray(data=np.array(fall_velocity_method, dtype=object)),
                'fall_velocity_value': xr.DataArray(data=np.array(fall_velocity_value, dtype=float)),
                'fall_velocity_units': xr.DataArray(data=np.array(fall_velocity_units, dtype=object)),
                'corey_shape_factor': xr.DataArray(data=np.array(corey_shape_factor, dtype=float)),
                'critical_shear_method': xr.DataArray(data=np.array(critical_shear_method, dtype=object)),
                'critical_shear_stress': xr.DataArray(data=np.array(critical_shear_stress, dtype=float))
            }
            data.advanced_sediment_diameters_table = xr.Dataset(data_vars=advanced_sediment_diameters_table)

    def _set_bed_layer_values(self, data, sim_widgets):
        """Sets the values in the bed layers of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        if 'edtLayerID' not in sim_widgets:
            return
        layer_id = [row[self._VALUE] for row in sim_widgets['edtLayerID']]
        if 0 in layer_id:
            layer_id = [layer + 1 for layer in layer_id]
        if 'ThicknessDset' in sim_widgets:
            layer_thickness_type = ['Dataset' for _ in range(len(layer_id))]
            layer_thickness = self._table_dataset_widget_to_uuids(sim_widgets['ThicknessDset'])
            layer_thickness_const = [0.5 for _ in range(len(layer_id))]  # default value
        elif 'edtThickness' in sim_widgets:
            layer_thickness_type = ['Constant' for _ in range(len(layer_id))]
            layer_thickness = ['' for _ in range(len(layer_id))]  # default value
            layer_thickness_const = [row[self._VALUE] for row in sim_widgets['edtThickness']]
        else:
            layer_thickness_type = ['Constant' for _ in range(len(layer_id))]
            layer_thickness = ['' for _ in range(len(layer_id))]  # default value
            layer_thickness_const = [0.5 for _ in range(len(layer_id))]
        if len(layer_thickness_type) == 1:
            layer_thickness_type = ['Automatic']
        d05 = self._table_dataset_widget_to_uuids(sim_widgets['colD05'])
        d10 = self._table_dataset_widget_to_uuids(sim_widgets['colD10'])
        d16 = self._table_dataset_widget_to_uuids(sim_widgets['colD16'])
        d20 = self._table_dataset_widget_to_uuids(sim_widgets['colD20'])
        d30 = self._table_dataset_widget_to_uuids(sim_widgets['colD30'])
        d35 = self._table_dataset_widget_to_uuids(sim_widgets['colD35'])
        d50 = self._table_dataset_widget_to_uuids(sim_widgets['colD50'])
        d65 = self._table_dataset_widget_to_uuids(sim_widgets['colD65'])
        d84 = self._table_dataset_widget_to_uuids(sim_widgets['colD84'])
        d90 = self._table_dataset_widget_to_uuids(sim_widgets['colD90'])
        d95 = self._table_dataset_widget_to_uuids(sim_widgets['colD95'])
        bed_layer_table = {
            'layer_id': xr.DataArray(data=np.array(layer_id, dtype=int)),
            'layer_thickness_type': xr.DataArray(data=np.array(layer_thickness_type, dtype=object)),
            'layer_thickness': xr.DataArray(data=np.array(layer_thickness, dtype=object)),
            'layer_thickness_const': xr.DataArray(data=np.array(layer_thickness_const, dtype=float)),
            'd05': xr.DataArray(data=np.array(d05, dtype=object)),
            'd10': xr.DataArray(data=np.array(d10, dtype=object)),
            'd16': xr.DataArray(data=np.array(d16, dtype=object)),
            'd20': xr.DataArray(data=np.array(d20, dtype=object)),
            'd30': xr.DataArray(data=np.array(d30, dtype=object)),
            'd35': xr.DataArray(data=np.array(d35, dtype=object)),
            'd50': xr.DataArray(data=np.array(d50, dtype=object)),
            'd65': xr.DataArray(data=np.array(d65, dtype=object)),
            'd84': xr.DataArray(data=np.array(d84, dtype=object)),
            'd90': xr.DataArray(data=np.array(d90, dtype=object)),
            'd95': xr.DataArray(data=np.array(d95, dtype=object))
        }
        data.bed_layer_table = xr.Dataset(data_vars=bed_layer_table)

    def _set_salinity_values(self, data, sim_widgets):
        """Sets the values in the salinity/temperature tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        data.salinity.attrs['WATER_DENSITY'] = sim_widgets['edtWaterDensity'][self._1_VALUE][self._VALUE]
        data.salinity.attrs['WATER_TEMP'] = sim_widgets['edtWaterTemp'][self._1_VALUE][self._VALUE]
        data.salinity.attrs['CALCULATE_SALINITY'] = sim_widgets['togCalcSalinity'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccTransportRate'], data.salinity, 'SALINITY_TRANSPORT_RATE_VALUE',
            'SALINITY_TRANSPORT_RATE_UNITS'
        )
        data.salinity.attrs['SALINITY_CONCENTRATION'] =\
            sim_widgets['cbxSalinityConcentration'][self._1_VALUE][self._VALUE]
        data.salinity.attrs['GLOBAL_CONCENTRATION'] = sim_widgets['edtGlobalConcentration'][self._1_VALUE][self._VALUE]
        data.salinity.attrs['SALINITY_INITIAL_CONCENTRATION'] =\
            self._dataset_widget_to_uuid(sim_widgets['dsetSalinity'])
        if 'togCalcTemperature' in sim_widgets:
            data.salinity.attrs['CALCULATE_TEMPERATURE'] = sim_widgets['togCalcTemperature'][self._1_VALUE][self._VALUE]
        if 'ccTempTransportRate' in sim_widgets:
            self._set_value_and_unit(
                sim_widgets['ccTempTransportRate'], data.salinity, 'TEMPERATURE_TRANSPORT_RATE_VALUE',
                'TEMPERATURE_TRANSPORT_RATE_UNITS'
            )
        if 'cbxTemperature' in sim_widgets:
            data.salinity.attrs['INITIAL_TEMPERATURE_TYPE'] =\
                sim_widgets['cbxTemperature'][self._1_VALUE][self._VALUE].capitalize()
        if 'dsetTemperature' in sim_widgets:
            data.salinity.attrs['INITIAL_TEMPERATURE_DATASET'] =\
                self._dataset_widget_to_uuid(sim_widgets['dsetTemperature'])

        if 'edtTempParamTime' in sim_widgets and sim_widgets['edtTempParamTime']:
            time = [row[self._VALUE] for row in sim_widgets['edtTempParamTime']]
            air_temp = [row[self._VALUE] for row in sim_widgets['edtAirTemp']]
            dewpoint = [row[self._VALUE] for row in sim_widgets['edtDewPoint']]
            cloud_cover = [row[self._VALUE] for row in sim_widgets['edtCloudCover']]
            solar_radiation = [row[self._VALUE] for row in sim_widgets['edtSolarRad']]
            atmospheric_table = {
                'time': xr.DataArray(data=np.array(time, dtype=float)),
                'air_temp': xr.DataArray(data=np.array(air_temp, dtype=float)),
                'dewpoint': xr.DataArray(data=np.array(dewpoint, dtype=float)),
                'cloud_cover': xr.DataArray(data=np.array(cloud_cover, dtype=float)),
                'solar_radiation': xr.DataArray(data=np.array(solar_radiation, dtype=float))
            }
            data.atmospheric_table = xr.Dataset(data_vars=atmospheric_table)

    def _set_wave_values(self, data, sim_widgets):
        """Sets the values in the wave tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        data.wave.attrs['WAVE_INFO'] = sim_widgets['cbxWaveInfo'][self._1_VALUE][self._VALUE]
        data.wave.attrs['WAVE_HEIGHT'] = self._dataset_widget_to_uuid(sim_widgets['dsetWaveHeight'])
        data.wave.attrs['PEAK_PERIOD'] = self._dataset_widget_to_uuid(sim_widgets['dsetPeakPeriod'])
        data.wave.attrs['MEAN_WAVE_DIR'] = self._dataset_widget_to_uuid(sim_widgets['dsetMeanWaveDir'])
        data.wave.attrs['WAVE_BREAKING'] = self._dataset_widget_to_uuid(sim_widgets['dsetWaveBreaking'])
        data.wave.attrs['WAVE_RADIATION'] = self._dataset_widget_to_uuid(sim_widgets['dsetWaveStressGradients'])
        data.wave.attrs['FILE_WAVE_SIM'] = sim_widgets['fileWaveSim'][self._1_VALUE][self._VALUE]
        data.wave.attrs['STEERING_INTERVAL_CONST'] = sim_widgets['edtSteeringInterval'][self._1_VALUE][self._VALUE]
        data.wave.attrs['WAVE_WATER_PREDICTOR'] = sim_widgets['cbxWaveWaterPredictor'][self._1_VALUE][self._VALUE]
        data.wave.attrs['EXTRAPOLATION_DISTANCE'] = sim_widgets['togExtrapolationDistance'][self._1_VALUE][self._VALUE]
        data.wave.attrs['FLOW_TO_WAVE'] = sim_widgets['cbxFlowToWave'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['lenFlowToWave'], data.wave, 'FLOW_TO_WAVE_USER_VALUE', 'FLOW_TO_WAVE_USER_UNITS'
        )
        data.wave.attrs['WAVE_TO_FLOW'] = sim_widgets['cbxWaveToFlow'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['lenWaveToFlow'], data.wave, 'WAVE_TO_FLOW_USER_VALUE', 'WAVE_TO_FLOW_USER_UNITS'
        )

    def _set_wind_values(self, data, sim_widgets):
        """Sets the values in the wind tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        data.wind.attrs['WIND_TYPE'] = sim_widgets['cbxWindType'][self._1_VALUE][self._VALUE]
        data.wind.attrs['ANEMOMETER'] = sim_widgets['edtAnemometer'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_FILE_TYPE'] = sim_widgets['cbxWindFileType'][self._1_VALUE][self._VALUE]
        if sim_widgets['cbxWindFileType'][self._1_VALUE][self._VALUE] == 'Navy fleet numeric with pressure':
            data.wind.attrs['WIND_FILE'] = sim_widgets['fileNavyFleet'][self._1_VALUE][self._VALUE]
        else:
            data.wind.attrs['WIND_FILE'] = sim_widgets['fileWindAscii'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_TYPE'] = sim_widgets['cbxWindGridType'][self._1_VALUE][self._VALUE].capitalize()
        data.wind.attrs['WIND_GRID_FILE'] = sim_widgets['fileWindGrid'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_NUM_X_VALUES'] = sim_widgets['edtFleetNumX'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_NUM_Y_VALUES'] = sim_widgets['edtFleetNumY'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_MIN_X_LOCATION'] = sim_widgets['edtFleetMinX'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_MAX_Y_LOCATION'] = sim_widgets['edtFleetMaxY'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_TIME_INCREMENT'] = sim_widgets['edtFleetInc'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_X_DISTANCE'] = sim_widgets['edtFleetDistX'][self._1_VALUE][self._VALUE]
        data.wind.attrs['WIND_GRID_Y_DISTANCE'] = sim_widgets['edtFleetDistY'][self._1_VALUE][self._VALUE]
        data.wind.attrs['OCEAN_WIND_FILE'] = sim_widgets['fileOceanWind'][self._1_VALUE][self._VALUE]
        data.wind.attrs['OCEAN_PRESSURE_FILE'] = sim_widgets['fileOceanPressure'][self._1_VALUE][self._VALUE]
        data.wind.attrs['OCEAN_XY_FILE'] = sim_widgets['fileOceanXY'][self._1_VALUE][self._VALUE]

        if 'tblColName' in sim_widgets and sim_widgets['tblColName']:
            name = [row[self._VALUE] for row in sim_widgets['tblColName']]
            x = [row[self._VALUE] for row in sim_widgets['tblColX']]
            y = [row[self._VALUE] for row in sim_widgets['tblColY']]
            height = [row[self._VALUE] for row in sim_widgets['tblColHeight']]
            direction = []
            next_curve_id = 0
            _curve_x = 2  # X values are in this index when coming from the widget map
            _curve_y = 3
            for curve_row in sim_widgets['tblColCrvDir']:
                next_curve_id += 1
                direction.append(next_curve_id)
                # The velocity column is the same as the direction column since the velocity column didn't exist in
                # older versions.
                new_curve = {
                    'time': xr.DataArray(data=np.array(curve_row[self._VALUE][_curve_x], dtype=float)),
                    'direction': xr.DataArray(data=np.array(curve_row[self._VALUE][_curve_y], dtype=float)),
                    'velocity': xr.DataArray(data=np.array(curve_row[self._VALUE][_curve_y], dtype=float)),
                }
                data.set_direction_curve_from_meteorological_station(next_curve_id, xr.Dataset(data_vars=new_curve))
            meteorological_stations_table = {
                'name': xr.DataArray(data=np.array(name, dtype=object)),
                'x': xr.DataArray(data=np.array(x, dtype=float)),
                'y': xr.DataArray(data=np.array(y, dtype=float)),
                'height': xr.DataArray(data=np.array(height, dtype=float)),
                'direction': xr.DataArray(data=np.array(direction, dtype=int))
            }
            meteorological_stations = {'NEXT_CURVE_ID': next_curve_id}
            data.meteorological_stations_table = xr.Dataset(
                attrs=meteorological_stations, data_vars=meteorological_stations_table
            )

        if 'edtWindTime' in sim_widgets and sim_widgets['edtWindTime']:
            time = [row[self._VALUE] for row in sim_widgets['edtWindTime']]
            direction = [row[self._VALUE] for row in sim_widgets['edtWindDir']]
            velocity = [row[self._VALUE] for row in sim_widgets['edtWindVel']]
            wind_from_table = {
                'time': xr.DataArray(data=np.array(time, dtype=float)),
                'direction': xr.DataArray(data=np.array(direction, dtype=float)),
                'velocity': xr.DataArray(data=np.array(velocity, dtype=float))
            }
            data.wind_from_table = xr.Dataset(data_vars=wind_from_table)

    def _set_output_values(self, data, sim_widgets):
        """Sets the values in the output tab of the model control from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        data.output.attrs['SIMULATION_LABEL'] = sim_widgets['edtSimLabel'][self._1_VALUE][self._VALUE]
        data.output.attrs['WSE_LIST'] = sim_widgets['cbxWSE'][self._1_VALUE][self._VALUE]
        data.output.attrs['CURRENT_VELOCITY_LIST'] = sim_widgets['cbxCurrentVelocity'][self._1_VALUE][self._VALUE]
        data.output.attrs['CURRENT_MAGNITUDE'] = sim_widgets['togCurrentMagnitude'][self._1_VALUE][self._VALUE]
        data.output.attrs['USE_MORPHOLOGY'] = sim_widgets['togMorph'][self._1_VALUE][self._VALUE]
        data.output.attrs['MORPHOLOGY_LIST'] = sim_widgets['cbxMorph'][self._1_VALUE][self._VALUE]
        data.output.attrs['MORPHOLOGY_CHANGE'] = sim_widgets['togMorphChange'][self._1_VALUE][self._VALUE]
        data.output.attrs['USE_TRANSPORT'] = sim_widgets['togTransport'][self._1_VALUE][self._VALUE]
        data.output.attrs['TRANSPORT_LIST'] = sim_widgets['cbxTransport'][self._1_VALUE][self._VALUE]
        data.output.attrs['SEDIMENT_TOTAL_LOAD_CAPACITY'] = sim_widgets['togSedTLCapacity'][self._1_VALUE][self._VALUE]
        data.output.attrs['SEDIMENT_TOTAL_LOAD_CONCENTRATION'] =\
            sim_widgets['togSedTLConcentration'][self._1_VALUE][self._VALUE]
        data.output.attrs['FRACTION_SUSPENDED'] = sim_widgets['togFractionSuspended'][self._1_VALUE][self._VALUE]
        data.output.attrs['FRACTION_BEDLOAD'] = sim_widgets['togFractionBedload'][self._1_VALUE][self._VALUE]
        data.output.attrs['USE_WAVE'] = sim_widgets['togWaves'][self._1_VALUE][self._VALUE]
        data.output.attrs['WAVE_LIST'] = sim_widgets['cbxWaves'][self._1_VALUE][self._VALUE]
        data.output.attrs['WAVE_DISSIPATION'] = sim_widgets['togWaveDissipation'][self._1_VALUE][self._VALUE]
        data.output.attrs['USE_WIND'] = sim_widgets['togWind'][self._1_VALUE][self._VALUE]
        data.output.attrs['WIND_LIST'] = sim_widgets['cbxWind'][self._1_VALUE][self._VALUE]
        data.output.attrs['WIND_SPEED'] = sim_widgets['togWindSpeed'][self._1_VALUE][self._VALUE]
        data.output.attrs['USE_EDDY_VISCOSITY'] = sim_widgets['togEddyVisc'][self._1_VALUE][self._VALUE]
        data.output.attrs['EDDY_VISCOSITY_LIST'] = sim_widgets['cbxEddyVisc'][self._1_VALUE][self._VALUE]
        if 'togWaveTable' not in sim_widgets:
            # Add all four objects.
            sim_widgets['togWaveTable'] = [('none', -1, 0)]
            sim_widgets['colWaveStartTime'] = [('none', 0, 0.0)]
            sim_widgets['colWaveIncrement'] = [('none', 0, 1.0)]
            sim_widgets['colWaveEndTime'] = [('none', 0, 720.0)]
        if 'togStatOutput' in sim_widgets:
            data.output.attrs['ENABLE_STATISTICS'] = sim_widgets['togStatOutput'][self._1_VALUE][self._VALUE]
        else:
            data.output.attrs['ENABLE_STATISTICS'] = 1 \
                if sim_widgets['togHydroTable'][self._1_VALUE][self._VALUE] == 1 or \
                sim_widgets['togSedTable'][self._1_VALUE][self._VALUE] == 1 or \
                sim_widgets['togSalinityTable'][self._1_VALUE][self._VALUE] == 1 or \
                sim_widgets['togWaveTable'][self._1_VALUE][self._VALUE] == 1 else 0
        data.output.attrs['ENABLE_HYDRO_STATISTICS'] = sim_widgets['togHydroTable'][self._1_VALUE][self._VALUE]
        data.output.attrs['HYDRO_START_TIME'] = sim_widgets['colHydroStartTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['HYDRO_INCREMENT'] = sim_widgets['colHydroIncrement'][self._1_VALUE][self._VALUE]
        data.output.attrs['HYDRO_END_TIME'] = sim_widgets['colHydroEndTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['ENABLE_SEDIMENT_STATISTICS'] = sim_widgets['togSedTable'][self._1_VALUE][self._VALUE]
        data.output.attrs['SEDIMENT_START_TIME'] = sim_widgets['colSedStartTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['SEDIMENT_INCREMENT'] = sim_widgets['colSedIncrement'][self._1_VALUE][self._VALUE]
        data.output.attrs['SEDIMENT_END_TIME'] = sim_widgets['colSedEndTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['ENABLE_SALINITY_STATISTICS'] = sim_widgets['togSalinityTable'][self._1_VALUE][self._VALUE]
        data.output.attrs['SALINITY_START_TIME'] = sim_widgets['colSalinityStartTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['SALINITY_INCREMENT'] = sim_widgets['colSalinityIncrement'][self._1_VALUE][self._VALUE]
        data.output.attrs['SALINITY_END_TIME'] = sim_widgets['colSalinityEndTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['ENABLE_WAVE_STATISTICS'] = sim_widgets['togWaveTable'][self._1_VALUE][self._VALUE]
        data.output.attrs['WAVE_START_TIME'] = sim_widgets['colWaveStartTime'][self._1_VALUE][self._VALUE]
        data.output.attrs['WAVE_INCREMENT'] = sim_widgets['colWaveIncrement'][self._1_VALUE][self._VALUE]
        data.output.attrs['WAVE_END_TIME'] = sim_widgets['colWaveEndTime'][self._1_VALUE][self._VALUE]
        if 'cbxSolutionOutput' in sim_widgets:
            sol_output_type = sim_widgets['cbxSolutionOutput'][self._1_VALUE][self._VALUE]
            if sol_output_type == 'XMDF Binary Output':
                sol_output_type = 'XMDF binary output'
            elif sol_output_type == 'ASCII Output only':
                sol_output_type = 'ASCII output only'
        else:
            sol_output_type = 'XMDF binary output'
        data.output.attrs['SOLUTION_OUTPUT'] = sol_output_type
        if 'togXmdfCompression' in sim_widgets:
            data.output.attrs['XMDF_COMPRESSION'] = sim_widgets['togXmdfCompression'][self._1_VALUE][self._VALUE]
        if 'togSingleSolution' in sim_widgets:
            data.output.attrs['SINGLE_SOLUTION'] = sim_widgets['togSingleSolution'][self._1_VALUE][self._VALUE]
        if 'togWriteASCIIinput' in sim_widgets:
            data.output.attrs['WRITE_ASCII'] = sim_widgets['togWriteASCIIinput'][self._1_VALUE][self._VALUE]
        data.output.attrs['TECPLOT'] = sim_widgets['togTecplot'][self._1_VALUE][self._VALUE]
        self._add_output_table(data, sim_widgets, 1)
        self._add_output_table(data, sim_widgets, 2)
        self._add_output_table(data, sim_widgets, 3)
        self._add_output_table(data, sim_widgets, 4)

    def _add_output_table(self, data, sim_widgets, list_number):
        """Adds an output time list from the widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
            list_number (int): The number of the list (1-4).
        """
        start_widget = f'colList{list_number}StartTime'
        if start_widget in sim_widgets and sim_widgets[start_widget]:
            start_time = [row[self._VALUE] for row in sim_widgets[start_widget]]
            end_time = [row[self._VALUE] for row in sim_widgets[f'colList{list_number}EndTime']]
            increment = [row[self._VALUE] for row in sim_widgets[f'colList{list_number}Increment']]
            list_table = {
                'start_time': xr.DataArray(data=np.array(start_time, dtype=float)),
                'increment': xr.DataArray(data=np.array(increment, dtype=float)),
                'end_time': xr.DataArray(data=np.array(end_time, dtype=float))
            }
            if list_number == 1:
                data.list_1_table = xr.Dataset(data_vars=list_table)
            elif list_number == 2:
                data.list_2_table = xr.Dataset(data_vars=list_table)
            elif list_number == 3:
                data.list_3_table = xr.Dataset(data_vars=list_table)
            elif list_number == 4:
                data.list_4_table = xr.Dataset(data_vars=list_table)

    def _set_dredge_values(self, data, sim_widgets):
        """Sets the values in the dredge dialog from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
        """
        if 'togEnableDredgeModule' not in sim_widgets:
            return
        data.dredge.attrs['ENABLE_DREDGE'] = sim_widgets['togEnableDredgeModule'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['DREDGE_NAME'] = sim_widgets['edtDM_DredgeOpName'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets['ccDMUpdateInterval'], data.dredge, 'UPDATE_INTERVAL_VALUE', 'UPDATE_INTERVAL_UNITS'
        )
        data.dredge.attrs['DREDGE_DATASET'] = self._dataset_widget_to_uuid(sim_widgets['dsetDMDredge'])
        data.dredge.attrs['DREDGE_METHOD'] = sim_widgets['cbxDMDredgeMethod'][self._1_VALUE][self._VALUE].capitalize()
        data.dredge.attrs['SPECIFIED_CELL'] = sim_widgets['edtDMDredgeCell'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['DREDGE_RATE_VALUE'] = sim_widgets['edtDMDredgeRate'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['DREDGE_RATE_UNITS'] = sim_widgets['cbxDMDredgeRate'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_METHOD'] = sim_widgets['cbxDMTriggerMethod'][self._1_VALUE][self._VALUE].capitalize()
        data.dredge.attrs['TRIGGER_DEPTH_VALUE'] = sim_widgets['edtDMTriggerDepth'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_DEPTH_UNITS'] = sim_widgets['cbxDMTriggerDepthUnit'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_VOLUME_VALUE'] = sim_widgets['edtDMTriggerVolume'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_VOLUME_UNITS'] = sim_widgets['cbxDMTriggerVolumeUnit'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_PERCENT'] = sim_widgets['edtDMTriggerPercent'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_PERCENT_DEPTH_VALUE'] =\
            sim_widgets['edtDMTriggerPercentDepth'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['TRIGGER_PERCENT_DEPTH_UNITS'] =\
            sim_widgets['cbxDMTriggerPercentDepthUnit'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['DISTRIBUTION'] = sim_widgets['cbxDMDistribution'][self._1_VALUE][self._VALUE]
        data.dredge.attrs['ENABLE_DIAGNOSTIC'] = sim_widgets['togEnableDredgeDiag'][self._1_VALUE][self._VALUE]

        if 'edtDMTimeStart' in sim_widgets:
            start = [row[self._VALUE] for row in sim_widgets['edtDMTimeStart']]
            finish = [row[self._VALUE] for row in sim_widgets['edtDMTimeFinish']]
            dredge_time_periods_table = {
                'start': xr.DataArray(data=np.array(start, dtype=float)),
                'finish': xr.DataArray(data=np.array(finish, dtype=float))
            }
            data.dredge_time_periods = xr.Dataset(data_vars=dredge_time_periods_table)
        self._set_dredge_placement_values(data, sim_widgets, 1)
        self._set_dredge_placement_values(data, sim_widgets, 2)
        self._set_dredge_placement_values(data, sim_widgets, 3)

    def _set_dredge_placement_values(self, data, sim_widgets, placement):
        """Sets the placement values in the dredge dialog from widget values.

        Args:
            data (SimulationData): The component based simulation data.
            sim_widgets (dict): A dictionary of widgets and values.
            placement (int): The placement number (1-3).
        """
        data.dredge_placement.attrs[f'DEFINE_PLACEMENT_{placement}'] =\
            sim_widgets[f'togDMPlace{placement}'][self._1_VALUE][self._VALUE]
        data.dredge_placement.attrs[f'PLACEMENT_{placement}_DATASET'] =\
            self._dataset_widget_to_uuid(sim_widgets[f'dsetDMPlace{placement}'])
        method = sim_widgets[f'cbxDMPlace{placement}Method'][self._1_VALUE][self._VALUE].capitalize()
        if method == 'Cell':  # Guess the widget value can be 'Cell' or 'Specified Cell' in old projects
            method = 'Specified cell'
        data.dredge_placement.attrs[f'PLACEMENT_{placement}_METHOD'] = method
        data.dredge_placement.attrs[f'PLACEMENT_{placement}_METHOD_CELL'] =\
            sim_widgets[f'edtDMPlace{placement}MethodCell'][self._1_VALUE][self._VALUE]
        data.dredge_placement.attrs[f'PLACEMENT_{placement}_PERCENTAGE'] =\
            sim_widgets[f'edtDMPlace{placement}Percentage'][self._1_VALUE][self._VALUE]
        data.dredge_placement.attrs[f'PLACEMENT_{placement}_LIMIT_METHOD'] =\
            sim_widgets[f'cbxDMPlace{placement}LimitMethod'][self._1_VALUE][self._VALUE]
        self._set_value_and_unit(
            sim_widgets[f'ccDMPlace{placement}LimitDepth'], data.dredge_placement,
            f'PLACEMENT_{placement}_LIMIT_DEPTH_VALUE', f'PLACEMENT_{placement}_LIMIT_DEPTH_UNITS'
        )
        self._set_value_and_unit(
            sim_widgets[f'ccDMPlace{placement}LimitThickness'], data.dredge_placement,
            f'PLACEMENT_{placement}_LIMIT_THICKNESS_VALUE', f'PLACEMENT_{placement}_LIMIT_THICKNESS_UNITS'
        )

    def _migrate_bc_coverage(self, arc_widgets, old_uuid):
        """Migrate a boundary condition coverage to current component based version.

        Args:
            arc_widgets (dict): A dictionary of arcs, with ids as keys and values being
                                dictionaries of widget name to list of values.
            old_uuid (str): UUID of the coverage this came from.

        Returns:
            xmsapi.dmi.Component: The new component for this boundary conditions coverage.
        """
        comp_uuid = self._get_next_cov_comp_uuid()
        cov_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(cov_comp_dir, exist_ok=True)

        # Create the component data_object to send back to SMS.
        cov_main_file = os.path.join(str(cov_comp_dir), 'bc_comp.nc')
        cov_comp = Component(
            comp_uuid=comp_uuid, model_name='CMS-Flow', unique_name='BC_Component', main_file=cov_main_file
        )

        # Fill component data from widget values in the old database.
        cov_py_comp = BCComponent(cov_main_file)  # Initialize some default data
        data = cov_py_comp.data

        att_ids = []
        next_comp_id = 1
        comp_ids = []

        name = []
        bc_type = []
        flow_source = []
        constant_flow = []
        flow_curve = []
        specify_inflow_direction = []
        flow_direction = []
        flow_conveyance = []
        wse_source = []
        wse_const = []
        wse_forcing_curve = []
        use_velocity = []
        parent_cmsflow = []
        parent_adcirc_14_uuid = []
        parent_adcirc_14 = []
        parent_adcirc_solution_type = []
        parent_adcirc_63 = []
        parent_adcirc_64 = []
        parent_adcirc_solution = []
        parent_adcirc_start_time = []
        wse_offset_type = []
        wse_offset_const = []
        wse_offset_curve = []
        use_salinity_curve = []
        salinity_curve = []
        use_temperature_curve = []
        temperature_curve = []
        harmonic_table = []
        tidal_table = []

        flow_curve_id = 1
        wse_curve_id = 1
        wse_offset_curve_id = 1
        salinity_curve_id = 1
        temperature_curve_id = 1
        harmonic_table_id = 1
        tidal_table_id = 1
        for arc_id, widgets in arc_widgets.items():
            if arc_id < 0:
                continue
            name.append(widgets['edtBCName'][self._1_VALUE][self._VALUE])
            widget_bc_type = widgets['cbxBCType'][self._1_VALUE][self._VALUE]
            # check for options that are no longer used
            if widget_bc_type in ['Unassigned', 'Flow rate-forcing', 'WSE-forcing']:
                bc_type.append(widget_bc_type)
            else:
                bc_type.append('Unassigned')
            flow_source.append(widgets['cbxFlowSource'][self._1_VALUE][self._VALUE])
            constant_flow.append(widgets['edtFlowConst'][self._1_VALUE][self._VALUE])
            x, y = self._add_bc_curve(widgets['crvFlowForcing'][self._1_VALUE])
            if len(x) > 0:
                flow_curve_data = {
                    'Time (hrs)': x,
                    'Flow (m^3/s Total, not per cell)': y,
                }
                data.set_flow_forcing_curve(flow_curve_id, xr.Dataset(data_vars=flow_curve_data))
                flow_curve.append(flow_curve_id)
                flow_curve_id += 1
            else:
                flow_curve.append(0)
            if 'togFluxDirection' in widgets:  # not in SMS 12.3
                specify_inflow_direction.append(widgets['togFluxDirection'][self._1_VALUE][self._VALUE])
            else:
                specify_inflow_direction.append(0)
            flow_direction.append(widgets['edtFlowDirection'][self._1_VALUE][self._VALUE])
            flow_conveyance.append(widgets['edtFlowConveyance'][self._1_VALUE][self._VALUE])
            widget_wse_source = widgets['cbxWSESource'][self._1_VALUE][self._VALUE]
            if widget_wse_source == 'Tidal Constituent':
                widget_wse_source = 'Tidal constituent'
            elif widget_wse_source == 'Tidal Database':
                widget_wse_source = 'External tidal'
            wse_source.append(widget_wse_source)
            wse_const.append(widgets['edtWSEConst'][self._1_VALUE][self._VALUE])
            x, y = self._add_bc_curve(widgets['crvWSEForcing'][self._1_VALUE])
            if len(x) > 0:
                wse_curve_data = {
                    'Time (hrs)': x,
                    'WSE (m)': y,
                }
                data.set_wse_forcing_curve(wse_curve_id, xr.Dataset(data_vars=wse_curve_data))
                wse_forcing_curve.append(wse_curve_id)
                wse_curve_id += 1
            else:
                wse_forcing_curve.append(0)
            use_velocity.append(widgets['togUseVelocity'][self._1_VALUE][self._VALUE])
            parent_cmsflow.append(widgets['fileCMCard'][self._1_VALUE][self._VALUE])
            parent_adcirc_14.append(widgets['fileADCIRC14'][self._1_VALUE][self._VALUE])
            parent_adcirc_solution_type.append(widgets['cbxADCIRCSolType'][self._1_VALUE][self._VALUE])
            # TODO: If solution type is XMDF should we read the fort.14 and send a mesh to SMS? Not sure it is worth
            #       I would rather get model native import working well.
            parent_adcirc_14_uuid.append('')
            parent_adcirc_63.append(widgets['fileADCIRC63'][self._1_VALUE][self._VALUE])
            parent_adcirc_64.append(widgets['fileADCIRC64'][self._1_VALUE][self._VALUE])
            parent_adcirc_solution.append(widgets['fileADCIRCSol'][self._1_VALUE][self._VALUE])
            start_time = self._create_date_time_string(widgets['dateParentStart'][self._1_VALUE][self._VALUE])
            parent_adcirc_start_time.append(start_time)
            if 'cbxWSEoffsetType' in widgets:  # not in SMS 12.3
                wse_offset_type.append(widgets['cbxWSEoffsetType'][self._1_VALUE][self._VALUE])
            else:
                wse_offset_type.append('Constant')
            wse_offset_const.append(widgets['edtOffset'][self._1_VALUE][self._VALUE])
            if 'crvWSEoffset' in widgets:  # not in SMS 12.3
                x, y = self._add_bc_curve(widgets['crvWSEoffset'][self._1_VALUE])
                if len(x) > 0:
                    wse_offset_curve_data = {
                        'Time (hrs)': x,
                        'WSE Offset (m)': y,
                    }
                    data.set_wse_offset_curve(wse_offset_curve_id, xr.Dataset(data_vars=wse_offset_curve_data))
                    wse_offset_curve.append(wse_offset_curve_id)
                    wse_offset_curve_id += 1
                else:
                    wse_offset_curve.append(0)
            else:
                wse_offset_curve.append(0)
            if 'togBCSalinity' in widgets:
                use_salinity_curve.append(widgets['togBCSalinity'][self._1_VALUE][self._VALUE])
            else:
                use_salinity_curve.append(0)
            x, y = self._add_bc_curve(widgets['crvSalinity'][self._1_VALUE])
            if len(x) > 0:
                salinity_curve_data = {
                    'Time (hrs)': x,
                    'Salinity (ppt)': y,
                }
                data.set_salinity_curve(salinity_curve_id, xr.Dataset(data_vars=salinity_curve_data))
                salinity_curve.append(salinity_curve_id)
                salinity_curve_id += 1
            else:
                salinity_curve.append(0)
            if 'togBCTemperature' in widgets:
                use_temperature_curve.append(widgets['togBCTemperature'][self._1_VALUE][self._VALUE])
            else:
                use_temperature_curve.append(0)
            if 'crvTemperature' in widgets:
                x, y = self._add_bc_curve(widgets['crvTemperature'][self._1_VALUE])
            else:
                x = []
                y = []
            if len(x) > 0:
                temperature_curve_data = {
                    'Time (hrs)': x,
                    'Temperature (dec C)': y,
                }
                data.set_temperature_curve(temperature_curve_id, xr.Dataset(data_vars=temperature_curve_data))
                temperature_curve.append(temperature_curve_id)
                temperature_curve_id += 1
            else:
                temperature_curve.append(0)

            if 'tblHarmonicSpeed' in widgets and widgets['tblHarmonicSpeed']:
                speed = [row[self._VALUE] for row in widgets['tblHarmonicSpeed']]
                amp = [row[self._VALUE] for row in widgets['tblHarmonicAmp']]
                phase = [row[self._VALUE] for row in widgets['tblHarmonicPhase']]
                harmonic_table_data = {
                    'speed': xr.DataArray(data=np.array(speed, dtype=float)),
                    'amplitude': xr.DataArray(data=np.array(amp, dtype=float)),
                    'phase': xr.DataArray(data=np.array(phase, dtype=float))
                }
                data.set_harmonic_table(harmonic_table_id, xr.Dataset(data_vars=harmonic_table_data))
            else:
                data.set_harmonic_table(harmonic_table_id, data.default_harmonic_table())
            harmonic_table.append(harmonic_table_id)
            harmonic_table_id += 1

            if 'cbxTidalConstituent' in widgets and widgets['cbxTidalConstituent']:
                constituent = [row[self._VALUE] for row in widgets['cbxTidalConstituent']]
                amp = [row[self._VALUE] for row in widgets['tblTidalAmp']]
                phase = [row[self._VALUE] for row in widgets['tblTidalPhase']]
                tidal_table_data = {
                    'constituent': xr.DataArray(data=np.array(constituent, dtype=object)),
                    'amplitude': xr.DataArray(data=np.array(amp, dtype=float)),
                    'phase': xr.DataArray(data=np.array(phase, dtype=float))
                }
                data.set_tidal_table(tidal_table_id, xr.Dataset(data_vars=tidal_table_data))
            else:
                data.set_tidal_table(tidal_table_id, data.default_tidal_table())

            tidal_table.append(tidal_table_id)
            tidal_table_id += 1

            if widget_wse_source == 'External tidal':
                # create a tidal constituents component
                db_type = widgets['cbxTidalDBType'][self._1_VALUE][self._VALUE]
                constituents = []
                if 'cbxTidalDBConstituent' in widgets and widgets['cbxTidalDBConstituent']:
                    constituents = [row[self._VALUE] for row in widgets['cbxTidalDBConstituent']]
                self._add_tidal_component(db_type, constituents, old_uuid)

            comp_ids.append(next_comp_id)
            next_comp_id += 1
            att_ids.append(arc_id)

        if next_comp_id > 1:
            arc_table = {
                'name': ('comp_id', np.array(name, dtype=object)),
                'bc_type': ('comp_id', np.array(bc_type, dtype=object)),
                'flow_source': ('comp_id', np.array(flow_source, dtype=object)),
                'constant_flow': ('comp_id', np.array(constant_flow, dtype=float)),
                'flow_curve': ('comp_id', np.array(flow_curve, dtype=int)),
                'specify_inflow_direction': ('comp_id', np.array(specify_inflow_direction, dtype=int)),
                'flow_direction': ('comp_id', np.array(flow_direction, dtype=float)),
                'flow_conveyance': ('comp_id', np.array(flow_conveyance, dtype=float)),
                'wse_source': ('comp_id', np.array(wse_source, dtype=object)),
                'wse_const': ('comp_id', np.array(wse_const, dtype=float)),
                'wse_forcing_curve': ('comp_id', np.array(wse_forcing_curve, dtype=int)),
                'use_velocity': ('comp_id', np.array(use_velocity, dtype=int)),
                'parent_cmsflow': ('comp_id', np.array(parent_cmsflow, dtype=object)),
                'parent_adcirc_14_uuid': ('comp_id', np.array(parent_adcirc_14_uuid, dtype=object)),
                'parent_adcirc_14': ('comp_id', np.array(parent_adcirc_14, dtype=object)),
                'parent_adcirc_solution_type': ('comp_id', np.array(parent_adcirc_solution_type, dtype=object)),
                'parent_adcirc_63': ('comp_id', np.array(parent_adcirc_63, dtype=object)),
                'parent_adcirc_64': ('comp_id', np.array(parent_adcirc_64, dtype=object)),
                'parent_adcirc_solution_wse': ('comp_id', np.array(parent_adcirc_solution, dtype=object)),
                'parent_adcirc_solution': ('comp_id', np.array(parent_adcirc_solution, dtype=object)),
                'parent_adcirc_start_time': ('comp_id', np.array(parent_adcirc_start_time, dtype=object)),
                'wse_offset_type': ('comp_id', np.array(wse_offset_type, dtype=object)),
                'wse_offset_const': ('comp_id', np.array(wse_offset_const, dtype=float)),
                'wse_offset_curve': ('comp_id', np.array(wse_offset_curve, dtype=int)),
                'use_salinity_curve': ('comp_id', np.array(use_salinity_curve, dtype=int)),
                'salinity_curve': ('comp_id', np.array(salinity_curve, dtype=int)),
                'use_temperature_curve': ('comp_id', np.array(use_temperature_curve, dtype=int)),
                'temperature_curve': ('comp_id', np.array(temperature_curve, dtype=int)),
                'harmonic_table': ('comp_id', np.array(harmonic_table, dtype=int)),
                'tidal_table': ('comp_id', np.array(tidal_table, dtype=int)),
            }
            coords = {'comp_id': comp_ids}
            data.info.attrs['next_comp_id'] = next_comp_id
            data.arcs = xr.Dataset(data_vars=arc_table, coords=coords)
        data.commit()

        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        id_file = os.path.join(str(cov_comp_dir), BC_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(str(cov_comp_dir), BC_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)
        return cov_comp

    def _add_tidal_component(self, db_type, constituents, old_cov_uuid):
        """Adds a tidal component.

        Args:
            db_type (str): The old type of database that was used.
            constituents (list): A list of strings of used constituent names.
            old_cov_uuid (str): UUID of the old coverage this came from.
        """
        # Create a folder and UUID for the new tidal component.
        comp_uuid = self._get_next_tidal_comp_uuid()
        tidal_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(tidal_comp_dir, exist_ok=True)

        # Create the tidal constituent simulation data_object to send back to SMS.
        tidal_sim_uuid = str(uuid.uuid4())
        tidal_sim = Simulation(model='Tidal Constituents', sim_uuid=tidal_sim_uuid, name='Tidal Constituents')

        # Create the component data_object to send back to SMS.
        tidal_main_file = os.path.join(str(tidal_comp_dir), 'tidal_comp.nc')
        tidal_comp = Component(
            comp_uuid=comp_uuid,
            model_name='Tidal Constituents',
            unique_name='Tidal_Component',
            main_file=tidal_main_file
        )

        # Fill component data from widget values in the old database.
        tidal_py_comp = TidalComponent(tidal_main_file)  # Initialize some default data
        data = tidal_py_comp.data
        if db_type.find('LeProvost') != -1:
            data._info.attrs['source'] = LEPROVOST_INDEX
            tidal_model = Constituents('leprovost')
        else:
            data._info.attrs['source'] = ADCIRC_INDEX
            tidal_model = Constituents('adcirc2015')
        all_cons = data.get_default_con_dataset(tidal_model)
        con_list = all_cons.name.data.tolist()
        for idx, con in enumerate(con_list):
            if con in constituents:
                all_cons.enabled[idx] = 1
        data.cons = all_cons
        data.commit()
        self._new_tidal_comps.append((tidal_sim, tidal_comp))
        self._cov_to_tidal_uuid[old_cov_uuid] = tidal_sim_uuid

    def _migrate_save_points_coverage(self, cov_widgets, point_widgets):
        """Migrate a save points coverage to current component based version.

        Args:
            cov_widgets (dict): A dictionary of widget name to list of values for coverage widgets.
            point_widgets (dict): A dictionary of points, with ids as keys and values being
                                  dictionaries of widget name to list of values.

        Returns:
            xmsapi.dmi.Component: The new component for this save points coverage.
        """
        # Create a folder and UUID for the new coverage component.
        comp_uuid = self._get_next_cov_comp_uuid()
        cov_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(cov_comp_dir, exist_ok=True)

        # Create the component data_object to send back to SMS.
        cov_main_file = os.path.join(str(cov_comp_dir), 'save_points_comp.nc')
        cov_comp = Component(
            comp_uuid=comp_uuid, model_name='CMS-Flow', unique_name='Save_Points_Component', main_file=cov_main_file
        )

        # Fill component data from widget values in the old database.
        cov_py_comp = SavePointsComponent(cov_main_file)  # Initialize some default data
        data = cov_py_comp.data

        self._set_value_and_unit(cov_widgets['ccSavePtsHydro'], data.general, 'HYDRO_VALUE', 'HYDRO_UNITS')
        self._set_value_and_unit(cov_widgets['ccSavePtsSed'], data.general, 'SEDIMENT_VALUE', 'SEDIMENT_UNITS')
        self._set_value_and_unit(cov_widgets['ccSavePtsSalinity'], data.general, 'SALINITY_VALUE', 'SALINITY_UNITS')
        self._set_value_and_unit(cov_widgets['ccSavePtsWaves'], data.general, 'WAVES_VALUE', 'WAVES_UNITS')
        att_ids = []
        next_comp_id = 1
        comp_ids = []

        name = []
        hydro = []
        sediment = []
        salinity = []
        waves = []
        for pt_id, widgets in point_widgets.items():
            if pt_id < 0:
                continue
            name.append(widgets['edtSavePointName'][self._1_VALUE][self._VALUE])
            hydro.append(widgets['togSavePointHydro'][self._1_VALUE][self._VALUE])
            sediment.append(widgets['togSavePointSed'][self._1_VALUE][self._VALUE])
            salinity.append(widgets['togSavePointSalinity'][self._1_VALUE][self._VALUE])
            waves.append(widgets['togSavePointWaves'][self._1_VALUE][self._VALUE])
            comp_ids.append(next_comp_id)
            next_comp_id += 1
            att_ids.append(pt_id)
        if next_comp_id > 1:
            point_table = {
                'name': ('comp_id', np.array(name, dtype=object)),
                'hydro': ('comp_id', np.array(hydro, dtype=np.int32)),
                'sediment': ('comp_id', np.array(sediment, dtype=np.int32)),
                'salinity': ('comp_id', np.array(salinity, dtype=np.int32)),
                'waves': ('comp_id', np.array(waves, dtype=np.int32)),
            }
            coords = {'comp_id': comp_ids}
            data.info.attrs['next_comp_id'] = next_comp_id
            data.points = xr.Dataset(data_vars=point_table, coords=coords)
        data.commit()

        # Write component id and save point att ids to a file so we can initialize them in get_initial_display_options
        id_file = os.path.join(str(cov_comp_dir), SAVE_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(str(cov_comp_dir), SAVE_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)
        return cov_comp

    def _get_do_coverage(self, name, pts=None, arcs=None, polys=None):
        """Get a data_objects Coverage to send back to SMS.

        Args:
            name (str): Name of the new coverage
            pts (list): List of disjoint data_objects.parameters.Point locations in the new coverage, if any
            arcs (list): List of disjoint data_objects.parameters.Arc definitions in the new coverage, if any
            polys (list):  List of data_objects.parameters.Polygon definitions in the new coverage, if any

        Returns:
            data_objects.parameters.Coverage: The new coverage

        """
        new_cov = Coverage(name=name, uuid=str(uuid.uuid4()), projection=self._xms_data['projection'])
        if pts:
            new_cov.points = pts
        if arcs:
            new_cov.arcs = arcs
        if polys:
            new_cov.polygons = polys
        new_cov.complete()
        return new_cov

    def _add_bc_curve(self, curve):
        """Creates data arrays from the widget curve values.

        Args:
            curve (dict): A dictionary of a curve widgets values.
        """
        _curve_x = 2  # X values are in this index when coming from the widget map
        _curve_y = 3
        if not curve or len(curve[self._VALUE]) == 0 or len(curve[self._VALUE][_curve_x]) == 0:
            return [], []
        return (
            xr.DataArray(data=np.array(curve[self._VALUE][_curve_x], dtype=float)),
            xr.DataArray(data=np.array(curve[self._VALUE][_curve_y], dtype=float))
        )

    def _create_date_time_string(self, widget_time_value):
        """Creates a date time string from a widget value. Code adapted from XMDF library code.

        Args:
            widget_time_value (float or str): A floating point Julian datetime. May be stringified.

        Returns:
            Returns a date time string representation.
        """
        try:
            widget_time_value = float(widget_time_value)
        except ValueError:
            return self._default_datetime

        intgr = math.floor(widget_time_value)
        frac = widget_time_value - intgr
        gregjd = 2299161
        # Gregorian Calendar Correction
        if intgr >= gregjd:
            tmp = math.floor(((intgr - 1867216) - 0.25) / 36524.25)
            j1 = intgr + 1 + tmp - math.floor(0.25 * tmp)
        else:
            j1 = intgr

        # correction for half day offset
        dayfrac = frac + 0.5
        if dayfrac >= 1.0:
            dayfrac -= 1.0
            j1 += 1

        j2 = j1 + 1524
        j3 = math.floor(6680.0 + ((j2 - 2439870) - 122.1) / 365.25)
        j4 = math.floor(j3 * 365.25)
        j5 = math.floor((j2 - j4) / 30.6001)

        a_day = int(math.floor(j2 - j4 - math.floor(j5 * 30.6001)))
        a_mo = int(math.floor(j5 - 1))
        if a_mo > 12:
            a_mo -= 12

        a_yr = int(math.floor(j3 - 4715))
        if a_mo > 2:
            a_yr -= 1
        if a_yr <= 0:
            a_yr -= 1

        # get time of day from day fraction
        a_hr = int(math.floor(dayfrac * 24.0))
        a_min = int(math.floor((dayfrac * 24.0 - a_hr) * 60.0))
        f = ((dayfrac * 24.0 - a_hr) * 60.0 - a_min) * 60.0
        a_sec = int(math.floor(f))
        f -= a_sec
        if f > 0.5:
            a_sec += 1
        if a_sec == 60:
            a_sec = 0
            a_min += 1
        if a_min == 60:
            a_min = 0
            a_hr += 1
        if a_hr == 24:
            a_hr = 0
            a_day += 1

        if a_yr < 0:
            a_yr *= -1

        start_date = datetime.datetime(year=a_yr, month=a_mo, day=a_day, hour=a_hr, minute=a_min, second=a_sec)
        date_str = start_date.strftime('%Y-%m-%dT%H:%M:%S')
        return date_str
