"""Runs model checks on the simulation."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ModelCheckError, Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.cmswave.components.simulation_component import SimulationComponent
from xms.cmswave.data import cmswave_consts as const


class SimulationCheck:
    """A class for running model checks on a simulation."""
    def __init__(self):
        """Constructor."""
        self.query = Query()
        self.errors = []
        self.sim_comp = None
        self.sim_uuid = None
        # self.debugger = open("debug.txt", "w")

    def get_simulation(self):
        """Gets the CMS-Wave simulation being checked."""
        # Get the simulation component main file.
        try:
            self.sim_uuid = self.query.current_item_uuid()
            do_comp = self.query.item_with_uuid(self.sim_uuid, model_name='CMS-Wave', unique_name='Sim_Component')
            self.sim_comp = SimulationComponent(do_comp.main_file)
        except Exception:
            raise Exception(
                'Unknown error retrieving CMS-Wave simulation Model Control data.',
                'Running CMS-Wave model checks requires the current CMS-Wave Model Control values.',
                'Please try running model checks again.'
            )

    def check_grid_projection(self, grid):
        """
        Checks to make sure vertical and horizontal units are in meters.

        Also a helper function for check_grid()

        Args:
            grid (:obj:`data_objects.Parameters.Spatial.SpatialVector.SpatialVector`): Grid to use for projection.
        """
        proj = grid.projection

        # check the vertical units
        vert_units = proj.vertical_units
        if vert_units != "METERS":
            vert_error = ModelCheckError()
            vert_error.problem_text = "The current vertical units of the grid are not metric."
            vert_error.description_text = "CMS-Wave requires vertical units to be meters."
            vert_error.fix_text = (
                "In the top-level \"Display\" menu, select the \"Display Projection...\" menu "
                "item. In the \"Vertical\" section of the \"Display Projection\" dialog, change"
                " the units to meters."
            )
            self.errors.append(vert_error)

        horiz_units = proj.horizontal_units
        if horiz_units != "METERS":
            horiz_error = ModelCheckError()
            horiz_error.problem_text = "The current horizontal units of the grid are not metric."
            horiz_error.description_text = "CMS-Wave requires horizontal units to be meters."
            horiz_error.fix_text = (
                "In the top-level \"Display\" menu, select the \"Display Projection...\" menu "
                "item. In the \"Horizontal\" section of the \"Display Projection\" dialog, change"
                " the units to meters."
            )
            self.errors.append(horiz_error)

    def check_grid(self):
        """Checks the cms wave grid geometry to make sure there are elements in it.

        In the case the grid is empty, error texts are set. If the grid is good, the projection is checked.
        """
        # Make sure that we have a grid, and that it's units are meters.
        sim_item = tree_util.find_tree_node_by_uuid(tree_node=self.query.project_tree, uuid=self.sim_uuid)
        grid_item = tree_util.descendants_of_type(tree_root=sim_item, xms_types=['TI_CGRID2D_PTR'], allow_pointers=True)
        grid = None

        # Check for missing grid and variable i or j sizes. CMS-Wave grids must have constant i and j sizes.
        grid_error = False
        if not grid_item:  # Check for missing grid
            grid_error = True
        else:
            grid = self.query.item_with_uuid(grid_item[0].uuid)
            co_grid = read_grid_from_file(grid.cogrid_file)
            i_locs = co_grid.locations_y
            j_locs = co_grid.locations_x
            if not i_locs or not j_locs:  # Check for undefined grid
                grid_error = True

        if grid_error:
            model_error = ModelCheckError()
            model_error.problem_text = "No grid exists in this simulation."
            model_error.description_text = "A cartesian grid is required to run a CMS-Wave simulation."
            model_error.fix_text = (
                "Create a grid by right clicking on \"Map Data\" and selecting \"New Coverage\" "
                "Select \"CGrid Generator\" and click \"OK\". Build the grid by creating a grid "
                "frame. Convert the grid frame to a grid (right click on the coverage and select "
                "\"Convert\" > \"Map->2D Grid\"). Drag and drop the grid under the simulation."
            )
            self.errors.append(model_error)

        # If we have a grid, check its projection
        if grid:
            self.check_grid_projection(grid)

    def check_dset_selectors(self):
        """Checks for none dataset specification in widgets.

        If current dataset, Darcy friction, Manning's N friction, surge, wind, or ice datasets are
        empty, error text is set and appended to errors.
        """
        info = self.sim_comp.data.info

        # Current input is only for half-plane. Mind the dependency.
        if info.attrs['current_interaction'] == const.TEXT_USE_DATASET:
            if not info.attrs['current_uuid']:
                curr_error = ModelCheckError()
                curr_error.problem_text = "No current dataset has been specified."
                curr_error.description_text = (
                    "The option for transient current input has been enabled, but no "
                    "dataset has been selected."
                )
                curr_error.fix_text = (
                    "Right click on the simulation and select \"Model Control...\". In the "
                    "\"Parameters\" tab, push the \"Select\" button in the \"Current interaction\" "
                    "section. Choose a vector dataset for transient current input."
                )
                self.errors.append(curr_error)

        if info.attrs['friction'] == const.TEXT_DARCY_DATASET:
            if not info.attrs['darcy_uuid']:
                darcy_error = ModelCheckError()
                darcy_error.problem_text = "No Darcy-Weisbach friction dataset has been specified."
                darcy_error.description_text = (
                    "The option for Darcy-Weisbach friction input has been enabled, but no "
                    "dataset has been selected."
                )
                darcy_error.fix_text = (
                    "Right click on the simulation and select \"Model Control...\". In the "
                    "\"Parameters\" tab, push the \"Select\" button in the \"Bottom friction\" "
                    "section. Choose a scalar dataset for Darcy-Weisbach friction input."
                )
                self.errors.append(darcy_error)

        if info.attrs['friction'] == const.TEXT_MANNINGS_DATASET:
            if not info.attrs['manning_uuid']:
                manning_error = ModelCheckError()
                manning_error.problem_text = "No Manning's N friction dataset has been specified."
                manning_error.description_text = (
                    "The option for Manning's N friction input has been enabled, but no "
                    "dataset has been selected."
                )
                manning_error.fix_text = (
                    "Right click on the simulation and select \"Model Control...\". In the "
                    "\"Parameters\" tab, push the \"Select\" button in the \"Bottom friction\" "
                    "section. Choose a scalar dataset for Manning's N friction input."
                )
                self.errors.append(manning_error)

        if info.attrs['surge'] == const.TEXT_USE_DATASET:
            if not info.attrs['surge_uuid']:
                surge_error = ModelCheckError()
                surge_error.problem_text = "No surge dataset has been specified."
                surge_error.description_text = (
                    "The option for spatially varying surge input has been enabled, but no"
                    " dataset has been selected."
                )
                surge_error.fix_text = (
                    "Right click on the simulation and select \"Model Control...\". In the "
                    "\"Parameters\" tab, push the \"Select\" button in the \"Surge fields\" "
                    "section. Choose a scalar dataset for spatially varying surge input."
                )
                self.errors.append(surge_error)

        # Wind input exists if not prop only. Mind the dependency.
        source_terms_prop = info.attrs['source_terms'] == 'Source terms and propagation'
        if source_terms_prop and info.attrs['wind'] == const.TEXT_USE_DATASET:
            if not info.attrs['wind_uuid']:
                wind_error = ModelCheckError()
                wind_error.problem_text = "No wind dataset has been specified."
                wind_error.description_text = (
                    "The option for spatially varying wind input has been enabled, but no"
                    " dataset has been selected."
                )
                wind_error.fix_text = (
                    "Right click on the simulation and select \"Model Control...\". In the "
                    "\"Parameters\" tab, push the \"Select\" button in the \"Wind fields\" "
                    "section. Choose a vector dataset for spatially varying wind input."
                )
                self.errors.append(wind_error)

    def check_cov_selectors(self):
        """Checks for none specification of coverages in query.

        If observation station, or spectral coverage are empty, error text is set and appended to errors.
        """
        if self.sim_comp.data.info.attrs['boundary_source'] == 'Spectra (+ Wind)':
            result = self.query.item_with_uuid(self.sim_comp.data.info.attrs['spectral_uuid'], generic_coverage=True)
            if result is None:
                spec_error = ModelCheckError()
                spec_error.problem_text = "No spectral coverage has been linked to this simulation."
                spec_error.description_text = (
                    "The option for using a spectral coverage as boundary conditions has "
                    "been selected, but the simulation contains no spectral coverage"
                )
                spec_error.fix_text = (
                    "Create a spectral coverage by right clicking on \"Map Data\" and selecting "
                    "\"New Coverage\" Select \"Spectral\" and click \"OK\". Drag and drop the new"
                    " spectral coverage under the simulation."
                )
                self.errors.append(spec_error)

    def check_for_case_times(self):
        """Checks for none specification of case times.

        If no case times are specified, error text is set and appended to errors.
        """
        if self.sim_comp.data.case_times.Time.size == 0:
            horiz_error = ModelCheckError()
            horiz_error.problem_text = "No case times have been specified for this simulation."
            horiz_error.description_text = "CMS-Wave simulations need at least one case time."
            horiz_error.fix_text = (
                "To add case times, right click on the simulation and select \"Model Control...\"."
                " Case times are specified on the \"Boundary Control\" tab (in the \"Case data\" "
                "section). Click the \"Insert Above\"/\"Insert Below\" "
                "buttons under the table to insert case times."
            )
            self.errors.append(horiz_error)

    def run_checks(self):
        """
        Run checks for widgets, grid, dataset selectors, coverage selectors, and case times.

        If any error text is present, it is sent to query.
        """
        self.get_simulation()
        self.check_grid()
        self.check_dset_selectors()
        self.check_cov_selectors()
        self.check_for_case_times()

        if self.errors:
            self.query.add_model_check_errors(self.errors)
        self.query.send()
