"""Writes CMS-Wave structures rile."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.cmswave.data.structures_data import struct_type_id_from_name
from xms.cmswave.mapping.coverage_mapper import CoverageMapper


class StructuresWriter:
    """Writes CMS-Wave structures file."""
    def __init__(self, query):
        """Constructor."""
        # Get the simulation tree item
        self._query = query
        self._file_name = ''
        self._struct_grid_cells = None
        self._struct_df = None
        self._co_grid = None
        self.struct_data = None
        self.get_xmsdata()

    def get_xmsdata(self):
        """Get the xms data to export this file."""
        sim_uuid = self._query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
        self._file_name = sim_item.name + ".struct"
        sim_comp = self._query.item_with_uuid(sim_uuid, model_name='CMS-Wave', unique_name='Sim_Component')
        worker = CoverageMapper(self._query, sim_comp.main_file)
        worker.do_map()
        self.struct_data = worker.structure_data
        # Do this after mapping, so we can print other warnings.
        if not self.struct_data:
            return
        self._struct_grid_cells = worker.structure_comp_id_to_grid_cell_ids
        self._struct_df = self.struct_data.structures.to_dataframe()
        self._co_grid = worker.co_grid

    def write(self):
        """Write the CMS-Wave structures file."""
        if self.struct_data is None:
            return False
        with open(self._file_name, 'w') as file:
            num_struct_cells = 0
            ss = StringIO()
            struct_df = self.struct_data.structures.to_dataframe()
            for struct_id, cells in self._struct_grid_cells.items():
                # Why do we care about the num of structure types?
                # if struct_id == 0 or len(cells) < 1 or struct_id >= len(self._structure_types):
                if struct_id == -1 or len(cells) < 1:
                    continue  # Skip an unassigned bathymetry modification structure
                num_struct_cells += len(cells)
                df_row = struct_df[struct_df['id'] == struct_id].iloc[0].to_dict()
                type_id = struct_type_id_from_name(df_row['type'])
                use_mod = 1 if df_row['use_mod'] else 0
                mod_val = df_row['mod_val']
                for cur_cell in range(len(cells)):
                    i, j = self._co_grid.get_cell_ij_from_index(cells[cur_cell])
                    ss.write(f'{i} {j} {type_id}')
                    if use_mod == 1:
                        ss.write(f' {mod_val}')
                    ss.write('\n')
            file.write(f'{num_struct_cells}\n')
            ss.seek(0)
            file.write(ss.getvalue())
