"""Parent dialog for the Structures properties dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.cmswave.data.structures_data import StructuresData
from xms.cmswave.gui.structures_dialog import StructuresDialog


class StructuresParentDialog(XmsDlg):
    """Container for the structures dialog. Makes importing data very simple."""
    def __init__(self, title, win_cont, structure_data, selected_structure=None):
        """Initializes the structures list and properties dialog.

        Args:
            title (:obj:`str`): Title of the dialog
            win_cont (:obj:`QWidget`): Parent dialog
            structure_data (StructuresData): The structure data to edit
            selected_structure (:obj:`int`): 0-based row index of the structure that should be selected.
                If provided, implies the dialog is coming from the Assign Structures command.
        """
        super().__init__(win_cont, 'xms.cmswave.gui.structures_parent_dialog')
        self.title = title
        self.structures_dialog = None
        self.selected_structure = selected_structure
        self.widgets = dict()
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])
        self.import_file(structure_data)
        self.setWindowTitle(title)

    def add_multi_polygon_select_warning(self):
        """Add a label if coming from the Assign Structures command with multiple polygons selected."""
        self.structures_dialog.widgets['lbl_multi_select'] = QLabel(
            'Structure assignment will be applied to all selected polygons.'
        )
        self.structures_dialog.widgets['lbl_multi_select'].setStyleSheet('font-weight: bold; color: red')
        self.structures_dialog.widgets['main_vert_layout'].insertWidget(
            0, self.structures_dialog.widgets['lbl_multi_select']
        )

    def accept(self):
        """Accept the dialog."""
        super().accept()

    def reject(self):
        """Reject the dialog."""
        super().reject()

    def import_file(self, structure_data):
        """Loads a structure properties file.

        Args:
            structure_data (StructuresData): The structure data to edit
        """
        if self.structures_dialog is not None:  # Overwrite the old data with the newly imported one.
            old_filename = self.structures_dialog.structure_data._filename
            attrs = self.structures_dialog.structure_data.info.attrs
            self.structures_dialog.structure_data.close()
            filesystem.copyfile(structure_data._filename, old_filename)
            structure_data = StructuresData(old_filename)
            structure_data.info.attrs = attrs
            structure_data.commit()
        else:
            structure_data = structure_data

        # if the layout is not empty then remove the dialog from it
        child = self.widgets['main_vert_layout'].takeAt(0)
        if child is not None:
            if child.widget() is not None:
                child.widget().deleteLater()

        dlg = StructuresDialog(title=self.title, parent=self, structure_data=structure_data)
        dlg.setWindowFlags(dlg.windowFlags() & ~Qt.Dialog)
        self.structures_dialog = dlg
        self.widgets['main_vert_layout'].addWidget(self.structures_dialog)
