"""Map Material and Sediment Material coverage locations and attributes to the SRH-2D domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_polygon_locations, write_display_options_to_json
)
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.snap.snap_polygon import SnapPolygon

# 4. Local modules
from xms.cmswave.components.mapped_structures_component import MappedStructuresComponent


class StructureMapper:
    """Class for mapping material coverage to a mesh for SRH-2D."""
    def __init__(self, coverage_mapper, wkt, comp_path):
        """Constructor."""
        self._generate_snap = False
        self._logger = logging.getLogger('xms.cmswave')
        self._co_grid = coverage_mapper.co_grid
        self._new_comp_unique_name = 'Mapped_Structures_Component'
        self._structures_coverage = coverage_mapper.structure_coverage
        self._structures_component = coverage_mapper.structure_component
        self._structures_data = coverage_mapper.structure_data
        self._snap_poly = SnapPolygon()
        self._snap_poly.set_grid(grid=self._co_grid, target_cells=False)
        self._snap_poly.add_polygons(polygons=self._structures_coverage.polygons)
        self._comp_main_file = ''
        self._poly_to_cells = {}
        self._comp_path = comp_path
        self._struct_df = self._structures_data.structures.to_dataframe()
        self._struct_comp_ids = self._struct_df['id'].to_list()
        self._struct_names = self._struct_df['name'].to_list()
        self.mapped_comp_uuid = None
        self.mapped_structures_display_uuid = None
        self.grid_wkt = wkt

    def do_map(self):
        """Creates the mapped structures component."""
        self._get_polygon_cells()

        if self._generate_snap:
            self._create_component_folder_and_copy_display_options()
            self._create_drawing()

            # Create the data_objects component
            do_comp = Component(
                main_file=self._comp_main_file,
                model_name='CMS-Wave',
                name=f'Snapped {self._structures_coverage.name} display',
                unique_name=self._new_comp_unique_name,
                locked=False,
                comp_uuid=os.path.basename(os.path.dirname(self._comp_main_file))
            )
            comp = MappedStructuresComponent(self._comp_main_file)
            return do_comp, comp

        return None, None  # pragma: no cover

    def _create_drawing(self):
        """Uses cell ids to get cell point coords to draw polygons for structures mapped to cells."""
        ugrid = self._co_grid.ugrid
        for comp_id, cell_ids in self._poly_to_cells.items():
            poly_list = []
            for cid in cell_ids:
                cell_locs = ugrid.get_cell_locations(cid)
                locs_list = [item for sublist in cell_locs for item in sublist]
                if len(locs_list) < 9:
                    continue  # pragma: no cover
                # repeat the first point
                locs_list.append(locs_list[0])
                locs_list.append(locs_list[1])
                locs_list.append(locs_list[2])
                outer_dict = {'outer': locs_list}
                poly_list.append(outer_dict)
            filename = os.path.join(self._comp_path, f'struct_{comp_id}.structid')
            write_display_option_polygon_locations(filename, poly_list)

    def _get_polygon_cells(self):
        """Uses xmssnap to get the cells for each polygon."""
        self._logger.info('Mapping structures coverage to grid.')
        num_cells = self._co_grid.ugrid.cell_count
        cell_flag = [True] * num_cells
        polys = self._structures_coverage.polygons
        for poly in polys:
            pid = poly.id
            cells = self._snap_poly.get_cells_in_polygon(pid)
            comp_id = self._structures_component.get_comp_id(TargetType.polygon, pid)
            if comp_id is None or comp_id < 0:
                comp_id = -1  # pragma: no cover
            if comp_id not in self._poly_to_cells:
                self._poly_to_cells[comp_id] = []
            self._poly_to_cells[comp_id].extend(cells)
            for cid in cells:
                cell_flag[cid] = False
        # add all unassigned cells to comp_id = 0 ('Bathymetry modification')
        if -1 not in self._poly_to_cells:
            self._poly_to_cells[-1] = []  # pragma: no cover
        for i in range(len(cell_flag)):
            if cell_flag[i]:
                self._poly_to_cells[-1].append(i)
        for struct_name, comp_id in zip(self._struct_names, self._struct_comp_ids):
            cells_from_comp_id = self._poly_to_cells.get(comp_id, [])
            if not cells_from_comp_id and comp_id > 0:
                self._logger.warning(f'Structure: {struct_name} was not assigned to any elements.')

    def _create_component_folder_and_copy_display_options(self):
        """Creates a folder for the mapped structure component and copies display options from the struct coverage."""
        if self.mapped_comp_uuid is None:
            comp_uuid = str(uuid.uuid4())  # pragma: no cover
        else:
            comp_uuid = self.mapped_comp_uuid
        self._logger.info('Creating component folder')
        self._comp_path = os.path.join(self._comp_path, comp_uuid)

        if os.path.exists(self._comp_path):
            shutil.rmtree(self._comp_path)  # pragma: no cover
        os.mkdir(self._comp_path)

        struct_comp_display_file = os.path.join(self._comp_path, 'structures_display_options.json')
        comp_display_file = os.path.join(self._comp_path, 'structures_display_options.json')
        if os.path.isfile(struct_comp_display_file):
            shutil.copyfile(struct_comp_display_file, comp_display_file)
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            json_dict = read_display_options_from_json(comp_display_file)
            if self.mapped_structures_display_uuid is None:
                json_dict['uuid'] = str(uuid.uuid4())  # pragma: no cover
            else:
                json_dict['uuid'] = self.mapped_structures_display_uuid
            json_dict['comp_uuid'] = comp_uuid
            json_dict['is_ids'] = 0
            # Set projection of free locations to be that of the mesh/current display
            categories.projection = {'wkt': self.grid_wkt}
            categories.from_dict(json_dict)
            write_display_options_to_json(comp_display_file, categories)
            self._comp_main_file = comp_display_file
        else:
            self._logger.info('Could not find structures_display_options.json file')  # pragma: no cover
