"""Base class for XMS DMI components."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import os
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.core.filesystem import filesystem as io_util

# 4. Local modules

# Type aliases
# List of tuple[str, str]: first string is message level (DEBUG, ERROR, WARNING, INFO), second string is message text.
Messages = list[tuple[str, str]]  # Messages return value
ActionRv = tuple[Messages, list[ActionRequest]]  # Return value for an action method


class ComponentBase:
    """Base class for XMS DMI components."""
    def __init__(self, main_file):
        """Construct the base class.

        Args:
            main_file (str): Every component should take a main file path as an argument at construction. This
                file will be used by XMS to instantiate Python instances of components. Any persistent data of
                the component should be stored in this file.
        """
        self.main_file = main_file.strip('"\'')
        self.display_option_list = []
        self.att_files = {}

        self.__class_name = ''
        self.__module_name = ''

    @property
    def class_name(self) -> str:
        """
        The name of this class.

        xmsapi uses this to know how to call methods on it.
        """
        return self.__class_name or self.__class__.__name__

    @class_name.setter
    def class_name(self, value: str):
        """The name of this class."""
        # xmsapi requires every component to have a class_name and module_name member in order to call methods on it
        # correctly. It used to be required that every component set this manually, which was error-prone and led to
        # hard-to-debug problems. The values of these members can normally be computed automatically, so this class
        # now defines properties that do so. This setter keeps existing code working until it can be updated, but is
        # no longer necessary and shouldn't be used.
        #
        # self.__class_name should be deleted when this setter is.
        warnings.warn('Setting self.class_name is no longer necessary.', category=DeprecationWarning, stacklevel=2)
        self.__class_name = value

    @property
    def module_name(self):
        """The name of the module containing this class."""
        return self.__module_name or self.__module__

    @module_name.setter
    def module_name(self, value):
        """The name of the module containing this class."""
        # xmsapi requires every component to have a class_name and module_name member in order to call methods on it
        # correctly. It used to be required that every component set this manually, which was error-prone and led to
        # hard-to-debug problems. The values of these members can normally be computed automatically, so this class
        # now defines properties that do so. This setter keeps existing code working until it can be updated, but is
        # no longer necessary and shouldn't be used.
        #
        # self.__module_name should be deleted when this setter is.
        warnings.warn('Setting self.module_name is no longer necessary.', category=DeprecationWarning, stacklevel=2)
        self.__module_name = value

    def copy_component_folder(self, destination):
        """Copies the contents of the component folder to a new folder.

        Notes:
            This is very specific to Windows. Requires the deprecated xcopy system utility. robocopy would be preferred,
            but it didn't work. No Python utilities worked reliably on network drives (os, shutil, dist_utils, pathlib).

            This didn't work in all cases either, so we now copy on the C++ side and this function just returns False
            if the two directories are the same.

        Args:
            destination (str): Path to the new component folder (the new UUID folder)

        Returns:
            bool: True if the folder was copied, False if the destination is the same as the source.
        """
        # Check if we are already in the new location
        new_main_file = os.path.join(destination, os.path.basename(self.main_file))
        if io_util.paths_are_equal(new_main_file, self.main_file):
            return False
        return True

    def save_to_location_base(self, new_path, save_type):
        """Save component files to a new location. Base class implementation to handle project open event.

        Derived components should not implement this method.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE - happens when re-saving this project.

                SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
                project.

                UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
                in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (:obj:`str`): The filename (absolute or relative to the UUID folder in temp).
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Call the save event first so the new component exists.
        new_main_file, messages, action_requests = self.save_to_location(new_path, save_type)
        # Handle XMS project open mode.
        if os.environ.get('XMS_OPENING_PROJECT', '') == 'TRUE':
            self.project_open_event(new_path)  # Allows components to initialize display lists.
        return new_main_file, messages, action_requests

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE - happens when re-saving this project.

                SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
                project.

                UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
                in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (:obj:`str`): The filename (absolute or relative to the UUID folder in temp).
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return '', [], []

    def project_open_event(self, new_path):
        """Called when XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list

        Args:
            new_path (str): Path to the new save location.
        """
        pass

    def renumber_event(self, renumbers, lock_state):
        """Called when coverages are renumbered in XMS.

        This will be called when a coverage that is used by a component, or that a component is part of, has renumbered
        some or all feature objects.

        Args:
            renumbers (dict): A dictionary with a key of string UUIDs of the coverages being renumbered. The value of
                this outer dictionary is a dictionary. The key of this inner dictionary is a string of the feature
                object entity type. It may be one of POINT, ARC, POLYGON. The value of this inner dictionary if a
                dictionary. The key of this innermost dictionary is the integer of the old id before the renumber. The
                value of this innermost dictionary is an integer of the new id.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (:obj:`list` of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def create_event(self, lock_state):
        """This will be called when XMS receives the component from an import script.

        This seems to only be called when the component is sent to XMS via Query from an import script. Creating a new
        coverage from XMS, changing a coverage's type to one that prompts XMS to create the component, and sending the
        component from any kind of component event, don't seem to trigger create events at all. This should not be
        relied on for anything that needs to happen outside of an import script.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def import_event(self, lock_state):
        """This will be called when the component's main file is imported into XMS.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def delete_event(self, lock_state):
        """This will be called when the component is deleted.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change or delete the files if
                locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def edit_event(self, edit_uuid, lock_state):
        """This will be called when a coverage or ugrid taken by this component is edited.

        Args:
            edit_uuid (str): A UUID of the coverage or ugrid that was edited.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def lock_event(self, new_lock_state):
        """This will be called when the component is being locked or unlocked for editing.

        Args:
            new_lock_state (bool): True if the the component is being locked for editing.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.
        """
        return []

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (:obj:`list` of :obj:`tuple`): A list of tuples containing the main files and lock states of
                the selected components of this type.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.MenuItem`):
                A list of menus and menu items to be shown. Note that this list can have objects of type
                xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list to indicate
                a separator.
        """
        return []

    def get_display_options(self):
        """Returns and clears the display option category lists.

        Returns:
            (:obj:`list` of :obj:`XmsDisplayMessage`):
        """
        disp_opts = []
        for disp in self.display_option_list:
            disp_opts.append(list(disp))
        self.display_option_list = []
        return disp_opts

    def get_initial_display_options(self, query: Query, _params):
        """
        Gets called to load initial display options if 'use_display' attribute is True in the component definition.

        Args:
            query: Query with a context at the component instance level.
            _params: Unused.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def get_component_coverage_ids(self):
        """Method used by coverage component implementations for getting id mappings.

        Do not implement if component does not derive from CoverageComponentBase.

        Returns:
            (:obj:`tuple` of None): Nothing (implemented by CoverageComponentBase).
        """
        return None, None

    def handle_merge(self, merge_list: list[tuple[str, dict]]) -> ActionRv:
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list: tuple containing 1) main_file: The absolute path to the main file of the old component this
             component is being merged from. 2) id_files: The dictionary keys are 'POINT', 'ARC', and 'POLYGON'. Each
             value is a tuple that may have two absolute file paths or none. The first file is for the ids in XMS on
             the coverage. The second file contains the ids the old component used for those objects. Both id files
             should be equal in length. This dictionary is only applicable if the component derives from
             CoverageComponentBase.

        Returns:
            (tuple): tuple containing:
                - messages: List of tuples with the first element of the tuple being the message level (DEBUG, ERROR,
                  WARNING, INFO) and the second element being the message text.
                - action_requests: List of actions for XMS to perform.
        """
        return [], []

    def get_shapefile_att_files(self):
        """Get shapefile att files written during write_shapefile_atts.

        Returns:
            dict: Dictionary where key is TargetType and value is list containing the column definition file and the
                att table file.
        """
        return self.att_files

    def write_shapefile_atts(self, query, params):
        """Implement on Coverage components to export attributes to a shapefile.

        Args:
            query (:obj:'xms.api.dmi.Query'): Query with a context at the component instance level.
            params: Contains 'target_type' enum list

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []

    def read_shapefile_atts(self, query, params):
        """Implement on Coverage components to populate attributes from a shapefile.

        Args:
            query (:obj:'xms.api.dmi.Query'): Query with a context at the component instance level.
            params: Contains 'filename' list

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []
